"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildTable = buildTable;
var _types = require("../types");
var _utils = require("../utils");
var _columns = require("../columns");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const ACCESSOR = 'metric_formula_accessor';
function buildVisualizationState(config) {
  const layer = config;
  return {
    layerId: _types.DEFAULT_LAYER_ID,
    layerType: 'data',
    columns: [{
      columnId: ACCESSOR
    }, ...(layer.breakdown || []).map((breakdown, i) => ({
      columnId: `${ACCESSOR}_breakdown_${i}`
    })), ...(layer.splitBy || []).map((breakdown, i) => ({
      columnId: `${ACCESSOR}_splitby_${i}`
    }))]
  };
}
function buildFormulaLayer(layer, i, dataView) {
  const layers = {
    [_types.DEFAULT_LAYER_ID]: {
      ...(0, _columns.getFormulaColumn)(ACCESSOR, (0, _utils.mapToFormula)(layer), dataView)
    }
  };
  const defaultLayer = layers[_types.DEFAULT_LAYER_ID];
  if (layer.breakdown) {
    layer.breakdown.reverse().forEach((breakdown, x) => {
      const columnName = `${ACCESSOR}_breakdown_${x}`;
      const breakdownColumn = (0, _columns.getBreakdownColumn)({
        options: breakdown,
        dataView
      });
      (0, _utils.addLayerColumn)(defaultLayer, columnName, breakdownColumn, true);
    });
  } else {
    throw new Error('breakdown must be defined for table!');
  }
  if (layer.splitBy) {
    layer.splitBy.forEach((breakdown, x) => {
      const columnName = `${ACCESSOR}_splitby_${x}`;
      const breakdownColumn = (0, _columns.getBreakdownColumn)({
        options: breakdown,
        dataView
      });
      (0, _utils.addLayerColumn)(defaultLayer, columnName, breakdownColumn, true);
    });
  }
  return defaultLayer;
}
function getValueColumns(layer) {
  if (layer.breakdown && layer.breakdown.filter(b => typeof b !== 'string').length) {
    throw new Error('breakdown must be a field name when not using index source');
  }
  if (layer.splitBy && layer.splitBy.filter(s => typeof s !== 'string').length) {
    throw new Error('xAxis must be a field name when not using index source');
  }
  return [...(layer.breakdown ? layer.breakdown.map((b, i) => {
    return (0, _columns.getValueColumn)(`${ACCESSOR}_breakdown_${i}`, b);
  }) : []), ...(layer.splitBy ? layer.splitBy.map((b, i) => {
    return (0, _columns.getValueColumn)(`${ACCESSOR}_splitby_${i}`, b);
  }) : []), (0, _columns.getValueColumn)(ACCESSOR, layer.value)];
}
async function buildTable(config, {
  dataViewsAPI
}) {
  const dataviews = {};
  const _buildFormulaLayer = (cfg, i, dataView) => buildFormulaLayer(cfg, i, dataView);
  const datasourceStates = await (0, _utils.buildDatasourceStates)(config, dataviews, _buildFormulaLayer, getValueColumns, dataViewsAPI);
  const {
    references,
    internalReferences,
    adHocDataViews
  } = (0, _utils.extractReferences)(dataviews);
  return {
    title: config.title,
    visualizationType: 'lnsDatatable',
    references,
    state: {
      datasourceStates,
      internalReferences,
      filters: [],
      query: {
        language: 'kuery',
        query: ''
      },
      visualization: buildVisualizationState(config),
      adHocDataViews
    }
  };
}