"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InterceptPrompter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var Rx = _interopRequireWildcard(require("rxjs"));
var _apmRum = require("@elastic/apm-rum");
var _service = require("./service");
var _user_intercept_run_persistence_service = require("./service/user_intercept_run_persistence_service");
var _constants = require("../../common/constants");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class InterceptPrompter {
  constructor() {
    (0, _defineProperty2.default)(this, "userInterceptRunPersistenceService", new _user_intercept_run_persistence_service.UserInterceptRunPersistenceService());
    (0, _defineProperty2.default)(this, "interceptDialogService", new _service.InterceptDialogService());
    (0, _defineProperty2.default)(this, "queueIntercept", void 0);
    // observer for page visibility changes, shared across all intercepts
    (0, _defineProperty2.default)(this, "pageHidden$", void 0);
    // Registry for intercept timers, used to track activation of a particular intercept for each user
    (0, _defineProperty2.default)(this, "interceptTimerRegistry", new Proxy({}, {
      get: (_target, prop) => {
        if (typeof prop === 'symbol') return undefined;
        const storage = JSON.parse(localStorage.getItem(_constants.INTERCEPT_PROMPTER_LOCAL_STORAGE_KEY) || '{}');
        return storage[prop];
      },
      set: (_target, prop, value) => {
        if (typeof prop === 'symbol') return false;
        const storage = JSON.parse(localStorage.getItem(_constants.INTERCEPT_PROMPTER_LOCAL_STORAGE_KEY) || '{}');
        storage[prop] = value;
        localStorage.setItem(_constants.INTERCEPT_PROMPTER_LOCAL_STORAGE_KEY, JSON.stringify(storage));
        return true;
      },
      deleteProperty: (_target, prop) => {
        if (typeof prop === 'symbol') return false;
        const storage = JSON.parse(localStorage.getItem(_constants.INTERCEPT_PROMPTER_LOCAL_STORAGE_KEY) || '{}');
        delete storage[prop];
        localStorage.setItem(_constants.INTERCEPT_PROMPTER_LOCAL_STORAGE_KEY, JSON.stringify(storage));
        return true;
      }
    }));
  }
  setup({
    analytics,
    notifications
  }) {
    this.interceptDialogService.setup({
      analytics,
      notifications
    });
    return {};
  }
  start({
    http,
    ...dialogServiceDeps
  }) {
    const {
      getUserTriggerData$,
      updateUserTriggerData
    } = this.userInterceptRunPersistenceService.start(http);
    ({
      add: this.queueIntercept
    } = this.interceptDialogService.start({
      ...dialogServiceDeps,
      persistInterceptRunId: updateUserTriggerData,
      staticAssetsHelper: http.staticAssets,
      resetInterceptTimingRecord: this.clearInterceptTimerStartRecord.bind(this)
    }));
    this.pageHidden$ = Rx.fromEvent(document, 'visibilitychange').pipe(Rx.map(() => document.hidden), Rx.startWith(document.hidden));
    return {
      /**
       * Configures the intercept journey that will be shown to the user, and returns an observable
       * that manages displaying the intercept at the appropriate time based on the interval that's been pre-configured for the
       * trigger ID matching the ID of this particular journey being configured.
       */
      registerIntercept: this.registerIntercept.bind(this, http, getUserTriggerData$)
    };
  }
  registerIntercept(http, getUserTriggerData$, intercept) {
    return Rx.from(http.post(_constants.TRIGGER_INFO_API_ROUTE, {
      body: JSON.stringify({
        triggerId: intercept.id
      })
    })).pipe(Rx.filter(response => !!response)).pipe(Rx.combineLatestWith(getUserTriggerData$(intercept.id)), Rx.switchMap(([response, userTriggerData]) => {
      if (!response.recurrent && userTriggerData && userTriggerData.lastInteractedInterceptId) {
        // if the intercept is not recurrent and the user has already interacted with it,
        // there's nothing to do
        return Rx.EMPTY;
      }
      let nextRunId;
      return this.pageHidden$.pipe(Rx.switchMap(isHidden => {
        // if the page is hidden, there's no need to run computations on whether to display the intercept or not
        if (isHidden) return Rx.EMPTY;
        const timerCalculations = this.calculateTimeTillTrigger(response.registeredAt, response.triggerIntervalInMs);

        // set the next run id
        nextRunId = timerCalculations.nextRunId;
        const now = Date.now();
        const timerStart = this.interceptTimerStartRecord(intercept.id).getTime();
        const timeElapsedSinceTimerStart = now - timerStart;
        if (timeElapsedSinceTimerStart >= response.triggerIntervalInMs) {
          // fetch user trigger again because it's possible that the user has already interacted with the intercept,
          // especially that the user might have interacted with the intercept in a different tab
          return getUserTriggerData$(intercept.id);
        } else {
          // Not yet time, return EMPTY to skip this iteration
          return Rx.EMPTY;
        }
      })).pipe(Rx.tap(async triggerData => {
        if (nextRunId !== triggerData.lastInteractedInterceptId) {
          try {
            var _this$queueIntercept;
            const interceptConfig = await intercept.config();
            (_this$queueIntercept = this.queueIntercept) === null || _this$queueIntercept === void 0 ? void 0 : _this$queueIntercept.call(this, {
              id: intercept.id,
              runId: nextRunId,
              ...interceptConfig
            });
            nextRunId++;
          } catch (err) {
            _apmRum.apm.captureError(err, {
              labels: {
                interceptId: intercept.id
              }
            });
          }
        }
      }), Rx.catchError(error => {
        _apmRum.apm.captureError(error, {
          labels: {
            interceptId: intercept.id,
            errorContext: 'registerIntercept'
          }
        });
        return Rx.EMPTY;
      }));
    }));
  }
  calculateTimeTillTrigger(registeredAt, triggerIntervalInMs) {
    const now = Date.now();
    const diff = now - Date.parse(registeredAt);
    const runs = Math.floor(diff / triggerIntervalInMs);
    const nextRunId = runs + 1;
    const timeTillNextRun = nextRunId * triggerIntervalInMs - diff;
    return {
      runs,
      shouldTriggerImmediately: timeTillNextRun <= 0,
      timeTillNextRun,
      nextRunId
    };
  }
  markInterceptTimerStart(interceptId) {
    this.interceptTimerRegistry[interceptId] = {
      timerStart: new Date()
    };
  }

  /**
   * Returns the date and time when the intercept timer was started for the given intercept id
   * If no record exists, it sets it to the current date and time and returns this value
   */
  interceptTimerStartRecord(interceptId) {
    if (!this.interceptTimerRegistry[interceptId]) {
      this.markInterceptTimerStart(interceptId);
    }
    return new Date(this.interceptTimerRegistry[interceptId].timerStart);
  }
  clearInterceptTimerStartRecord(interceptId) {
    delete this.interceptTimerRegistry[interceptId];
  }
}
exports.InterceptPrompter = InterceptPrompter;