"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDataProviders = registerDataProviders;
var _get_hosts = require("../routes/infra/lib/host/get_hosts");
var _get_infra_metrics_client = require("../lib/helpers/get_infra_metrics_client");
var _get_infra_alerts_client = require("../lib/helpers/get_infra_alerts_client");
var _get_preferred_schema = require("../lib/helpers/get_preferred_schema");
var _get_infra_request_handler_context = require("../utils/get_infra_request_handler_context");
var _get_apm_data_access_client = require("../lib/helpers/get_apm_data_access_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_HOST_METRICS = ['cpuV2', 'memory', 'memoryFree', 'diskSpaceUsage', 'normalizedLoad1m'];
function registerDataProviders({
  core,
  plugins,
  libs,
  logger
}) {
  const {
    observabilityAgentBuilder
  } = plugins;
  if (!observabilityAgentBuilder) {
    return;
  }
  observabilityAgentBuilder.registerDataProvider('infraHosts', async ({
    request,
    from,
    to,
    limit,
    query
  }) => {
    const infraToolResources = await buildInfraToolResources({
      core,
      plugins,
      libs,
      request
    });
    const fromMs = new Date(from).getTime();
    const toMs = new Date(to).getTime();
    const {
      preferredSchema
    } = await (0, _get_preferred_schema.getPreferredSchema)({
      infraMetricsClient: infraToolResources.infraMetricsClient,
      dataSource: 'host',
      from: fromMs,
      to: toMs
    });
    if (!preferredSchema) {
      logger.info('Could not determine preferred schema ');
      return {
        nodes: []
      };
    }
    const result = await (0, _get_hosts.getHosts)({
      from: fromMs,
      to: toMs,
      metrics: [...DEFAULT_HOST_METRICS],
      limit,
      query,
      alertsClient: infraToolResources.alertsClient,
      infraMetricsClient: infraToolResources.infraMetricsClient,
      apmDataAccessServices: infraToolResources.apmDataAccessServices,
      schema: preferredSchema
    });
    return result;
  });
}
async function buildInfraToolResources({
  core,
  plugins,
  libs,
  request
}) {
  const [coreStart] = await core.getStartServices();
  const soClient = coreStart.savedObjects.getScopedClient(request, {
    includedHiddenTypes: []
  });
  const uiSettingsClient = coreStart.uiSettings.asScopedToClient(soClient);
  const esClient = coreStart.elasticsearch.client.asScoped(request);
  const coreContext = {
    savedObjects: {
      client: soClient
    },
    uiSettings: {
      client: uiSettingsClient
    },
    elasticsearch: {
      client: esClient
    }
  };
  const infraContext = await (0, _get_infra_request_handler_context.getInfraRequestHandlerContext)({
    coreContext,
    request,
    plugins
  });
  const context = {
    core: Promise.resolve(coreContext),
    infra: Promise.resolve(infraContext)
  };
  const infraMetricsClient = await (0, _get_infra_metrics_client.getInfraMetricsClient)({
    libs,
    context,
    request
  });
  const alertsClient = await (0, _get_infra_alerts_client.getInfraAlertsClient)({
    libs,
    request
  });
  const apmDataAccessClient = (0, _get_apm_data_access_client.getApmDataAccessClient)({
    request,
    libs,
    context
  });
  const apmDataAccessServices = await apmDataAccessClient.getServices();
  return {
    infraMetricsClient,
    alertsClient,
    apmDataAccessServices
  };
}