"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useInfraMLModule = void 0;
var _react = require("react");
var _use_kibana = require("../../hooks/use_kibana");
var _use_tracked_promise = require("../../hooks/use_tracked_promise");
var _infra_ml_module_status = require("./infra_ml_module_status");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useInfraMLModule = ({
  sourceConfiguration,
  moduleDescriptor
}) => {
  const {
    services
  } = (0, _use_kibana.useKibanaContextForPlugin)();
  const {
    spaceId,
    sourceId
  } = sourceConfiguration;
  const [moduleStatus, dispatchModuleStatus] = (0, _infra_ml_module_status.useModuleStatus)(moduleDescriptor.jobTypes);
  const [, fetchJobStatus] = (0, _use_tracked_promise.useTrackedPromise)({
    cancelPreviousOn: 'resolution',
    createPromise: async () => {
      dispatchModuleStatus({
        type: 'fetchingJobStatuses'
      });
      return await moduleDescriptor.getJobSummary(spaceId, sourceId, services.http.fetch);
    },
    onResolve: jobResponse => {
      dispatchModuleStatus({
        type: 'fetchedJobStatuses',
        payload: jobResponse,
        spaceId,
        sourceId
      });
    },
    onReject: () => {
      dispatchModuleStatus({
        type: 'failedFetchingJobStatuses'
      });
    }
  }, [spaceId, sourceId]);
  const [, setUpModule] = (0, _use_tracked_promise.useTrackedPromise)({
    cancelPreviousOn: 'resolution',
    createPromise: async (selectedIndices, start, end, filter, partitionField) => {
      dispatchModuleStatus({
        type: 'startedSetup'
      });
      const setupResult = await moduleDescriptor.setUpModule({
        start,
        end,
        filter,
        moduleSourceConfiguration: {
          indices: selectedIndices,
          sourceId,
          spaceId
        },
        partitionField
      }, services.http.fetch);
      const jobSummaries = await moduleDescriptor.getJobSummary(spaceId, sourceId, services.http.fetch);
      return {
        setupResult,
        jobSummaries
      };
    },
    onResolve: ({
      setupResult: {
        datafeeds,
        jobs
      },
      jobSummaries
    }) => {
      dispatchModuleStatus({
        type: 'finishedSetup',
        datafeedSetupResults: datafeeds,
        jobSetupResults: jobs,
        jobSummaries,
        spaceId,
        sourceId
      });
    },
    onReject: () => {
      dispatchModuleStatus({
        type: 'failedSetup'
      });
    }
  }, [moduleDescriptor.setUpModule, spaceId, sourceId]);
  const [cleanUpModuleRequest, cleanUpModule] = (0, _use_tracked_promise.useTrackedPromise)({
    cancelPreviousOn: 'resolution',
    createPromise: async () => {
      return await moduleDescriptor.cleanUpModule(spaceId, sourceId, services.http.fetch);
    },
    onReject: e => {
      throw new Error(`Failed to clean up previous ML job: ${e}`);
    }
  }, [spaceId, sourceId]);
  const isCleaningUp = (0, _react.useMemo)(() => cleanUpModuleRequest.state === 'pending', [cleanUpModuleRequest.state]);
  const cleanUpAndSetUpModule = (0, _react.useCallback)((selectedIndices, start, end, filter, partitionField) => {
    dispatchModuleStatus({
      type: 'startedSetup'
    });
    cleanUpModule().then(() => {
      setUpModule(selectedIndices, start, end, filter, partitionField);
    }).catch(e => {
      dispatchModuleStatus({
        type: 'failedSetup',
        reason: e.toString()
      });
    });
  }, [cleanUpModule, dispatchModuleStatus, setUpModule]);
  const viewResults = (0, _react.useCallback)(() => {
    dispatchModuleStatus({
      type: 'viewedResults'
    });
  }, [dispatchModuleStatus]);
  const jobIds = (0, _react.useMemo)(() => moduleDescriptor.getJobIds(spaceId, sourceId), [moduleDescriptor, spaceId, sourceId]);
  return {
    cleanUpAndSetUpModule,
    cleanUpModule,
    fetchJobStatus,
    isCleaningUp,
    jobIds,
    jobStatus: moduleStatus.jobStatus,
    jobSummaries: moduleStatus.jobSummaries,
    lastSetupErrorMessages: moduleStatus.lastSetupErrorMessages,
    moduleDescriptor,
    setUpModule,
    setupStatus: moduleStatus.setupStatus,
    sourceConfiguration,
    viewResults
  };
};
exports.useInfraMLModule = useInfraMLModule;