"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ArtifactService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _axios = _interopRequireDefault(require("axios"));
var _crypto = require("crypto");
var _admZip = _interopRequireDefault(require("adm-zip"));
var _lodash = require("lodash");
var _artifact = require("./artifact.errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ArtifactService {
  constructor(logger, clusterInfo, cdnConfig) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "cache", void 0);
    (0, _defineProperty2.default)(this, "cdnConfig", void 0);
    (0, _defineProperty2.default)(this, "clusterInfo", void 0);
    (0, _defineProperty2.default)(this, "manifestUrl", void 0);
    this.logger = logger.get(ArtifactService.name);
    this.cache = new Map();
    this.configure(clusterInfo, cdnConfig);
  }
  configure(clusterInfo, cdnConfig) {
    var _clusterInfo$version, _this$cdnConfig;
    this.logger.debug('Configuring artifact service with cluster info', {
      clusterInfo
    });
    if (!((_clusterInfo$version = clusterInfo.version) !== null && _clusterInfo$version !== void 0 && _clusterInfo$version.number)) {
      throw new Error('Cluster info must include version number, impossible to calculate the manifest url');
    }
    this.clusterInfo = clusterInfo;
    this.cdnConfig = cdnConfig;
    const version = clusterInfo.version.number.substring(0, clusterInfo.version.number.indexOf('-')) || clusterInfo.version.number;
    this.manifestUrl = `${(_this$cdnConfig = this.cdnConfig) === null || _this$cdnConfig === void 0 ? void 0 : _this$cdnConfig.url}/downloads/kibana/manifest/artifacts-${version}.zip`;
    this.logger.debug('Artifact service started', {
      manifestUrl: this.manifestUrl
    });
  }
  async getArtifact(name) {
    var _this$cdnConfig2;
    this.logger.debug('Getting artifact', {
      name
    });
    return _axios.default.get(this.getManifestUrl(), {
      headers: this.headers(name),
      timeout: (_this$cdnConfig2 = this.cdnConfig) === null || _this$cdnConfig2 === void 0 ? void 0 : _this$cdnConfig2.requestTimeout,
      validateStatus: status => status < 500,
      responseType: 'arraybuffer'
    }).then(async response => {
      switch (response.status) {
        case 200:
          const manifest = {
            data: await this.getManifest(name, response.data),
            modified: true
          };
          // only update etag if we got a valid manifest
          if (response.headers && response.headers.etag) {
            var _response$headers$eta, _response$headers;
            const cacheEntry = {
              manifest: {
                ...manifest,
                modified: false
              },
              etag: (_response$headers$eta = (_response$headers = response.headers) === null || _response$headers === void 0 ? void 0 : _response$headers.etag) !== null && _response$headers$eta !== void 0 ? _response$headers$eta : ''
            };
            this.cache.set(name, cacheEntry);
          }
          return (0, _lodash.cloneDeep)(manifest);
        case 304:
          return (0, _lodash.cloneDeep)(this.getCachedManifest(name));
        case 404:
          // just in case, remove the entry
          this.cache.delete(name);
          throw new _artifact.ManifestNotFoundError(this.manifestUrl);
        default:
          throw Error(`Failed to download manifest, unexpected status code: ${response.status}`);
      }
    });
  }
  getManifestUrl() {
    if (!this.manifestUrl) {
      var _this$clusterInfo, _this$clusterInfo$ver;
      throw Error(`No manifest url for version ${(_this$clusterInfo = this.clusterInfo) === null || _this$clusterInfo === void 0 ? void 0 : (_this$clusterInfo$ver = _this$clusterInfo.version) === null || _this$clusterInfo$ver === void 0 ? void 0 : _this$clusterInfo$ver.number}`);
    }
    return this.manifestUrl;
  }
  getCachedManifest(name) {
    const entry = this.cache.get(name);
    if (!entry) {
      throw new _artifact.ArtifactNotFoundError(name);
    }
    return entry.manifest;
  }
  async getManifest(name, data) {
    const zip = new _admZip.default(data);
    const manifestFile = zip.getEntry('manifest.json');
    const signatureFile = zip.getEntry('manifest.sig');
    if (!manifestFile) {
      throw Error('No manifest.json in artifact zip');
    }
    if (!signatureFile) {
      throw Error('No manifest.sig in artifact zip');
    }
    if (!this.isSignatureValid(manifestFile.getData(), signatureFile.getData())) {
      throw Error('Invalid manifest signature');
    }
    const manifest = JSON.parse(manifestFile.getData().toString());
    const artifact = manifest.artifacts[name];
    if (artifact) {
      var _this$cdnConfig3, _this$cdnConfig4;
      const url = `${(_this$cdnConfig3 = this.cdnConfig) === null || _this$cdnConfig3 === void 0 ? void 0 : _this$cdnConfig3.url}${artifact.relative_url}`;
      const artifactResponse = await _axios.default.get(url, {
        timeout: (_this$cdnConfig4 = this.cdnConfig) === null || _this$cdnConfig4 === void 0 ? void 0 : _this$cdnConfig4.requestTimeout
      });
      return artifactResponse.data;
    } else {
      throw new _artifact.ArtifactNotFoundError(name);
    }
  }
  headers(name) {
    const cacheEntry = this.cache.get(name);
    if (cacheEntry !== null && cacheEntry !== void 0 && cacheEntry.etag) {
      return {
        'If-None-Match': cacheEntry.etag
      };
    }
    return {};
  }
  isSignatureValid(data, signature) {
    if (!this.cdnConfig) {
      throw Error('No CDN configuration provided');
    }
    const verifier = (0, _crypto.createVerify)('RSA-SHA256');
    verifier.update(data);
    verifier.end();
    return verifier.verify(this.cdnConfig.pubKey, signature);
  }
}
exports.ArtifactService = ArtifactService;