"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getReviewFields = getReviewFields;
exports.isCollapsiblePattern = isCollapsiblePattern;
exports.sanitize = sanitize;
var _lodash = require("lodash");
var _constants = require("../constants");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Generates an object of fields with their corresponding GROK component and example values.
 */
function getReviewFields(nodes, numExamples = 5) {
  return nodes.reduce((reviewFields, node) => {
    if ((0, _utils.isNamedField)(node)) {
      reviewFields[node.id] = {
        grok_component: node.component,
        example_values: (0, _lodash.uniq)(node.values).slice(0, numExamples)
      };
    }
    return reviewFields;
  }, {});
}

/**
 * Result from LLM review of fields where ECS field names have already been mapped to OpenTelemetry fields.
 *
 * Example value:
 *
 * ```json
 * {
 *     "log_source": "Apache HTTP Server Log",
 *     "fields": [
 *         {
 *             "name": "@timestamp",
 *             "columns": ["field_0", "field_1", "field_2"],
 *             "grok_components": ["DAY", "SYSLOGTIMESTAMP", "YEAR"]
 *         },
 *         {
 *             "name": "log.level",
 *             "columns": ["field_3"],
 *             "grok_components": ["LOGLEVEL"]
 *         },
 *         {
 *             "name": "message",
 *             "columns": ["field_4"],
 *             "grok_components": ["GREEDYDATA"]
 *         }
 *     ]
 * }
 * ```
 */

function isCollapsiblePattern(pattern) {
  return _constants.COLLAPSIBLE_PATTERNS.includes(pattern);
}
function sanitize(value) {
  return value.replaceAll(/[\.\[\]\{\}]/g, '\\$&');
}