"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.appendFile = appendFile;
exports.appendFileSync = appendFileSync;
exports.createReadStream = createReadStream;
exports.createWriteStream = createWriteStream;
exports.deleteFile = deleteFile;
exports.deleteFileSync = deleteFileSync;
exports.existsSync = existsSync;
exports.readFile = readFile;
exports.readFileSync = readFileSync;
exports.writeFile = writeFile;
exports.writeFileSync = writeFileSync;
var _fs = require("fs");
var _utils = require("./utils");
var _validations = require("./validations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Writes data to a file asynchronously, with automatic validation and sanitization.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param content - The content to write. Can be a string, Buffer, TypedArray, Iterable, AsyncIterable, or Stream
 * @param options - Optional configuration
 * @returns Metadata about the written file, including its alias and full path
 *
 * @example
 * ```ts
 * const metadata = await writeFile('report.csv', csvData, {
 *   volume: 'reports',
 * });
 * ```
 */
async function writeFile(name, content, options) {
  const {
    volume,
    override = true
  } = options !== null && options !== void 0 ? options : {};
  const {
    fullPath,
    alias
  } = (0, _utils.getSafePath)(name, volume);
  if (!override && (0, _fs.existsSync)(fullPath)) {
    throw new Error(`File already exists: ${name}. Use override: true to replace it.`);
  }
  await (0, _utils.ensureDirectory)(fullPath);
  const validatedContent = (0, _validations.validateAndSanitizeFileData)(content, fullPath);
  await _fs.promises.writeFile(fullPath, validatedContent);
  return {
    alias,
    path: fullPath
  };
}

/**
 * Appends data to a file, creating the file if it does not yet exist.
 * The content is validated and sanitized before appending.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param content - The content to append (string or Uint8Array)
 * @param options - Optional configuration
 * @returns Metadata about the file, including its alias and full path
 *
 * @example
 * ```ts
 * // Appends to: data/logs/debug.log
 * await appendFile('debug.log', 'Debug message\n', { volume: 'logs' });
 * ```
 */
async function appendFile(name, content, options) {
  const {
    fullPath,
    alias
  } = (0, _utils.getSafePath)(name, options === null || options === void 0 ? void 0 : options.volume);
  await (0, _utils.ensureDirectory)(fullPath);
  const validatedContent = (0, _validations.validateAndSanitizeFileData)(content, fullPath);
  await _fs.promises.appendFile(fullPath, validatedContent);
  return {
    alias,
    path: fullPath
  };
}

/**
 * Creates a Writable stream for writing to a file.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param volume - Optional namespace within the data directory to organize files
 * @param options - Stream options (encoding or StreamOptions)
 * @returns A Writable stream that writes to the specified file
 *
 * @example
 * ```ts
 * // Writes to: data/reports/report.csv
 * const stream = createWriteStream('report.csv', 'reports', { encoding: 'utf8' });
 * stream.write(data);
 * stream.end();
 * ```
 */
function createWriteStream(name, volume, options) {
  const {
    fullPath
  } = (0, _utils.getSafePath)(name, volume);
  (0, _utils.ensureDirectorySync)(fullPath);
  return (0, _fs.createWriteStream)(fullPath, options);
}

/**
 * Reads the entire contents of a file asynchronously.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param volume - Optional namespace within the data directory
 * @returns The file contents as a Buffer (or string if encoding is specified)
 *
 * @example
 * ```ts
 * // Reads from: data/reports/report.csv
 * const report = await readFile('report.csv', 'reports');
 * ```
 */
async function readFile(name, volume) {
  const {
    fullPath
  } = (0, _utils.getSafePath)(name, volume);
  return await _fs.promises.readFile(fullPath);
}

/**
 * Creates a Readable stream from a file.
 * Useful for reading large files incrementally without loading everything into memory.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param volume - Optional namespace within the data directory
 * @param options - Stream options (encoding or StreamOptions)
 * @returns A Readable stream that reads from the specified file
 *
 * @example
 * ```ts
 * // Reads from: data/reports/report.csv
 * const stream = createReadStream('report.csv', 'reports', { encoding: 'utf8' });
 * ```
 */
function createReadStream(name, volume, options) {
  const {
    fullPath
  } = (0, _utils.getSafePath)(name, volume);
  return (0, _fs.createReadStream)(fullPath, options);
}

/**
 * Deletes a file asynchronously.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param options - Optional configuration
 *
 * @example
 * ```ts
 * // Deletes: data/reports/old-report.csv
 * await deleteFile('old-report.csv', { volume: 'reports' });
 * ```
 */
async function deleteFile(name, options) {
  const {
    fullPath
  } = (0, _utils.getSafePath)(name, options === null || options === void 0 ? void 0 : options.volume);
  await _fs.promises.unlink(fullPath);
}

/**
 * Writes data to a file synchronously, with automatic validation and sanitization.
 * All files are written to the data directory with path traversal protection.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param content - The content to write (string or ArrayBufferView)
 * @param options - Optional configuration
 * @returns Metadata about the written file, including its alias and full path
 *
 * @example
 * ```ts
 * // Writes to: data/reports/report.csv
 * writeFileSync('report.csv', csvData, { volume: 'reports' });
 * ```
 */
function writeFileSync(name, content, options) {
  const {
    volume,
    override = true
  } = options !== null && options !== void 0 ? options : {};
  const {
    fullPath,
    alias
  } = (0, _utils.getSafePath)(name, volume);

  // Check if file exists and override is not allowed
  if (!override && (0, _fs.existsSync)(fullPath)) {
    throw new Error(`File already exists: ${name}. Use override: true to replace it.`);
  }
  (0, _utils.ensureDirectorySync)(fullPath);
  const validatedContent = (0, _validations.validateAndSanitizeFileData)(content, fullPath);
  (0, _fs.writeFileSync)(fullPath, validatedContent);
  return {
    alias,
    path: fullPath
  };
}

/**
 * Appends data to a file synchronously, creating the file if it does not yet exist.
 * The content is validated and sanitized before appending.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param content - The content to append (string or Uint8Array)
 * @param options - Optional configuration
 * @returns Metadata about the file, including its alias and full path
 *
 * @example
 * ```ts
 * // Appends to: data/logs/debug.log
 * appendFileSync('debug.log', 'Debug message\n', { volume: 'logs' });
 * ```
 */
function appendFileSync(name, content, options) {
  const {
    volume
  } = options !== null && options !== void 0 ? options : {};
  const {
    fullPath,
    alias
  } = (0, _utils.getSafePath)(name, volume);
  (0, _utils.ensureDirectorySync)(fullPath);
  const validatedContent = (0, _validations.validateAndSanitizeFileData)(content, fullPath);
  (0, _fs.appendFileSync)(fullPath, validatedContent);
  return {
    alias,
    path: fullPath
  };
}

/**
 * Reads the entire contents of a file synchronously.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param volume - Optional namespace within the data directory
 * @returns The file contents as a Buffer (or string if encoding is specified)
 *
 * @example
 * ```ts
 * // Reads from: data/reports/report.csv
 * const report = readFileSync('report.csv', 'reports');
 * ```
 */
function readFileSync(name, volume) {
  const {
    fullPath
  } = (0, _utils.getSafePath)(name, volume);
  return (0, _fs.readFileSync)(fullPath);
}

/**
 * Deletes a file synchronously.
 *
 * @param name - The filename (relative path within the data directory or volume)
 * @param options - Optional configuration
 *
 * @example
 * ```ts
 * // Deletes: data/reports/old-report.csv
 * deleteFileSync('old-report.csv', { volume: 'reports' });
 * ```
 */
function deleteFileSync(name, options) {
  const {
    fullPath
  } = (0, _utils.getSafePath)(name, options === null || options === void 0 ? void 0 : options.volume);
  (0, _fs.unlinkSync)(fullPath);
}

/**
 * Checks if a file exists synchronously.
 *
 * @param path - The file path (relative path within the data directory or volume)
 * @param volume - Optional namespace within the data directory
 * @returns `true` if the file exists, `false` otherwise
 *
 * @example
 * ```ts
 * // File exists at: data/reports/report.csv
 * if (existsSync('report.csv', 'reports')) {
 * }
 * ```
 */
function existsSync(path, volume) {
  const {
    fullPath
  } = (0, _utils.getSafePath)(path, volume);
  return (0, _fs.existsSync)(fullPath);
}