"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.throwIfAborted = exports.cleanupPolicyRevisions = void 0;
var _get_policies_to_clean = require("./get_policies_to_clean");
var _populate_minimum_revisions_used_by_agents = require("./populate_minimum_revisions_used_by_agents");
var _delete_policy_revisions = require("./delete_policy_revisions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const defaultConfig = {
  maxRevisions: 10,
  maxPolicies: 100,
  maxDocsToDelete: 5000,
  timeout: '5m'
};
const cleanupPolicyRevisions = async (esClient, context) => {
  const {
    logger,
    abortController
  } = context;
  const config = {
    ...defaultConfig,
    ...context.config
  };
  logger.debug(`[FleetPolicyRevisionsCleanupTask] Starting cleanup with max_revisions: ${config.maxRevisions}, max_policies_per_run: ${config.maxPolicies}`);
  const policiesToClean = await (0, _get_policies_to_clean.getPoliciesToClean)(esClient, {
    ...context,
    config
  });
  if (Object.keys(policiesToClean).length === 0) {
    logger.info(`[FleetPolicyRevisionsCleanupTask] No policies found with more than ${config.maxRevisions} revisions. Exiting cleanup task.`);
    return;
  }
  logger.info(`[FleetPolicyRevisionsCleanupTask] Found ${Object.keys(policiesToClean).length} policies with more than ${config.maxRevisions} revisions.`);
  throwIfAborted(abortController);
  const policiesRevisionSummaries = await (0, _populate_minimum_revisions_used_by_agents.populateMinimumRevisionsUsedByAgents)(esClient, policiesToClean, {
    ...context,
    config
  });
  throwIfAborted(abortController);
  const docCount = Object.values(policiesRevisionSummaries).reduce((sum, summary) => sum + (summary.count - config.maxRevisions), 0);
  logger.debug(`[FleetPolicyRevisionsCleanupTask] Attempting to delete potentially ${docCount >= config.maxDocsToDelete ? config.maxDocsToDelete : docCount} policy revision documents.`);
  const result = await (0, _delete_policy_revisions.deletePolicyRevisions)(esClient, policiesRevisionSummaries, {
    ...context,
    config
  });
  if (result) {
    logger.debug(`[FleetPolicyRevisionsCleanupTask] Deleted ${result.deleted} policy revision documents.`);
  }
  logger.debug('[FleetPolicyRevisionsCleanupTask] Cleanup completed');
  return {
    totalDeletedRevisions: result ? result.deleted : 0
  };
};
exports.cleanupPolicyRevisions = cleanupPolicyRevisions;
const throwIfAborted = abortController => {
  if (abortController !== null && abortController !== void 0 && abortController.signal.aborted) {
    throw new Error('Task was aborted');
  }
};
exports.throwIfAborted = throwIfAborted;