"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UpgradeAgentlessDeploymentsTask = exports.UPGRADE_AGENT_DEPLOYMENTS_TASK_VERSION = exports.UPGRADE_AGENTLESS_DEPLOYMENTS_TASK_TYPE = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _task = require("@kbn/task-manager-plugin/server/task");
var _services = require("../../services");
var _constants = require("../../constants");
var _agents = require("../../services/agents");
var _agentless_agent = require("../../services/agents/agentless_agent");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const UPGRADE_AGENTLESS_DEPLOYMENTS_TASK_TYPE = exports.UPGRADE_AGENTLESS_DEPLOYMENTS_TASK_TYPE = 'fleet:upgrade-agentless-deployments-task';
const UPGRADE_AGENT_DEPLOYMENTS_TASK_VERSION = exports.UPGRADE_AGENT_DEPLOYMENTS_TASK_VERSION = '1.0.0';
const TITLE = 'Fleet upgrade agentless deployments Task';
const TIMEOUT = '2m';
const INTERVAL = '1d';
const LOGGER_SUBJECT = '[UpgradeAgentlessDeploymentsTask]';
const BATCH_SIZE = 10;
const AGENTLESS_DEPLOYMENTS_SIZE = 40;
class UpgradeAgentlessDeploymentsTask {
  constructor(setupContract) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "startedTaskRunner", false);
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager
    }) => {
      if (!taskManager) {
        this.logger.error(`${LOGGER_SUBJECT} Missing required service during start`);
      }
      this.startedTaskRunner = true;
      this.logger.info(`${LOGGER_SUBJECT} Started with interval of [${INTERVAL}]`);
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: UPGRADE_AGENTLESS_DEPLOYMENTS_TASK_TYPE,
          schedule: {
            interval: INTERVAL
          },
          state: {},
          params: {
            version: UPGRADE_AGENT_DEPLOYMENTS_TASK_VERSION
          }
        });
      } catch (e) {
        this.logger.error(`Error scheduling task ${LOGGER_SUBJECT}, error: ${e.message}`, e);
      }
    });
    (0, _defineProperty2.default)(this, "processInBatches", async ({
      agentlessPolicies,
      agents,
      abortController
    }, batchSize, processFunction) => {
      if (!agents.length) {
        this.endRun('No agents found');
        return;
      }
      for (let i = 0; i < agentlessPolicies.length; i += batchSize) {
        const currentAgentPolicyBatch = agentlessPolicies.slice(i, i + batchSize);
        await Promise.allSettled(await currentAgentPolicyBatch.map(async agentPolicy => {
          const agentlessAgent = agents.find(agent => agent.policy_id === agentPolicy.id);
          if (!agentlessAgent) {
            this.endRun('No active online agentless agent found');
            return;
          }
          this.logger.info(`${LOGGER_SUBJECT} processing agentless agent ${JSON.stringify(agentlessAgent.agent)}`);
          if (abortController.signal.aborted) {
            this.logger.info(`${LOGGER_SUBJECT} Task runner canceled!`);
            abortController.signal.throwIfAborted();
          }
          return processFunction(agentPolicy, agentlessAgent);
        }));
        if (abortController.signal.aborted) {
          this.logger.info(`${LOGGER_SUBJECT} Task runner canceled!`);
          abortController.signal.throwIfAborted();
        }
      }
    });
    (0, _defineProperty2.default)(this, "processUpgradeAgentlessDeployments", async (esClient, soClient, abortController) => {
      const SAVED_OBJECT_TYPE = 'fleet-agent-policies';
      const policiesKuery = `${SAVED_OBJECT_TYPE}.supports_agentless: true`;
      try {
        const agentPolicyFetcher = await _services.agentPolicyService.fetchAllAgentPolicies(soClient, {
          kuery: policiesKuery,
          perPage: BATCH_SIZE,
          spaceId: '*'
        });
        this.logger.info(`[${LOGGER_SUBJECT}] running task to upgrade agentless deployments with kuery: ${policiesKuery}`);
        for await (const agentlessPolicies of agentPolicyFetcher) {
          this.logger.info(`[${LOGGER_SUBJECT}] Found "${agentlessPolicies.length}" agentless policies`);
          if (!agentlessPolicies.length) {
            this.endRun('Found no agentless policies to upgrade');
            return;
          }

          // Upgrade agentless deployments
          try {
            const kuery = `(${_constants.AGENTS_PREFIX}.policy_id:${agentlessPolicies.map(policy => `"${policy.id}"`).join(' or ')}) and ${_constants.AGENTS_PREFIX}.status:online`;
            const res = await (0, _agents.getAgentsByKuery)(esClient, soClient, {
              kuery,
              showInactive: false,
              page: 1,
              perPage: AGENTLESS_DEPLOYMENTS_SIZE
            });
            this.logger.info(`${LOGGER_SUBJECT} Found "${res.agents.length}" agentless agents`);
            await this.processInBatches({
              agentlessPolicies,
              agents: res.agents,
              abortController
            }, BATCH_SIZE, this.upgradeAgentlessDeployments);
          } catch (e) {
            this.logger.error(`${LOGGER_SUBJECT} Failed to get agentless agents error: ${e}`);
          }
          if (abortController.signal.aborted) {
            this.logger.info(`${LOGGER_SUBJECT} Task runner canceled!`);
            abortController.signal.throwIfAborted();
          }
        }
      } catch (e) {
        this.logger.error(`${LOGGER_SUBJECT} Failed to get agentless policies error: ${e}`);
      }
      this.logger.info(`${LOGGER_SUBJECT} [runTask()] finished`);
    });
    (0, _defineProperty2.default)(this, "upgradeAgentlessDeployments", async (agentPolicy, agent) => {
      this.logger.info(`Validating if agentless policy ${agentPolicy.id} needs to be upgraded`);

      // Compare the current agent version with the latest agent version And upgrade if necessary
      if (agent.status === 'online') {
        try {
          this.logger.info(`${LOGGER_SUBJECT} Requesting to check version and update agentless deployment for policy ${agentPolicy.id}`);
          await _agentless_agent.agentlessAgentService.upgradeAgentlessDeployment(agentPolicy.id);
          this.logger.info(`${LOGGER_SUBJECT} Successfully sent the upgrade deployment request for ${agentPolicy.id}`);
        } catch (e) {
          this.logger.error(`${LOGGER_SUBJECT} Failed to request an agentless deployment upgrade for ${agentPolicy.id} error: ${e}`);
          throw e;
        }
      } else {
        this.logger.info(`${LOGGER_SUBJECT} No upgrade request sent for agentless policy ${agentPolicy.id} because the agent status is ${agent.status}`);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core, abortController) => {
      const cloudSetup = _services.appContextService.getCloud();
      if (!this.startedTaskRunner) {
        this.logger.info(`${LOGGER_SUBJECT} runTask Aborted. Task not started yet`);
        return;
      }
      if (taskInstance.id !== this.taskId) {
        this.logger.info(`${LOGGER_SUBJECT} Outdated task version: Received [${taskInstance.id}] from task instance. Current version is [${this.taskId}]`);
        return (0, _task.getDeleteTaskRunResult)();
      }
      if (!_services.appContextService.getExperimentalFeatures().enabledUpgradeAgentlessDeploymentsTask) {
        this.endRun('Upgrade Agentless Deployments Task is disabled');
        return;
      }
      if (cloudSetup !== null && cloudSetup !== void 0 && cloudSetup.isServerlessEnabled) {
        this.endRun('Upgrading Agentless deployments is only supported in cloud');
        return;
      }
      this.logger.info(`[runTask()] started`);
      const [coreStart] = await core.getStartServices();
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = new _server.SavedObjectsClient(coreStart.savedObjects.createInternalRepository());
      await this.processUpgradeAgentlessDeployments(esClient, soClient, abortController);
    });
    const {
      core: _core,
      taskManager: _taskManager,
      logFactory
    } = setupContract;
    this.logger = logFactory.get(this.taskId);
    _taskManager.registerTaskDefinitions({
      [UPGRADE_AGENTLESS_DEPLOYMENTS_TASK_TYPE]: {
        title: TITLE,
        timeout: TIMEOUT,
        createTaskRunner: ({
          taskInstance,
          abortController
        }) => {
          return {
            run: async () => {
              return this.runTask(taskInstance, _core, abortController);
            },
            cancel: async () => {}
          };
        }
      }
    });
  }
  get taskId() {
    return `${UPGRADE_AGENTLESS_DEPLOYMENTS_TASK_TYPE}:${UPGRADE_AGENT_DEPLOYMENTS_TASK_VERSION}`;
  }
  endRun(msg = '') {
    this.logger.info(`${LOGGER_SUBJECT} runTask ended${msg ? ': ' + msg : ''}`);
  }
}
exports.UpgradeAgentlessDeploymentsTask = UpgradeAgentlessDeploymentsTask;