"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerUpgradeManagedPackagePoliciesTask = registerUpgradeManagedPackagePoliciesTask;
exports.upgradeManagedPackagePolicies = exports.setupUpgradeManagedPackagePolicies = void 0;
var _lt = _interopRequireDefault(require("semver/functions/lt"));
var _constants = require("../../constants");
var _app_context = require("../app_context");
var _packages = require("../epm/packages");
var _package_policy = require("../package_policy");
var _cache = require("../epm/packages/cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'fleet:setup:upgrade_managed_package_policies';
function registerUpgradeManagedPackagePoliciesTask(taskManagerSetup) {
  taskManagerSetup.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: 'Fleet Setup Upgrade managed package policies',
      timeout: '1h',
      maxAttempts: 1,
      createTaskRunner: ({
        taskInstance
      }) => {
        const pkgName = taskInstance.params.packageName;
        return {
          async run() {
            const esClient = _app_context.appContextService.getInternalUserESClient();
            const soClient = _app_context.appContextService.getInternalUserSOClientWithoutSpaceExtension();
            await (0, _cache.runWithCache)(() => upgradeManagedPackagePolicies(soClient, esClient, pkgName));
          },
          async cancel() {}
        };
      }
    }
  });
}
async function runUpgradeManagedPackagePoliciesTask(taskManagerStart, pkgName) {
  await taskManagerStart.ensureScheduled({
    id: `${TASK_TYPE}:${pkgName}`,
    scope: ['fleet'],
    params: {
      packageName: pkgName
    },
    taskType: TASK_TYPE,
    runAt: new Date(Date.now() + 3 * 1000),
    state: {}
  });
}

/**
 *
 * @param soClient
 * @param esClient
 * @returns
 */
const setupUpgradeManagedPackagePolicies = async (soClient, esClient) => {
  _app_context.appContextService.getLogger().debug('Scheduling required package policies upgrades for managed policies');
  const installedPackages = await (0, _packages.getInstallations)(soClient, {
    filter: `${_constants.PACKAGES_SAVED_OBJECT_TYPE}.attributes.install_status:installed AND ${_constants.PACKAGES_SAVED_OBJECT_TYPE}.attributes.keep_policies_up_to_date:true`
  });
  for (const {
    attributes: installedPackage
  } of installedPackages.saved_objects) {
    const packagePoliciesFinder = await getPackagePoliciesNotMatchingVersion(soClient, installedPackage.name, installedPackage.version);
    let shouldRegisterTask = false;
    for await (const packagePolicies of packagePoliciesFinder) {
      for (const packagePolicy of packagePolicies) {
        if (isPolicyVersionLtInstalledVersion(packagePolicy, installedPackage)) {
          shouldRegisterTask = true;
          break;
        }
      }
      if (shouldRegisterTask) {
        break;
      }
    }
    if (shouldRegisterTask) {
      _app_context.appContextService.getLogger().debug(`Scheduled package policies upgrades for package: ${installedPackage.name}@${installedPackage.version}`);
      await runUpgradeManagedPackagePoliciesTask(_app_context.appContextService.getTaskManagerStart(), installedPackage.name);
    }
  }
};

/**
 * Upgrade any package policies for packages installed through setup that are denoted as `AUTO_UPGRADE` packages
 * or have the `keep_policies_up_to_date` flag set to `true`
 */
exports.setupUpgradeManagedPackagePolicies = setupUpgradeManagedPackagePolicies;
const upgradeManagedPackagePolicies = async (soClient, esClient, pkgName) => {
  _app_context.appContextService.getLogger().debug('Running required package policies upgrades for managed policies');
  const results = [];
  const installedPackage = await (0, _packages.getInstallation)({
    pkgName,
    savedObjectsClient: soClient,
    logger: _app_context.appContextService.getLogger()
  });
  if (!installedPackage) {
    _app_context.appContextService.getLogger().debug('Aborting upgrading managed package policies: package is not installed');
    return [];
  }
  const packagePoliciesFinder = await getPackagePoliciesNotMatchingVersion(soClient, installedPackage.name, installedPackage.version);
  for await (const packagePolicies of packagePoliciesFinder) {
    for (const packagePolicy of packagePolicies) {
      if (isPolicyVersionLtInstalledVersion(packagePolicy, installedPackage)) {
        await upgradePackagePolicy(soClient, esClient, packagePolicy, installedPackage, results);
      }
    }
  }
  return results;
};
exports.upgradeManagedPackagePolicies = upgradeManagedPackagePolicies;
async function getPackagePoliciesNotMatchingVersion(soClient, pkgName, pkgVersion) {
  return _package_policy.packagePolicyService.fetchAllItems(soClient, {
    perPage: 50,
    kuery: `${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${pkgName} AND NOT ${_constants.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.version:${pkgVersion}`
  });
}
function isPolicyVersionLtInstalledVersion(packagePolicy, installedPackage) {
  return packagePolicy.package !== undefined && (0, _lt.default)(packagePolicy.package.version, installedPackage.version);
}
async function upgradePackagePolicy(soClient, esClient, packagePolicy, installedPackage, results) {
  // Since upgrades don't report diffs/errors, we need to perform a dry run first in order
  // to notify the user of any granular policy upgrade errors that occur during Fleet's
  // preconfiguration check
  const dryRunResults = await _package_policy.packagePolicyService.getUpgradeDryRunDiff(soClient, packagePolicy.id, packagePolicy, installedPackage.version);
  if (dryRunResults.hasErrors) {
    var _dryRunResults$diff, _dryRunResults$body;
    const errors = dryRunResults.diff ? (_dryRunResults$diff = dryRunResults.diff) === null || _dryRunResults$diff === void 0 ? void 0 : _dryRunResults$diff[1].errors : [(_dryRunResults$body = dryRunResults.body) === null || _dryRunResults$body === void 0 ? void 0 : _dryRunResults$body.message];
    _app_context.appContextService.getLogger().error(new Error(`Error upgrading package policy ${packagePolicy.id}: ${JSON.stringify(errors)}`));
    results.push({
      packagePolicyId: packagePolicy.id,
      diff: dryRunResults.diff,
      errors
    });
    return;
  }
  try {
    await _package_policy.packagePolicyService.upgrade(soClient, esClient, packagePolicy.id, {
      force: true
    }, packagePolicy, installedPackage.version);
    results.push({
      packagePolicyId: packagePolicy.id,
      diff: dryRunResults.diff,
      errors: []
    });
  } catch (error) {
    results.push({
      packagePolicyId: packagePolicy.id,
      diff: dryRunResults.diff,
      errors: [error]
    });
  }
}