"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._getUpgradePackagePolicyInfo = _getUpgradePackagePolicyInfo;
exports._packagePoliciesBulkUpgrade = _packagePoliciesBulkUpgrade;
exports._packagePoliciesGetUpgradeDryRunDiff = _packagePoliciesGetUpgradeDryRunDiff;
exports._packagePoliciesUpgrade = _packagePoliciesUpgrade;
var _lodash = require("lodash");
var _fp = require("lodash/fp");
var _pMap = _interopRequireDefault(require("p-map"));
var _i18n = require("@kbn/i18n");
var _lt = _interopRequireDefault(require("semver/functions/lt"));
var _cache = require("../epm/packages/cache");
var _constants = require("../../constants");
var _packages = require("../epm/packages");
var _errors = require("../../errors");
var _package_policy = require("../package_policy");
var _services = require("../../../common/services");
var _get = require("../epm/packages/get");
var _app_context = require("../app_context");
var _agent_policies = require("../agent_policies");
var _upgrade_sender = require("../upgrade_sender");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function _getUpgradePackagePolicyInfo({
  packagePolicyService,
  soClient,
  id,
  packagePolicy,
  pkgVersion
}) {
  if (!packagePolicy) {
    var _await$packagePolicyS;
    packagePolicy = (_await$packagePolicyS = await packagePolicyService.get(soClient, id)) !== null && _await$packagePolicyS !== void 0 ? _await$packagePolicyS : undefined;
  }
  let experimentalDataStreamFeatures = [];
  if (!pkgVersion && packagePolicy) {
    var _installedPackage$exp;
    const installedPackage = await (0, _packages.getInstallation)({
      savedObjectsClient: soClient,
      pkgName: packagePolicy.package.name
    });
    if (!installedPackage) {
      throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.packagePolicy.packageNotInstalledError', {
        defaultMessage: 'Package {name} is not installed',
        values: {
          name: packagePolicy.package.name
        }
      }));
    }
    pkgVersion = installedPackage.version;
    experimentalDataStreamFeatures = (_installedPackage$exp = installedPackage.experimental_data_stream_features) !== null && _installedPackage$exp !== void 0 ? _installedPackage$exp : [];
  }
  let packageInfo;
  if (packagePolicy) {
    var _pkgVersion;
    packageInfo = await (0, _packages.getPackageInfo)({
      savedObjectsClient: soClient,
      pkgName: packagePolicy.package.name,
      pkgVersion: (_pkgVersion = pkgVersion) !== null && _pkgVersion !== void 0 ? _pkgVersion : '',
      prerelease: !!pkgVersion // using prerelease only if version is specified
    });
  }
  validateUpgradePackagePolicy(id, packageInfo, packagePolicy);
  return {
    packagePolicy: packagePolicy,
    packageInfo: packageInfo,
    experimentalDataStreamFeatures
  };
}
async function doUpgrade(soClient, esClient, packagePolicy, packageInfo, options) {
  if (packagePolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
    throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot upgrade package policy ${packagePolicy.id}`);
  }
  const updatePackagePolicy = (0, _package_policy.updatePackageInputs)({
    ...(0, _lodash.omit)(packagePolicy, 'id', 'spaceIds', 'secret_references'),
    inputs: packagePolicy.inputs,
    package: {
      ...packagePolicy.package,
      version: packageInfo.version
    }
  }, packageInfo, (0, _services.packageToPackagePolicyInputs)(packageInfo));
  const assetsMap = await (0, _get.getAgentTemplateAssetsMap)({
    logger: _app_context.appContextService.getLogger(),
    packageInfo,
    savedObjectsClient: soClient
  });
  updatePackagePolicy.inputs = (0, _package_policy._compilePackagePolicyInputs)(packageInfo, updatePackagePolicy.vars || {}, updatePackagePolicy.inputs, assetsMap);
  updatePackagePolicy.elasticsearch = packageInfo.elasticsearch;
  return updatePackagePolicy;
}
async function getPackagePoliciesWithRelatedPackageVersionForUpgrade({
  soClient,
  packagePolicyService,
  ids,
  pkgVersion
}) {
  const packagePolicies = await packagePolicyService.getByIDs(soClient, ids, {
    ignoreMissing: true
  });
  const packagePoliciesById = (0, _fp.indexBy)('id', packagePolicies);
  const packageVersionMap = new Map();
  if (!pkgVersion) {
    var _packagePolicies$map$;
    const packageNames = (_packagePolicies$map$ = packagePolicies === null || packagePolicies === void 0 ? void 0 : packagePolicies.map(policy => {
      var _policy$package;
      return (_policy$package = policy.package) === null || _policy$package === void 0 ? void 0 : _policy$package.name;
    }).filter(packageName => typeof packageName !== 'undefined')) !== null && _packagePolicies$map$ !== void 0 ? _packagePolicies$map$ : [];
    await (0, _pMap.default)((0, _fp.uniq)(packageNames), async pkgName => {
      const installation = await (0, _packages.getInstallation)({
        savedObjectsClient: soClient,
        pkgName
      });
      if (installation) {
        packageVersionMap.set(pkgName, installation.version);
      }
    }, {
      concurrency: _constants.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS_10
    });
  }
  return {
    packageVersionMap,
    packagePoliciesById,
    packagePolicies
  };
}
function validateUpgradePackagePolicy(id, packageInfo, packagePolicy) {
  var _packagePolicy$packag, _packageInfo$version;
  if (!packagePolicy) {
    throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.packagePolicy.policyNotFoundError', {
      defaultMessage: 'Package policy with id {id} not found',
      values: {
        id
      }
    }));
  }
  if (!((_packagePolicy$packag = packagePolicy.package) !== null && _packagePolicy$packag !== void 0 && _packagePolicy$packag.name)) {
    throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.packagePolicy.packageNotFoundError', {
      defaultMessage: 'Package policy with id {id} has no named package',
      values: {
        id
      }
    }));
  }
  const isInstalledVersionLessThanPolicyVersion = (0, _lt.default)((_packageInfo$version = packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.version) !== null && _packageInfo$version !== void 0 ? _packageInfo$version : '', packagePolicy.package.version);
  if (isInstalledVersionLessThanPolicyVersion) {
    throw new _errors.PackagePolicyIneligibleForUpgradeError(_i18n.i18n.translate('xpack.fleet.packagePolicy.ineligibleForUpgradeError', {
      defaultMessage: "Package policy {id}'s package version {version} of package {name} is newer than the installed package version. Please install the latest version of {name}.",
      values: {
        id: packagePolicy.id,
        name: packagePolicy.package.name,
        version: packagePolicy.package.version
      }
    }));
  }
}
function sendUpgradeTelemetry(packagePolicyPackage, latestVersion, hasErrors, errors) {
  if (packagePolicyPackage.version !== latestVersion) {
    const upgradeTelemetry = {
      packageName: packagePolicyPackage.name,
      currentVersion: packagePolicyPackage.version,
      newVersion: latestVersion,
      status: hasErrors ? 'failure' : 'success',
      error: hasErrors ? errors : undefined,
      dryRun: true,
      eventType: 'package-policy-upgrade'
    };
    (0, _upgrade_sender.sendTelemetryEvents)(_app_context.appContextService.getLogger(), _app_context.appContextService.getTelemetryEventsSender(), upgradeTelemetry);
    _app_context.appContextService.getLogger().info(`Package policy upgrade dry run ${hasErrors ? 'resulted in errors' : 'ran successfully'}`);
    _app_context.appContextService.getLogger().debug(() => JSON.stringify(upgradeTelemetry));
  }
}
async function calculateDiff(packagePolicy, packageInfo, assetsMap) {
  var _packagePolicy$packag2;
  const updatedPackagePolicy = (0, _package_policy.updatePackageInputs)({
    ...(0, _lodash.omit)(packagePolicy, 'id', 'spaceIds'),
    inputs: packagePolicy.inputs,
    package: {
      ...packagePolicy.package,
      version: packageInfo.version,
      experimental_data_stream_features: (_packagePolicy$packag2 = packagePolicy.package) === null || _packagePolicy$packag2 === void 0 ? void 0 : _packagePolicy$packag2.experimental_data_stream_features
    }
  }, packageInfo, (0, _services.packageToPackagePolicyInputs)(packageInfo), true);
  // Validate input are valid when compiling agent template
  try {
    updatedPackagePolicy.inputs = (0, _package_policy._compilePackagePolicyInputs)(packageInfo, updatedPackagePolicy.vars || {}, updatedPackagePolicy.inputs, assetsMap);
  } catch (error) {
    if (!updatedPackagePolicy.errors) {
      updatedPackagePolicy.errors = [];
    }
    updatedPackagePolicy.errors.push({
      key: undefined,
      message: error.message
    });
  }
  updatedPackagePolicy.elasticsearch = packageInfo.elasticsearch;
  const hasErrors = 'errors' in updatedPackagePolicy;
  sendUpgradeTelemetry(packagePolicy.package, packageInfo.version, hasErrors, updatedPackagePolicy.errors);
  return {
    name: updatedPackagePolicy.name,
    diff: [packagePolicy, updatedPackagePolicy],
    // TODO: Currently only returns the agent inputs for current package policy, not the upgraded one
    // as we only show this version in the UI
    agent_diff: [(0, _agent_policies.storedPackagePolicyToAgentInputs)(packagePolicy, packageInfo)],
    hasErrors
  };
}
async function _packagePoliciesBulkUpgrade({
  packagePolicyService,
  soClient,
  esClient,
  ids,
  options,
  pkgVersion
}) {
  // Bulk upgrade packages policies
  // 1. bulk get policies
  // 2. get pkg version if not specificed based on installed version
  // 3. for each policies get upgraded
  // 4. bulkUpdate policies
  return (0, _cache.runWithCache)(async () => {
    const result = [];
    for (const chunkedIds of (0, _fp.chunk)(_constants.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS, ids)) {
      const {
        packagePoliciesById,
        packagePolicies,
        packageVersionMap
      } = await getPackagePoliciesWithRelatedPackageVersionForUpgrade({
        soClient,
        packagePolicyService,
        ids: chunkedIds,
        pkgVersion
      });
      const updatedPackagePolicies = [];
      for (const id of chunkedIds) {
        try {
          var _packagePolicy$packag3, _packagePolicy$packag4;
          const packagePolicy = packagePoliciesById[id];
          if (!packagePolicy) {
            throw new _errors.FleetNotFoundError(`Package policy ${id} not found`);
          }
          const upgradePkgVersion = pkgVersion ? pkgVersion : packageVersionMap.get((_packagePolicy$packag3 = (_packagePolicy$packag4 = packagePolicy.package) === null || _packagePolicy$packag4 === void 0 ? void 0 : _packagePolicy$packag4.name) !== null && _packagePolicy$packag3 !== void 0 ? _packagePolicy$packag3 : '');
          const {
            packagePolicy: currentPackagePolicy,
            packageInfo
          } = await _getUpgradePackagePolicyInfo({
            packagePolicyService,
            soClient,
            id,
            packagePolicy,
            pkgVersion: upgradePkgVersion
          });
          const updatePackagePolicy = await doUpgrade(soClient, esClient, currentPackagePolicy, packageInfo, {
            force: options === null || options === void 0 ? void 0 : options.force
          });
          updatedPackagePolicies.push({
            ...updatePackagePolicy,
            id
          });
        } catch (error) {
          result.push({
            id,
            success: false,
            ...(0, _errors.fleetErrorToResponseOptions)(error)
          });
        }
      }
      const bulkUpdateResults = await packagePolicyService.bulkUpdate(soClient, esClient, updatedPackagePolicies, {
        asyncDeploy: true,
        fromBulkUpgrade: true,
        force: options === null || options === void 0 ? void 0 : options.force,
        user: options === null || options === void 0 ? void 0 : options.user,
        oldPackagePolicies: packagePolicies
      });
      for (const res of (_bulkUpdateResults$up = bulkUpdateResults.updatedPolicies) !== null && _bulkUpdateResults$up !== void 0 ? _bulkUpdateResults$up : []) {
        var _bulkUpdateResults$up;
        result.push({
          id: res.id,
          name: res.name,
          success: true
        });
      }
      for (const res of (_bulkUpdateResults$fa = bulkUpdateResults.failedPolicies) !== null && _bulkUpdateResults$fa !== void 0 ? _bulkUpdateResults$fa : []) {
        var _bulkUpdateResults$fa;
        result.push({
          id: res.packagePolicy.id,
          success: false,
          ...(0, _errors.fleetErrorToResponseOptions)(res.error)
        });
      }
    }
    return result;
  });
}
async function _packagePoliciesUpgrade({
  soClient,
  esClient,
  packagePolicyService,
  id,
  options,
  packagePolicy,
  pkgVersion
}) {
  return (0, _cache.runWithCache)(async () => {
    const result = [];
    try {
      const {
        packagePolicy: currentPackagePolicy,
        packageInfo
      } = await _getUpgradePackagePolicyInfo({
        packagePolicyService,
        soClient,
        id,
        packagePolicy,
        pkgVersion
      });
      const updatePackagePolicy = await doUpgrade(soClient, esClient, currentPackagePolicy, packageInfo, {
        force: options === null || options === void 0 ? void 0 : options.force
      });
      const updateOptions = {
        skipUniqueNameVerification: true,
        ...options
      };
      await packagePolicyService.update(soClient, esClient, id, updatePackagePolicy, updateOptions);
      result.push({
        id,
        name: updatePackagePolicy.name,
        success: true
      });
    } catch (error) {
      result.push({
        id,
        success: false,
        ...(0, _errors.fleetErrorToResponseOptions)(error)
      });
    }
    return result;
  });
}
async function _packagePoliciesGetUpgradeDryRunDiff({
  soClient,
  packagePolicyService,
  id,
  packagePolicy,
  pkgVersion
}) {
  try {
    let packageInfo;
    let experimentalDataStreamFeatures;
    ({
      packagePolicy,
      packageInfo,
      experimentalDataStreamFeatures
    } = await _getUpgradePackagePolicyInfo({
      packagePolicyService,
      soClient,
      id,
      packagePolicy,
      pkgVersion
    }));
    const assetsMap = await (0, _get.getAgentTemplateAssetsMap)({
      logger: _app_context.appContextService.getLogger(),
      packageInfo,
      savedObjectsClient: soClient
    });

    // Ensure the experimental features from the Installation saved object come through on the package policy
    // during an upgrade dry run
    if (packagePolicy.package) {
      packagePolicy.package.experimental_data_stream_features = experimentalDataStreamFeatures;
    }
    return calculateDiff(packagePolicy, packageInfo, assetsMap);
  } catch (error) {
    return {
      hasErrors: true,
      ...(0, _errors.fleetErrorToResponseOptions)(error)
    };
  }
}