"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncAgentlessDeployments = syncAgentlessDeployments;
var _pMap = _interopRequireDefault(require("p-map"));
var _ = require("..");
var _constants = require("../../constants");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AGENTLESS_CONCURRENCY = 1;
const PAGE_SIZE = 20;
function dryRunTag(dryRun = false) {
  if (dryRun) {
    return `[Dry Run]`;
  }
  return '';
}
async function syncAgentlessDeployments({
  logger,
  agentlessAgentService
}, opts) {
  logger.info(`[Agentless Deployment Sync] Starting sync process`);
  const soClient = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
  try {
    var _opts$abortController3;
    // retrieve all agentless deployments
    const currentDeployments = [];
    let hasMore = true;
    let nextPageToken;
    while (hasMore) {
      var _opts$abortController;
      if (opts !== null && opts !== void 0 && (_opts$abortController = opts.abortController) !== null && _opts$abortController !== void 0 && _opts$abortController.signal.aborted) {
        throw new Error('Task was aborted');
      }
      const deploymentRes = await agentlessAgentService.listAgentlessDeployments({
        perPage: PAGE_SIZE,
        nextPageToken
      });
      if (!deploymentRes) {
        return;
      }
      nextPageToken = deploymentRes.next_token;
      if (!nextPageToken) {
        hasMore = false;
      }
      const policyIds = deploymentRes.deployments.map(d => d.policy_id);
      const agentPolicies = await _.agentPolicyService.getByIds(soClient, policyIds, {
        spaceId: '*',
        ignoreMissing: true
      });
      await (0, _pMap.default)(deploymentRes.deployments, async deployment => {
        var _opts$abortController2, _agentPolicy$monitori, _agentPolicy$monitori2;
        if (opts !== null && opts !== void 0 && (_opts$abortController2 = opts.abortController) !== null && _opts$abortController2 !== void 0 && _opts$abortController2.signal.aborted) {
          throw new Error('Task was aborted');
        }
        const agentPolicy = agentPolicies.find(ap => ap.id === deployment.policy_id);
        if (!agentPolicy) {
          logger.info(`[Agentless Deployment Sync]${dryRunTag(opts === null || opts === void 0 ? void 0 : opts.dryRun)} Deleting deployment ${deployment.policy_id}`);
          if (!(opts !== null && opts !== void 0 && opts.dryRun)) {
            await agentlessAgentService.deleteAgentlessAgent(deployment.policy_id).catch(error => {
              logger.error(`[Agentless Deployment Sync] Failed to delete deployment ${deployment.policy_id}`, {
                error
              });
            });
          }
        } else if (!agentPolicy.keep_monitoring_alive || ((_agentPolicy$monitori = (_agentPolicy$monitori2 = agentPolicy.monitoring_enabled) === null || _agentPolicy$monitori2 === void 0 ? void 0 : _agentPolicy$monitori2.length) !== null && _agentPolicy$monitori !== void 0 ? _agentPolicy$monitori : 0) > 0) {
          const esClient = _.appContextService.getInternalUserESClient();
          const spacedScoppedSoClient = _.appContextService.getInternalUserSOClientForSpaceId(agentPolicy.namespace || undefined);
          logger.info(`[Agentless Deployment Sync]${dryRunTag(opts === null || opts === void 0 ? void 0 : opts.dryRun)} Updating agentless policy monitoring settings ${agentPolicy.id}`);
          await _.agentPolicyService.update(spacedScoppedSoClient, esClient, agentPolicy.id, {
            // Ensure agentless policies keep monitoring alive so http monitoring server continue to work even without monitoring enabled
            keep_monitoring_alive: true,
            monitoring_enabled: []
          });
        } else if (deployment.revision_idx === undefined || deployment.revision_idx < agentPolicy.revision) {
          logger.info(`[Agentless Deployment Sync]${dryRunTag(opts === null || opts === void 0 ? void 0 : opts.dryRun)} Updating deployment ${deployment.policy_id}`);
          if (!(opts !== null && opts !== void 0 && opts.dryRun)) {
            const spacedScoppedSoClient = _.appContextService.getInternalUserSOClientForSpaceId(agentPolicy.namespace || undefined);
            const esClient = _.appContextService.getInternalUserESClient();
            await agentlessAgentService.createAgentlessAgent(esClient, spacedScoppedSoClient, agentPolicy).catch(error => {
              logger.error(`[Agentless Deployment Sync] Failed to update deployment ${deployment.policy_id}`, {
                error
              });
            });
          }
        }
      }, {
        concurrency: AGENTLESS_CONCURRENCY
      });
      currentDeployments.push(...deploymentRes.deployments);
    }
    const agentlessPolicies = await _.agentPolicyService.list(soClient, {
      perPage: _constants.SO_SEARCH_LIMIT,
      fields: ['revision', 'supports_agentless', 'global_data_tags', 'fleet_server_host_id', 'agentless'],
      kuery: `${_constants.AGENT_POLICY_SAVED_OBJECT_TYPE}.supports_agentless:true`,
      spaceId: '*'
    });
    if (opts !== null && opts !== void 0 && (_opts$abortController3 = opts.abortController) !== null && _opts$abortController3 !== void 0 && _opts$abortController3.signal.aborted) {
      throw new Error('Task was aborted');
    }
    await (0, _pMap.default)(agentlessPolicies.items, async agentPolicy => {
      var _opts$abortController4;
      if (opts !== null && opts !== void 0 && (_opts$abortController4 = opts.abortController) !== null && _opts$abortController4 !== void 0 && _opts$abortController4.signal.aborted) {
        throw new Error('Task was aborted');
      }
      const deployment = currentDeployments.find(d => d.policy_id === agentPolicy.id);
      if (!deployment) {
        logger.info(`[Agentless Deployment Sync]${dryRunTag(opts === null || opts === void 0 ? void 0 : opts.dryRun)} Creating deployment for policy ${agentPolicy.id}`);
        if (!(opts !== null && opts !== void 0 && opts.dryRun)) {
          const spacedScoppedSoClient = _.appContextService.getInternalUserSOClientForSpaceId(agentPolicy.namespace || undefined);
          const esClient = _.appContextService.getInternalUserESClient();
          await agentlessAgentService.createAgentlessAgent(esClient, spacedScoppedSoClient, agentPolicy).catch(error => {
            logger.error(`[Agentless Deployment Sync] Failed to create deployment ${agentPolicy.id}`, {
              error
            });
          });
        }
      }
    }, {
      concurrency: AGENTLESS_CONCURRENCY
    });
    logger.info(`[Agentless Deployment Sync] Finished sync process`);
  } catch (error) {
    if (error instanceof _errors.AgentlessAgentListNotFoundError) {
      logger.warn(`[Agentless Deployment Sync] Skipping sync agentless list API not found`, {
        error
      });
      return;
    }
    logger.error(`[Agentless Deployment Sync] Sync failed with error`, {
      error
    });
    throw error;
  }
}