"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postBulkUpgradePackagesHandler = exports.postBulkUninstallPackagesHandler = exports.postBulkRollbackPackagesHandler = exports.getPackagePolicyIdsForCurrentUser = exports.getOneBulkOperationPackagesHandler = void 0;
var _services = require("../../services");
var _http_authorization_header = require("../../../common/http_authorization_header");
var _get = require("../../services/epm/packages/get");
var _errors = require("../../errors");
var _packages_bulk_operations = require("../../tasks/packages_bulk_operations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function validateInstalledPackages(savedObjectsClient, packages, operation) {
  const pkgNames = packages.map(({
    name
  }) => name);
  const installations = await (0, _get.getInstallationsByName)({
    savedObjectsClient,
    pkgNames
  });
  const nonInstalledPackages = pkgNames.filter(pkgName => !installations.some(installation => installation.name === pkgName));
  if (nonInstalledPackages.length) {
    throw new _errors.FleetError(`Cannot ${operation} non-installed packages: ${nonInstalledPackages.join(', ')}`);
  }
}
function getTaskManagerStart() {
  const taskManagerStart = _services.appContextService.getTaskManagerStart();
  if (!taskManagerStart) {
    throw new Error('Task manager not defined');
  }
  return taskManagerStart;
}
const postBulkUpgradePackagesHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const savedObjectsClient = fleetContext.internalSoClient;
  const spaceId = fleetContext.spaceId;
  const user = _services.appContextService.getSecurityCore().authc.getCurrentUser(request) || undefined;
  const authorizationHeader = _http_authorization_header.HTTPAuthorizationHeader.parseFromRequest(request, user === null || user === void 0 ? void 0 : user.username);
  const taskManagerStart = getTaskManagerStart();
  await validateInstalledPackages(savedObjectsClient, request.body.packages, 'upgrade');
  const taskId = await (0, _packages_bulk_operations.scheduleBulkUpgrade)(taskManagerStart, {
    authorizationHeader,
    spaceId,
    packages: request.body.packages,
    upgradePackagePolicies: request.body.upgrade_package_policies,
    force: request.body.force,
    prerelease: request.body.prerelease
  });
  const body = {
    taskId
  };
  return response.ok({
    body
  });
};
exports.postBulkUpgradePackagesHandler = postBulkUpgradePackagesHandler;
const postBulkUninstallPackagesHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const savedObjectsClient = fleetContext.internalSoClient;
  const taskManagerStart = getTaskManagerStart();
  await validateInstalledPackages(savedObjectsClient, request.body.packages, 'uninstall');
  const taskId = await (0, _packages_bulk_operations.scheduleBulkUninstall)(taskManagerStart, {
    packages: request.body.packages,
    force: request.body.force
  });
  const body = {
    taskId
  };
  return response.ok({
    body
  });
};
exports.postBulkUninstallPackagesHandler = postBulkUninstallPackagesHandler;
const getOneBulkOperationPackagesHandler = async (context, request, response) => {
  const taskManagerStart = getTaskManagerStart();
  const results = await (0, _packages_bulk_operations.getBulkOperationTaskResults)(taskManagerStart, request.params.taskId);
  const body = {
    status: results.status,
    error: results.error,
    results: results.results
  };
  return response.ok({
    body
  });
};
exports.getOneBulkOperationPackagesHandler = getOneBulkOperationPackagesHandler;
const getPackagePolicyIdsForCurrentUser = async (request, packages) => {
  const soClient = _services.appContextService.getInternalUserSOClient(request);
  const packagePolicyIdsByPackageName = {};
  for (const pkg of packages) {
    const packagePolicySORes = await _services.packagePolicyService.getPackagePolicySavedObjects(soClient, {
      searchFields: ['package.name'],
      search: pkg.name,
      spaceIds: ['*'],
      fields: ['id', 'name']
    });
    packagePolicyIdsByPackageName[pkg.name] = packagePolicySORes.saved_objects.map(so => so.id);
  }
  return packagePolicyIdsByPackageName;
};
exports.getPackagePolicyIdsForCurrentUser = getPackagePolicyIdsForCurrentUser;
const postBulkRollbackPackagesHandler = async (context, request, response) => {
  const fleetContext = await context.fleet;
  const savedObjectsClient = fleetContext.internalSoClient;
  const spaceId = fleetContext.spaceId;
  if (!_services.licenseService.isEnterprise()) {
    throw new _errors.FleetUnauthorizedError('Rollback integration requires an enterprise license.');
  }
  const taskManagerStart = getTaskManagerStart();
  await validateInstalledPackages(savedObjectsClient, request.body.packages, 'rollback');
  const taskId = await (0, _packages_bulk_operations.scheduleBulkRollback)(taskManagerStart, {
    packages: request.body.packages,
    spaceId,
    packagePolicyIdsForCurrentUser: await getPackagePolicyIdsForCurrentUser(request, request.body.packages)
  });
  const body = {
    taskId
  };
  return response.ok({
    body
  });
};
exports.postBulkRollbackPackagesHandler = postBulkRollbackPackagesHandler;