"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../constants");
var _constants2 = require("../../../common/constants");
var _types = require("../../types");
var _errors = require("../schema/errors");
var _api_privileges = require("../../constants/api_privileges");
var _utils = require("../schema/utils");
var _handler = require("./handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerRoutes = router => {
  router.versioned.get({
    path: _constants.ENROLLMENT_API_KEY_ROUTES.INFO_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Get an enrollment API key`,
    description: `Get an enrollment API key by ID.`,
    options: {
      tags: ['oas-tag:Fleet enrollment API keys']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetOneEnrollmentAPIKeyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.EnrollmentAPIKeyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getOneEnrollmentApiKeyHandler);
  router.versioned.delete({
    path: _constants.ENROLLMENT_API_KEY_ROUTES.DELETE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Revoke an enrollment API key`,
    description: `Revoke an enrollment API key by ID by marking it as inactive.`,
    options: {
      tags: ['oas-tag:Fleet enrollment API keys']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: _types.DeleteEnrollmentAPIKeyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.DeleteEnrollmentAPIKeyResponseSchema
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.deleteEnrollmentApiKeyHandler);
  router.versioned.get({
    path: _constants.ENROLLMENT_API_KEY_ROUTES.LIST_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [{
          anyRequired: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL, _api_privileges.FLEET_API_PRIVILEGES.SETUP]
        }]
      }
    },
    summary: `Get enrollment API keys`,
    options: {
      tags: ['oas-tag:Fleet enrollment API keys']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: _types.GetEnrollmentAPIKeysRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => (0, _utils.ListResponseSchema)(_types.EnrollmentAPIKeySchema).extends({
            list: _configSchema.schema.arrayOf(_types.EnrollmentAPIKeySchema, {
              meta: {
                deprecated: true
              }
            })
          })
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.getEnrollmentApiKeysHandler);
  router.versioned.post({
    path: _constants.ENROLLMENT_API_KEY_ROUTES.CREATE_PATTERN,
    security: {
      authz: {
        requiredPrivileges: [_api_privileges.FLEET_API_PRIVILEGES.AGENTS.ALL]
      }
    },
    summary: `Create an enrollment API key`,
    options: {
      tags: ['oas-tag:Fleet enrollment API keys']
    }
  }).addVersion({
    version: _constants2.API_VERSIONS.public.v1,
    validate: {
      request: _types.PostEnrollmentAPIKeyRequestSchema,
      response: {
        200: {
          description: 'OK: A successful request.',
          body: () => _types.EnrollmentAPIKeyResponseSchema.extends({
            action: _configSchema.schema.literal('created')
          })
        },
        400: {
          description: 'A bad request.',
          body: _errors.genericErrorResponse
        }
      }
    }
  }, _handler.postEnrollmentApiKeyHandler);
};
exports.registerRoutes = registerRoutes;