"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getListHandler = exports.getDeprecatedILMCheckHandler = void 0;
var _lodash = require("lodash");
var _pMap = _interopRequireDefault(require("p-map"));
var _configSchema = require("@kbn/config-schema");
var _elasticsearch = require("@elastic/elasticsearch");
var _types = require("../../../common/types");
var _get = require("../../services/epm/packages/get");
var _data_streams = require("../../services/data_streams");
var _constants = require("../../constants");
var _services = require("../../services");
var _errors = require("../../errors");
var _get_data_streams_query_metadata = require("./get_data_streams_query_metadata");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MANAGED_BY = 'fleet';
const LEGACY_MANAGED_BY = 'ingest-manager';
const getListHandler = async (context, request, response) => {
  try {
    var _appContextService$ge, _appContextService$ge2;
    // Query datastreams as the current user as the Kibana internal user may not have all the required permission
    const {
      savedObjects,
      elasticsearch
    } = await context.core;
    const esClient = elasticsearch.client.asCurrentUser;
    const body = {
      data_streams: []
    };
    const useMeteringApi = (_appContextService$ge = _services.appContextService.getConfig()) === null || _appContextService$ge === void 0 ? void 0 : (_appContextService$ge2 = _appContextService$ge.internal) === null || _appContextService$ge2 === void 0 ? void 0 : _appContextService$ge2.useMeteringApi;

    // Get matching data streams, their stats, and package SOs
    const [dataStreamsInfo, dataStreamStatsOrUndefined, dataStreamMeteringStatsorUndefined, packageSavedObjects] = await Promise.all([_data_streams.dataStreamService.getAllFleetDataStreams(esClient), useMeteringApi ? undefined : _data_streams.dataStreamService.getAllFleetDataStreamsStats(elasticsearch.client.asSecondaryAuthUser), useMeteringApi ? _data_streams.dataStreamService.getAllFleetMeteringStats(elasticsearch.client.asSecondaryAuthUser) : undefined, (0, _get.getPackageSavedObjects)(savedObjects.client)]);

    // managed_by property 'ingest-manager' added to allow for legacy data streams to be displayed
    // See https://github.com/elastic/elastic-agent/issues/654

    const filteredDataStreamsInfo = dataStreamsInfo.filter(ds => {
      var _ds$_meta, _ds$_meta2;
      return (ds === null || ds === void 0 ? void 0 : (_ds$_meta = ds._meta) === null || _ds$_meta === void 0 ? void 0 : _ds$_meta.managed_by) === MANAGED_BY || (ds === null || ds === void 0 ? void 0 : (_ds$_meta2 = ds._meta) === null || _ds$_meta2 === void 0 ? void 0 : _ds$_meta2.managed_by) === LEGACY_MANAGED_BY;
    });
    const dataStreamsInfoByName = (0, _lodash.keyBy)(filteredDataStreamsInfo, 'name');
    let dataStreamsStatsByName = {};
    if (dataStreamStatsOrUndefined) {
      const filteredDataStreamsStats = dataStreamStatsOrUndefined.filter(dss => !!dataStreamsInfoByName[dss.data_stream]);
      dataStreamsStatsByName = (0, _lodash.keyBy)(filteredDataStreamsStats, 'data_stream');
    }
    let dataStreamsMeteringStatsByName = {};
    if (dataStreamMeteringStatsorUndefined) {
      dataStreamsMeteringStatsByName = (0, _lodash.keyBy)(dataStreamMeteringStatsorUndefined, 'name');
    }

    // Combine data stream info
    const dataStreams = (0, _lodash.merge)(dataStreamsInfoByName, dataStreamsStatsByName, dataStreamsMeteringStatsByName);
    const dataStreamNames = (0, _lodash.keys)(dataStreams);

    // Map package SOs
    const packageSavedObjectsByName = (0, _lodash.keyBy)(packageSavedObjects.saved_objects, 'id');
    const packageMetadata = {};

    // Get dashboard information for all packages
    const dashboardIdsByPackageName = packageSavedObjects.saved_objects.reduce((allDashboards, pkgSavedObject) => {
      var _pkgSavedObject$attri;
      const dashboards = [];
      (((_pkgSavedObject$attri = pkgSavedObject.attributes) === null || _pkgSavedObject$attri === void 0 ? void 0 : _pkgSavedObject$attri.installed_kibana) || []).forEach(o => {
        if (o.type === _types.KibanaSavedObjectType.dashboard) {
          dashboards.push(o.id);
        }
      });
      allDashboards[pkgSavedObject.id] = dashboards;
      return allDashboards;
    }, {});
    const allDashboardSavedObjectsResponse = await savedObjects.client.bulkGet(Object.values(dashboardIdsByPackageName).flatMap(dashboardIds => dashboardIds.map(id => ({
      id,
      type: _types.KibanaSavedObjectType.dashboard,
      fields: ['title']
    }))));
    // Ignore dashboards not found
    const allDashboardSavedObjects = allDashboardSavedObjectsResponse.saved_objects.filter(so => {
      if (so.error) {
        if (so.error.statusCode === 404) {
          return false;
        }
        throw so.error;
      }
      return true;
    });
    const allDashboardSavedObjectsById = (0, _lodash.keyBy)(allDashboardSavedObjects, dashboardSavedObject => dashboardSavedObject.id);

    // Query additional information for each data stream
    const queryDataStreamInfo = async dataStreamName => {
      var _dataStream$_meta, _dataStream$_meta$pac;
      const dataStream = dataStreams[dataStreamName];
      const dataStreamResponse = {
        index: dataStreamName,
        dataset: '',
        namespace: '',
        type: '',
        package: ((_dataStream$_meta = dataStream._meta) === null || _dataStream$_meta === void 0 ? void 0 : (_dataStream$_meta$pac = _dataStream$_meta.package) === null || _dataStream$_meta$pac === void 0 ? void 0 : _dataStream$_meta$pac.name) || '',
        package_version: '',
        last_activity_ms: dataStream.maximum_timestamp,
        // overridden below if maxIngestedTimestamp agg returns a result
        size_in_bytes: dataStream.store_size_bytes || dataStream.size_in_bytes,
        // `store_size` should be available from ES due to ?human=true flag
        // but fallback to bytes just in case
        size_in_bytes_formatted: dataStream.store_size || new _configSchema.ByteSizeValue(dataStream.store_size_bytes || dataStream.size_in_bytes || 0).toString(),
        dashboards: [],
        serviceDetails: null
      };
      const {
        maxIngested,
        namespace,
        dataset,
        type,
        serviceNames,
        environments
      } = await (0, _get_data_streams_query_metadata.getDataStreamsQueryMetadata)({
        dataStreamName: dataStream.name,
        esClient
      });

      // some integrations e.g custom logs don't have event.ingested
      if (maxIngested) {
        dataStreamResponse.last_activity_ms = maxIngested;
      }
      if ((serviceNames === null || serviceNames === void 0 ? void 0 : serviceNames.length) === 1) {
        const serviceDetails = {
          serviceName: serviceNames[0],
          environment: (environments === null || environments === void 0 ? void 0 : environments.length) === 1 ? environments[0] : 'ENVIRONMENT_ALL'
        };
        dataStreamResponse.serviceDetails = serviceDetails;
      }
      dataStreamResponse.dataset = dataset;
      dataStreamResponse.namespace = namespace;
      dataStreamResponse.type = type;

      // Find package saved object
      const pkgName = dataStreamResponse.package;
      const pkgSavedObject = pkgName ? packageSavedObjectsByName[pkgName] : null;
      if (pkgSavedObject) {
        // if
        // - the data stream is associated with a package
        // - and the package has been installed through EPM
        // - and we didn't pick the metadata in an earlier iteration of this map()
        if (!packageMetadata[pkgName]) {
          var _pkgSavedObject$attri2;
          // then pick the dashboards from the package saved object
          const packageDashboardIds = dashboardIdsByPackageName[pkgName] || [];
          const packageDashboards = packageDashboardIds.reduce((dashboards, dashboardId) => {
            const dashboard = allDashboardSavedObjectsById[dashboardId];
            if (dashboard) {
              dashboards.push({
                id: dashboard.id,
                title: dashboard.attributes.title || dashboard.id
              });
            }
            return dashboards;
          }, []);
          packageMetadata[pkgName] = {
            version: ((_pkgSavedObject$attri2 = pkgSavedObject.attributes) === null || _pkgSavedObject$attri2 === void 0 ? void 0 : _pkgSavedObject$attri2.version) || '',
            dashboards: packageDashboards
          };
        }

        // Set values from package information
        dataStreamResponse.package = pkgName;
        dataStreamResponse.package_version = packageMetadata[pkgName].version;
        dataStreamResponse.dashboards = packageMetadata[pkgName].dashboards;
      }
      return dataStreamResponse;
    };

    // Return final data streams objects sorted by last activity, descending
    // After filtering out data streams that are missing dataset/namespace/type/package fields
    body.data_streams = (await (0, _pMap.default)(dataStreamNames, dataStreamName => queryDataStreamInfo(dataStreamName), {
      concurrency: _constants.MAX_CONCURRENT_DATASTREAM_OPERATIONS
    })).filter(({
      dataset,
      namespace,
      type
    }) => dataset && namespace && type).sort((a, b) => b.last_activity_ms - a.last_activity_ms);
    return response.ok({
      body
    });
  } catch (err) {
    var _err$body, _err$body$error;
    const isResponseError = err instanceof _elasticsearch.errors.ResponseError;
    if (isResponseError && (err === null || err === void 0 ? void 0 : (_err$body = err.body) === null || _err$body === void 0 ? void 0 : (_err$body$error = _err$body.error) === null || _err$body$error === void 0 ? void 0 : _err$body$error.type) === 'security_exception') {
      throw new _errors.FleetUnauthorizedError(`Not enough permissions to query datastreams: ${err.message}`);
    }
    throw err;
  }
};
exports.getListHandler = getListHandler;
const getDeprecatedILMCheckHandler = async (context, request, response) => {
  try {
    var _appContextService$ge3, _appContextService$ge4, _appContextService$ge5;
    const {
      elasticsearch
    } = await context.core;
    const esClient = elasticsearch.client.asCurrentUser;

    //  Before doing anything, check if the ILM policies are disabled and return early if they are
    const isILMPolicyDisabled = (_appContextService$ge3 = (_appContextService$ge4 = _services.appContextService.getConfig()) === null || _appContextService$ge4 === void 0 ? void 0 : (_appContextService$ge5 = _appContextService$ge4.internal) === null || _appContextService$ge5 === void 0 ? void 0 : _appContextService$ge5.disableILMPolicies) !== null && _appContextService$ge3 !== void 0 ? _appContextService$ge3 : false;
    if (isILMPolicyDisabled) {
      return response.ok({
        body: {
          deprecatedILMPolicies: []
        }
      });
    }
    const DEPRECATED_ILM_POLICY_TYPES = ['logs', 'metrics', 'synthetics'];

    // Fetch all ILM policies
    const ilmResponse = await esClient.ilm.getLifecycle();

    // Check each deprecated policy type to see if we should show a callout
    const deprecatedILMPolicies = [];
    for (const policyType of DEPRECATED_ILM_POLICY_TYPES) {
      const deprecatedPolicyName = policyType;
      const lifecyclePolicyName = `${policyType}@lifecycle`;
      const deprecatedPolicy = ilmResponse[deprecatedPolicyName];
      const lifecyclePolicy = ilmResponse[lifecyclePolicyName];

      // Skip if deprecated policy doesn't exist
      if (!deprecatedPolicy) {
        continue;
      }

      // Fetch Fleet-managed component templates for this policy type
      const componentTemplateResponse = await esClient.cluster.getComponentTemplate({
        name: `${policyType}-*@package`
      }, {
        ignore: [404]
      });

      // Filter component templates that actually use the deprecated policy
      const fleetManagedTemplates = (componentTemplateResponse.component_templates || []).filter(template => {
        var _template$component_t, _template$component_t2, _template$component_t3, _template$component_t4, _template$component_t5;
        const ilmPolicyName = (_template$component_t = template.component_template) === null || _template$component_t === void 0 ? void 0 : (_template$component_t2 = _template$component_t.template) === null || _template$component_t2 === void 0 ? void 0 : (_template$component_t3 = _template$component_t2.settings) === null || _template$component_t3 === void 0 ? void 0 : (_template$component_t4 = _template$component_t3.index) === null || _template$component_t4 === void 0 ? void 0 : (_template$component_t5 = _template$component_t4.lifecycle) === null || _template$component_t5 === void 0 ? void 0 : _template$component_t5.name;
        return ilmPolicyName === deprecatedPolicyName;
      }).map(template => template.name);

      // If no Fleet-managed component templates are using this deprecated policy, skip
      if (fleetManagedTemplates.length === 0) {
        continue;
      }

      // Case 1: Using deprecated policy but @lifecycle doesn't exist → show callout
      if (!lifecyclePolicy) {
        deprecatedILMPolicies.push({
          policyName: deprecatedPolicyName,
          version: deprecatedPolicy.version,
          componentTemplates: fleetManagedTemplates
        });
        continue;
      }

      // Case 2: Both policies exist
      // Don't show callout if both are unmodified (version 1) - auto-migration will happen
      if (deprecatedPolicy.version === 1 && lifecyclePolicy.version === 1) {
        // Both unmodified, auto-migration will handle this, skip
        continue;
      }

      // Case 3: At least one policy is modified (version > 1) → show callout
      if (deprecatedPolicy.version > 1 || lifecyclePolicy.version > 1) {
        deprecatedILMPolicies.push({
          policyName: deprecatedPolicyName,
          version: deprecatedPolicy.version,
          componentTemplates: fleetManagedTemplates
        });
      }
    }
    return response.ok({
      body: {
        deprecatedILMPolicies
      }
    });
  } catch (err) {
    var _err$body2, _err$body2$error;
    const isResponseError = err instanceof _elasticsearch.errors.ResponseError;
    if (isResponseError && (err === null || err === void 0 ? void 0 : (_err$body2 = err.body) === null || _err$body2 === void 0 ? void 0 : (_err$body2$error = _err$body2.error) === null || _err$body2$error === void 0 ? void 0 : _err$body2$error.type) === 'security_exception') {
      throw new _errors.FleetUnauthorizedError(`Not enough permissions to query ILM policies: ${err.message}`);
    }
    throw err;
  }
};
exports.getDeprecatedILMCheckHandler = getDeprecatedILMCheckHandler;