"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.appendWhereClauseToESQLQuery = appendWhereClauseToESQLQuery;
var _esqlAst = require("@kbn/esql-ast");
var _sanitaze_input = require("../sanitaze_input");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Creates filter expression for both single and multi-value cases
 * For single values, it creates standard comparison expressions
 * For multi-value arrays, it creates MATCH clauses combined with AND/NOT
 */
function createFilterExpression(field, value, operation, fieldType) {
  // Handle is not null / is null operations
  if (operation === 'is_not_null' || operation === 'is_null') {
    const fieldName = (0, _sanitaze_input.sanitazeESQLInput)(field);
    const operator = operation === 'is_not_null' ? 'is not null' : 'is null';
    return {
      expression: `${fieldName} ${operator}`
    };
  }
  // Handle multi-value arrays with MATCH operator
  if (Array.isArray(value)) {
    const fieldName = (0, _sanitaze_input.sanitazeESQLInput)(field);
    if (!fieldName) {
      return {
        expression: '',
        isMultiValue: true
      };
    }
    const matchClauses = value.map(val => {
      const escapedValue = typeof val === 'string' ? (0, _utils.escapeStringValue)(val) : val;
      return `MATCH(${fieldName}, ${escapedValue})`;
    });
    if (operation === '-') {
      return {
        expression: matchClauses.map(clause => `NOT ${clause}`).join(' AND '),
        isMultiValue: true
      };
    }
    return {
      expression: matchClauses.join(' AND '),
      isMultiValue: true
    };
  }

  // Handle single values with standard operators
  const {
    operator
  } = (0, _utils.getOperator)(operation);
  const filterValue = typeof value === 'string' ? (0, _utils.escapeStringValue)(value) : value;
  let fieldName = (0, _sanitaze_input.sanitazeESQLInput)(field);
  if (!fieldName) {
    return {
      expression: '',
      isMultiValue: false
    };
  }
  if (fieldType === undefined || !_utils.PARAM_TYPES_NO_NEED_IMPLICIT_STRING_CASTING.includes(fieldType)) {
    fieldName = `${fieldName}::string`;
  }
  return {
    expression: `${fieldName} ${operator} ${filterValue}`.trim(),
    isMultiValue: false
  };
}
/**
 * Handles existing filters in the WHERE clause by checking if the filter exists,
 * updating the operator if necessary, or appending a new filter.
 */
function handleExistingFilter(fullQuery, commandText, field, operation, value, filterExpression) {
  const matches = commandText.match(new RegExp(field + '(.*)' + String(value)));
  if (matches) {
    var _matches$;
    const {
      operator
    } = (0, _utils.getOperator)(operation);
    const existingOperator = (_matches$ = matches[1]) === null || _matches$ === void 0 ? void 0 : _matches$.trim().replace('`', '').toLowerCase();
    if (existingOperator === operator.trim()) {
      // not changing anything
      return fullQuery;
    }
    if (!(0, _utils.getSupportedOperators)().includes(existingOperator.trim())) {
      return (0, _utils.appendToESQLQuery)(fullQuery, `AND ${filterExpression}`);
    }
    const existingFilter = matches[0].trim();
    const newFilter = existingFilter.replace(existingOperator, operator);
    return fullQuery.replace(existingFilter, newFilter);
  }
  return (0, _utils.appendToESQLQuery)(fullQuery, `AND ${filterExpression}`);
}

/**
 * Handles existing multi-value filters in the WHERE clause by checking existing MATCH functions
 * and determining whether to keep existing filters, negate them, or append new ones.
 */
function handleExistingFilterForMultiValues(baseESQLQuery, lastWhereCommand, field, value, operation, filterExpression) {
  const existingMatchFunctionsList = _esqlAst.Walker.findAll(lastWhereCommand, node => node.type === 'function' && node.name === 'match');

  // Gather the values used in MATCH functions
  const existingValues = [];
  existingMatchFunctionsList.forEach(matchFunction => {
    const details = (0, _utils.extractMatchFunctionDetails)(matchFunction);
    if (details && details.columnName === field && Array.isArray(value) && value.includes(details.literalValue)) {
      existingValues.push(details.literalValue);
    }
  });

  // Check if all values in the array already exist as filters
  const allValuesExist = Array.isArray(value) && value.length > 0 && value.every(val => existingValues.includes(String(val)));
  if (allValuesExist) {
    if (operation === '+') {
      // All positive filters already exist, no changes needed
      return baseESQLQuery;
    } else if (operation === '-') {
      // All negative filters exist as positive - need to negate them
      let updatedQuery = baseESQLQuery;
      const matchesToNegate = [];
      existingMatchFunctionsList.forEach(matchFunction => {
        const details = (0, _utils.extractMatchFunctionDetails)(matchFunction);
        if (details && details.columnName === field && Array.isArray(value) && value.includes(details.literalValue)) {
          const fieldName = (0, _sanitaze_input.sanitazeESQLInput)(field);
          const escapedValue = (0, _utils.escapeStringValue)(details.literalValue);
          const matchString = `MATCH(${fieldName}, ${escapedValue})`;
          matchesToNegate.push(matchString);
        }
      });

      // Replace each MATCH function with NOT MATCH
      matchesToNegate.forEach(matchString => {
        updatedQuery = updatedQuery.replace(matchString, `NOT ${matchString}`);
      });
      return updatedQuery;
    }
  }
  return (0, _utils.appendToESQLQuery)(baseESQLQuery, `AND ${filterExpression}`);
}

/**
 * Appends a WHERE clause to an existing ES|QL query string.
 * @param baseESQLQuery the base ES|QL query to append the WHERE clause to.
 * @param field the field to filter on.
 * @param value the value to filter by.
 * @param operation the operation to perform ('+', '-', 'is_not_null', 'is_null').
 * @param fieldType the type of the field being filtered (optional).
 * @returns the modified ES|QL query string with the appended WHERE clause, or undefined if no changes were made.
 */
function appendWhereClauseToESQLQuery(baseESQLQuery, field, value, operation, fieldType) {
  const {
    root
  } = _esqlAst.Parser.parse(baseESQLQuery);
  const lastCommand = root.commands[root.commands.length - 1];
  const isLastCommandWhere = lastCommand.name === 'where';
  const {
    expression: filterExpression,
    isMultiValue
  } = createFilterExpression(field, value, operation, fieldType);
  if (!filterExpression) {
    return baseESQLQuery;
  }
  if (!isLastCommandWhere) {
    return (0, _utils.appendToESQLQuery)(baseESQLQuery, `| WHERE ${filterExpression}`);
  }

  // if where command already exists in the end of the query:
  // - we need to append with and if the filter doesn't exist
  // - we need to change the filter operator if the filter exists with different operator
  // - we do nothing if the filter exists with the same operator
  const whereAstText = lastCommand.text;
  const pipesArray = baseESQLQuery.split('|');
  const whereClause = pipesArray[pipesArray.length - 1];

  // Handles is not null / is null operations
  if (operation === 'is_not_null' || operation === 'is_null') {
    return handleExistingFilter(baseESQLQuery, whereClause, field, operation, '', filterExpression);
  }

  // Handles multi-value filters
  if (isMultiValue) {
    return handleExistingFilterForMultiValues(baseESQLQuery, lastCommand, field, value, operation, filterExpression);
  }
  // Handles single value filters
  const filterValue = typeof value === 'string' ? (0, _utils.escapeStringValue)(value) : value;
  const shouldCheckFilterValue = whereAstText.includes(String(filterValue));
  if (whereAstText.includes(field) && shouldCheckFilterValue) {
    return handleExistingFilter(baseESQLQuery, whereClause, field, operation, filterValue, filterExpression);
  }
  return (0, _utils.appendToESQLQuery)(baseESQLQuery, `AND ${filterExpression}`);
}