"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QueryColumns = exports.NOT_SUGGESTED_TYPES = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _ = require("../..");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const NOT_SUGGESTED_TYPES = exports.NOT_SUGGESTED_TYPES = ['unsupported'];

// Helper to check if a command is a source command that should be cached
function isSourceCommand(commandName) {
  return _.SOURCE_COMMANDS.has(commandName.toLowerCase());
}

/**
 * Responsible for efficiently computing the list of columns generated by a specific ESQL query.
 */
class QueryColumns {
  /**
   * Retrieves from cache the columns for a given query, ignoring case.
   * @param query
   * @returns
   */
  static fromCache(query) {
    const key = query.toLowerCase();
    const result = QueryColumns.cache.get(key);
    if (result) {
      // Move it to the end to mark it as recently used
      QueryColumns.cache.delete(key);
      QueryColumns.cache.set(key, result);
    }
    return result;
  }

  /**
   * Remove from the cache the least recently used when cache is full.
   */
  static setCache(key, value) {
    const normalizedKey = key.toLowerCase();
    if (QueryColumns.cache.has(normalizedKey)) {
      QueryColumns.cache.delete(normalizedKey);
    }

    // Remove oldest entry if cache is full
    while (QueryColumns.cache.size >= QueryColumns.MAX_CACHE_SIZE) {
      const oldestKey = QueryColumns.cache.keys().next().value;
      if (oldestKey) {
        QueryColumns.cache.delete(oldestKey);
      }
    }

    // Add new entry at the end
    QueryColumns.cache.set(normalizedKey, value);
  }
  constructor(query, originalQueryText, resourceRetriever, options) {
    (0, _defineProperty2.default)(this, "getFields", async queryToES => {
      var _this$options;
      if (!((_this$options = this.options) !== null && _this$options !== void 0 && _this$options.invalidateColumnsCache)) {
        const cached = QueryColumns.fromCache(queryToES);
        if (cached) {
          return cached;
        }
      }
      const fields = await (0, _helpers.getFieldsFromES)(queryToES, this.resourceRetriever);
      QueryColumns.setCache(queryToES, fields);
      return fields;
    });
    (0, _defineProperty2.default)(this, "getPolicies", async () => {
      var _await$this$resourceR, _this$resourceRetriev, _this$resourceRetriev2;
      const policies = (_await$this$resourceR = await ((_this$resourceRetriev = this.resourceRetriever) === null || _this$resourceRetriev === void 0 ? void 0 : (_this$resourceRetriev2 = _this$resourceRetriev.getPolicies) === null || _this$resourceRetriev2 === void 0 ? void 0 : _this$resourceRetriev2.call(_this$resourceRetriev))) !== null && _await$this$resourceR !== void 0 ? _await$this$resourceR : [];
      return new Map(policies.map(p => [p.name, p]));
    });
    this.query = query;
    this.originalQueryText = originalQueryText;
    this.resourceRetriever = resourceRetriever;
    this.options = options;
  }

  /**
   * Returns columns for this query, filtered by type and optionally ignoring some names.
   */
  async byType(expectedType = 'any', ignored = []) {
    const types = Array.isArray(expectedType) ? expectedType : [expectedType];
    const fields = await this.getColumnsForQuery(this.query, this.getFields, this.getPolicies);
    return (fields === null || fields === void 0 ? void 0 : fields.filter(({
      name,
      type
    }) => {
      const ts = Array.isArray(type) ? type : [type];
      return !ignored.includes(name) && (types[0] === 'any' || ts.some(t => types.includes(t))) && !NOT_SUGGESTED_TYPES.includes(type);
    })) || [];
  }

  /**
   * Returns a map of column name to column metadata for this query.
   */
  async asMap() {
    return this.getColumnsAsMap();
  }

  /**
   * Internal method to get columns as map.
   */
  async getColumnsAsMap() {
    const fields = await this.getColumnsForQuery(this.query, this.getFields, this.getPolicies);
    const result = new Map();
    for (let i = 0; i < fields.length; i++) {
      const field = fields[i];
      result.set(field.name, field);
    }
    return result;
  }
  /**
   * Get fields from previous commands building up the chain.
   */
  async getFieldsFromPreviousCommands(query, fetchFields, getPolicies) {
    if (query.commands.length <= 1) {
      return [];
    }
    let fields = [];
    for (let i = 0; i < query.commands.length - 1; i++) {
      const subQuery = {
        ...query,
        commands: query.commands.slice(0, i + 1)
      };
      const currentCommand = query.commands[i];
      if (!(currentCommand !== null && currentCommand !== void 0 && currentCommand.name)) {
        continue;
      }
      const isSource = isSourceCommand(currentCommand.name);
      if (isSource) {
        // For source commands, check cache first
        const queryString = _.BasicPrettyPrinter.print(subQuery);
        const cachedFields = QueryColumns.fromCache(queryString);
        if (cachedFields) {
          fields = cachedFields;
          continue;
        }
      }

      // Build fields using the command's columnsAfter method
      fields = await (0, _helpers.getCurrentQueryAvailableColumns)(subQuery.commands, fields, fetchFields, getPolicies, this.originalQueryText);
    }
    return fields;
  }

  /**
   * Returns the fields for a given query.
   * @param query
   * @param fetchFields
   * @param getPolicies
   */
  async getColumnsForQuery(query, fetchFields, getPolicies) {
    const fieldsAvailableAfterPreviousCommand = await this.getFieldsFromPreviousCommands(query, fetchFields, getPolicies);
    const availableFields = await (0, _helpers.getCurrentQueryAvailableColumns)(query.commands, fieldsAvailableAfterPreviousCommand, fetchFields, getPolicies, this.originalQueryText);
    return availableFields;
  }
}
exports.QueryColumns = QueryColumns;
(0, _defineProperty2.default)(QueryColumns, "cache", new Map());
// Adding a max size to the cache to prevent unbounded memory growth
(0, _defineProperty2.default)(QueryColumns, "MAX_CACHE_SIZE", 100);