"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PromQLCstToAstConverter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var cst = _interopRequireWildcard(require("../../parser/antlr/promql_parser"));
var _tokens = require("../../parser/core/tokens");
var _builder = require("../builder");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// TODO: Handle `SubqueryResolutionContext`, `NonReservedContext`

/**
 * Transforms an ANTLR PromQL Concrete Syntax Tree (CST) into a
 * Kibana Abstract Syntax Tree (AST).
 */
class PromQLCstToAstConverter {
  constructor(parser) {
    var _parser$options$offse;
    /**
     * Character offset to add to all location values.
     */
    (0, _defineProperty2.default)(this, "offset", void 0);
    this.parser = parser;
    this.offset = (_parser$options$offse = parser.options.offset) !== null && _parser$options$offse !== void 0 ? _parser$options$offse : 0;
  }

  // -------------------------------------------------------------------- utils

  getParserFields(ctx) {
    const location = (0, _tokens.getPosition)(ctx.start, ctx.stop);
    return {
      text: ctx.getText(),
      location: {
        min: location.min + this.offset,
        max: location.max + this.offset
      },
      incomplete: Boolean(ctx.exception)
    };
  }
  createParserFieldsFromToken(token, text = token.text) {
    const location = (0, _tokens.getPosition)(token, token);
    return {
      text,
      location: {
        min: location.min + this.offset,
        max: location.max + this.offset
      },
      incomplete: false
    };
  }
  fromParserRuleToUnknown(ctx) {
    const location = (0, _tokens.getPosition)(ctx.start, ctx.stop);
    return _builder.PromQLBuilder.unknown({
      text: ctx.getText(),
      location: {
        min: location.min + this.offset,
        max: location.max + this.offset
      },
      incomplete: Boolean(ctx.exception)
    });
  }

  // -------------------------------------------------------------------- query

  fromSingleStatement(ctx) {
    if (!ctx) return undefined;
    const exprCtx = ctx.expression();
    const expression = exprCtx ? this.fromExpression(exprCtx) : undefined;
    return _builder.PromQLBuilder.expression.query(expression, this.getParserFields(ctx));
  }

  // --------------------------------------------------------------- expression

  fromExpression(ctx) {
    if (!ctx) return undefined;

    // ArithmeticUnary: unary +/- expression
    if (ctx instanceof cst.ArithmeticUnaryContext) {
      return this.fromArithmeticUnary(ctx);
    }

    // ValueExpression: function, selector, or constant
    if (ctx instanceof cst.ValueExpressionContext) {
      return this.fromValueExpression(ctx);
    }

    // Parenthesized: (expression)
    if (ctx instanceof cst.ParenthesizedContext) {
      return this.fromParenthesized(ctx);
    }

    // ArithmeticBinary: binary expressions
    if (ctx instanceof cst.ArithmeticBinaryContext) {
      return this.fromArithmeticBinary(ctx);
    }

    // Subquery: expression[range:resolution]
    if (ctx instanceof cst.SubqueryContext) {
      return this.fromSubquery(ctx);
    }

    // Fallback: return unknown
    return this.fromParserRuleToUnknown(ctx);
  }

  // -------------------------------------------------------------------- unary

  fromArithmeticUnary(ctx) {
    const operatorToken = ctx._operator;
    const operator = operatorToken === null || operatorToken === void 0 ? void 0 : operatorToken.text;
    const argCtx = ctx.expression();
    const arg = argCtx ? this.fromExpression(argCtx) : undefined;
    if (!arg) {
      return this.fromParserRuleToUnknown(ctx);
    }
    return _builder.PromQLBuilder.expression.unary(operator, arg, this.getParserFields(ctx));
  }

  // -------------------------------------------------------------------- value

  fromValueExpression(ctx) {
    const valueCtx = ctx.value();
    if (!valueCtx) return undefined;
    return this.fromValue(valueCtx);
  }
  fromValue(ctx) {
    const funcCtx = ctx.function_();
    if (funcCtx) {
      return this.fromFunction(funcCtx);
    }
    const selectorCtx = ctx.selector();
    if (selectorCtx) {
      return this.fromSelector(selectorCtx);
    }
    const constantCtx = ctx.constant();
    if (constantCtx) {
      return this.fromConstant(constantCtx);
    }
    return undefined;
  }

  // ----------------------------------------------------------------- function

  fromFunction(ctx) {
    var _identToken$getText;
    const identToken = ctx.IDENTIFIER();
    const name = (_identToken$getText = identToken === null || identToken === void 0 ? void 0 : identToken.getText()) !== null && _identToken$getText !== void 0 ? _identToken$getText : '';
    const paramsCtx = ctx.functionParams();
    const args = [];
    if (paramsCtx) {
      const exprCtxs = paramsCtx.expression_list();
      for (const exprCtx of exprCtxs) {
        const expr = this.fromExpression(exprCtx);
        if (expr) {
          args.push(expr);
        }
      }
    }
    const groupingCtx = ctx.grouping();
    const grouping = groupingCtx ? this.fromGrouping(groupingCtx) : undefined;

    // Determine grouping position by comparing token positions
    // If grouping exists and its start position is before the LP token, it's 'before'
    let groupingPosition;
    if (grouping && groupingCtx) {
      const lpToken = ctx.LP();
      if (lpToken) {
        var _groupingCtx$start$st, _groupingCtx$start;
        const groupingStart = (_groupingCtx$start$st = (_groupingCtx$start = groupingCtx.start) === null || _groupingCtx$start === void 0 ? void 0 : _groupingCtx$start.start) !== null && _groupingCtx$start$st !== void 0 ? _groupingCtx$start$st : 0;
        const lpStart = lpToken.symbol.start;
        groupingPosition = groupingStart < lpStart ? 'before' : 'after';
      }
    }
    return _builder.PromQLBuilder.expression.func.call(name, args, grouping, groupingPosition, this.getParserFields(ctx));
  }
  fromGrouping(ctx) {
    const byToken = ctx.BY();
    const kind = byToken ? 'by' : 'without';
    const labelListCtx = ctx.labelList();
    const labels = this.fromLabelList(labelListCtx);
    return _builder.PromQLBuilder.grouping(kind, labels, this.getParserFields(ctx));
  }
  fromLabelList(ctx) {
    const labels = [];
    const labelNameCtxs = ctx.labelName_list();
    for (const labelNameCtx of labelNameCtxs) {
      const label = this.fromLabelName(labelNameCtx);
      if (label) {
        labels.push(label);
      }
    }
    return labels;
  }

  // ----------------------------------------------------------------- selector

  fromSelector(ctx) {
    const seriesMatcherCtx = ctx.seriesMatcher();
    const {
      metric,
      labelMap
    } = this.fromSeriesMatcher(seriesMatcherCtx);

    // Range: [duration]
    const lsb = ctx.LSB();
    const durationCtx = ctx.duration();
    let range;
    if (lsb && durationCtx) {
      range = this.fromDuration(durationCtx);
    }

    // Evaluation: offset and @
    const evaluationCtx = ctx.evaluation();
    const evaluation = evaluationCtx ? this.fromEvaluation(evaluationCtx) : undefined;
    return _builder.PromQLBuilder.expression.selector.node({
      metric,
      labelMap,
      duration: range,
      evaluation
    }, this.getParserFields(ctx));
  }
  fromSeriesMatcher(ctx) {
    const identCtx = ctx.identifier();
    const metric = identCtx ? this.fromIdentifier(identCtx) : undefined;
    const labelsCtx = ctx.labels();
    const labelMap = labelsCtx ? this.fromLabels(labelsCtx) : undefined;
    return {
      metric,
      labelMap
    };
  }
  fromLabels(ctx) {
    const labels = [];
    const labelCtxs = ctx.label_list();
    for (const labelCtx of labelCtxs) {
      const label = this.fromLabel(labelCtx);
      if (label) {
        labels.push(label);
      }
    }
    return _builder.PromQLBuilder.labelMap(labels, this.getParserFields(ctx));
  }
  fromLabel(ctx) {
    const labelNameCtx = ctx.labelName();
    const labelName = labelNameCtx ? this.fromLabelName(labelNameCtx) : undefined;
    if (!labelName) return undefined;
    const kindToken = ctx._kind;
    let operator = '=';
    if (kindToken) {
      const kindText = kindToken.text;
      if (kindText === '=' || kindText === '!=' || kindText === '=~' || kindText === '!~') {
        operator = kindText;
      }
    }
    const stringToken = ctx.STRING();
    let value;
    if (stringToken) {
      value = this.fromStringToken(stringToken.symbol);
    }
    return _builder.PromQLBuilder.label(labelName, operator, value, this.getParserFields(ctx));
  }
  fromLabelName(ctx) {
    const identCtx = ctx.identifier();
    if (identCtx) {
      return this.fromIdentifier(identCtx);
    }
    const stringToken = ctx.STRING();
    if (stringToken) {
      return this.fromStringToken(stringToken.symbol);
    }
    const numberCtx = ctx.number_();
    if (numberCtx) {
      return this.fromNumber(numberCtx);
    }
    return undefined;
  }

  // ------------------------------------------------------------------- binary

  fromArithmeticBinary(ctx) {
    var _opToken$text;
    const leftCtx = ctx._left;
    const rightCtx = ctx._right;
    const opToken = ctx._op;
    const left = leftCtx ? this.fromExpression(leftCtx) : undefined;
    const right = rightCtx ? this.fromExpression(rightCtx) : undefined;
    if (!left || !right) {
      return this.fromParserRuleToUnknown(ctx);
    }
    const operatorText = (_opToken$text = opToken === null || opToken === void 0 ? void 0 : opToken.text) !== null && _opToken$text !== void 0 ? _opToken$text : '';
    const operator = this.toBinaryOperator(operatorText);

    // Check for BOOL modifier
    const boolToken = ctx.BOOL();
    const bool = !!boolToken;

    // Check for modifier (ON/IGNORING)
    const modifierCtx = ctx.modifier();
    const modifier = modifierCtx ? this.fromModifier(modifierCtx) : undefined;
    return _builder.PromQLBuilder.expression.binary(operator, left, right, {
      bool,
      modifier
    }, this.getParserFields(ctx));
  }
  toBinaryOperator(text) {
    const lowerText = text.toLowerCase();
    return lowerText;
  }
  fromModifier(ctx) {
    var _matchingToken$text;
    const matchingToken = ctx._matching;
    const kind = (matchingToken === null || matchingToken === void 0 ? void 0 : (_matchingToken$text = matchingToken.text) === null || _matchingToken$text === void 0 ? void 0 : _matchingToken$text.toLowerCase()) === 'on' ? 'on' : 'ignoring';
    const labelListCtx = ctx._modifierLabels;
    const labels = labelListCtx ? this.fromLabelList(labelListCtx) : [];

    // Group modifier (group_left/group_right)
    const joiningToken = ctx._joining;
    let groupModifier;
    if (joiningToken) {
      var _joiningToken$text, _joiningToken$text2;
      const joiningKind = ((_joiningToken$text = joiningToken.text) === null || _joiningToken$text === void 0 ? void 0 : _joiningToken$text.toLowerCase()) === 'group_left' ? 'group_left' : 'group_right';
      const groupLabelsCtx = ctx._groupLabels;
      const groupLabels = groupLabelsCtx ? this.fromLabelList(groupLabelsCtx) : [];
      groupModifier = _builder.PromQLBuilder.groupModifier(joiningKind, groupLabels, {
        text: (_joiningToken$text2 = joiningToken.text) !== null && _joiningToken$text2 !== void 0 ? _joiningToken$text2 : '',
        location: (0, _tokens.getPosition)(joiningToken, joiningToken),
        incomplete: false
      });
    }
    return _builder.PromQLBuilder.modifier(kind, labels, groupModifier, this.getParserFields(ctx));
  }

  // ------------------------------------------------------------ parenthesized

  fromParenthesized(ctx) {
    const exprCtx = ctx.expression();
    const child = exprCtx ? this.fromExpression(exprCtx) : undefined;
    if (!child) {
      return this.fromParserRuleToUnknown(ctx);
    }
    return _builder.PromQLBuilder.expression.parens(child, this.getParserFields(ctx));
  }

  // ----------------------------------------------------------------- subquery

  fromSubquery(ctx) {
    const exprCtx = ctx.expression();
    const expr = exprCtx ? this.fromExpression(exprCtx) : undefined;
    if (!expr) {
      return this.fromParserRuleToUnknown(ctx);
    }
    const rangeCtx = ctx._range;
    const range = rangeCtx ? this.fromDuration(rangeCtx) : undefined;
    if (!range) {
      return this.fromParserRuleToUnknown(ctx);
    }

    // Resolution from subqueryResolution
    const resolutionCtx = ctx.subqueryResolution();
    let resolution;
    if (resolutionCtx) {
      // Check for duration first (COLON duration?)
      const resDurationCtx = resolutionCtx._resolution;
      if (resDurationCtx) {
        resolution = this.fromDuration(resDurationCtx);
      } else {
        // Check for TIME_VALUE_WITH_COLON (e.g., ":5m")
        const timeValueWithColon = resolutionCtx.TIME_VALUE_WITH_COLON();
        if (timeValueWithColon) {
          // TIME_VALUE_WITH_COLON includes the leading colon, e.g., ":5m"
          // We need to strip the colon to get the time value
          const text = timeValueWithColon.getText();
          const timeValue = text.startsWith(':') ? text.slice(1) : text;
          resolution = _builder.PromQLBuilder.expression.literal.time(timeValue, this.createParserFieldsFromToken(timeValueWithColon.symbol, timeValue));
        }
      }
    }
    const evaluationCtx = ctx.evaluation();
    const evaluation = evaluationCtx ? this.fromEvaluation(evaluationCtx) : undefined;
    return _builder.PromQLBuilder.expression.subquery(expr, range, resolution, evaluation, this.getParserFields(ctx));
  }

  // --------------------------------------------------------------- evaluation

  fromEvaluation(ctx) {
    const offsetCtx = ctx.offset();
    const atCtx = ctx.at();
    const offset = offsetCtx ? this.fromOffset(offsetCtx) : undefined;
    const at = atCtx ? this.fromAt(atCtx) : undefined;
    return _builder.PromQLBuilder.evaluation(offset, at, this.getParserFields(ctx));
  }
  fromOffset(ctx) {
    const minusToken = ctx.MINUS();
    const negative = !!minusToken;
    const durationCtx = ctx.duration();
    const duration = durationCtx ? this.fromDuration(durationCtx) : _builder.PromQLBuilder.unknown({
      incomplete: true
    });
    return _builder.PromQLBuilder.offset(duration, negative, this.getParserFields(ctx));
  }
  fromAt(ctx) {
    const minusToken = ctx.MINUS();
    const negative = !!minusToken;
    const atStartToken = ctx.AT_START();
    const atEndToken = ctx.AT_END();
    if (atStartToken) {
      return _builder.PromQLBuilder.at('start()', negative, this.getParserFields(ctx));
    }
    if (atEndToken) {
      return _builder.PromQLBuilder.at('end()', negative, this.getParserFields(ctx));
    }
    const timeValueCtx = ctx.timeValue();
    const timeValue = timeValueCtx ? this.fromTimeValue(timeValueCtx) : _builder.PromQLBuilder.expression.literal.time('', {
      incomplete: true
    });
    return _builder.PromQLBuilder.at(timeValue, negative, this.getParserFields(ctx));
  }

  // ----------------------------------------------------------------- duration

  fromDuration(ctx) {
    // Duration is now defined as `expression` in the grammar
    const exprCtx = ctx.expression();
    if (exprCtx) {
      var _this$fromExpression;
      return (_this$fromExpression = this.fromExpression(exprCtx)) !== null && _this$fromExpression !== void 0 ? _this$fromExpression : this.fromParserRuleToUnknown(ctx);
    }
    return this.fromParserRuleToUnknown(ctx);
  }

  // ----------------------------------------------------------------- constant

  fromConstant(ctx) {
    const numberCtx = ctx.number_();
    if (numberCtx) {
      return this.fromNumber(numberCtx);
    }
    const stringCtx = ctx.string_();
    if (stringCtx) {
      return this.fromString(stringCtx);
    }
    const timeValueCtx = ctx.timeValue();
    if (timeValueCtx) {
      return this.fromTimeValue(timeValueCtx);
    }
    return this.fromParserRuleToUnknown(ctx);
  }

  // ------------------------------------------------------------------- number

  fromNumber(ctx) {
    const text = ctx.getText();
    const parserFields = this.getParserFields(ctx);

    // Integer
    if (ctx instanceof cst.IntegerLiteralContext) {
      const value = parseInt(text, 10);
      return _builder.PromQLBuilder.expression.literal.integer(value, parserFields);
    }

    // Decimal
    if (ctx instanceof cst.DecimalLiteralContext) {
      const lowerText = text.toLowerCase();
      let value;
      if (lowerText === 'inf') {
        value = Infinity;
      } else if (lowerText === 'nan') {
        value = NaN;
      } else {
        value = parseFloat(text);
      }
      return _builder.PromQLBuilder.expression.literal.decimal(value, parserFields);
    }

    // Hexadecimal
    if (ctx instanceof cst.HexLiteralContext) {
      const value = parseInt(text, 16);
      return _builder.PromQLBuilder.expression.literal.hexadecimal(value, text, parserFields);
    }

    // Fallback
    return _builder.PromQLBuilder.expression.literal.integer(0, {
      ...parserFields,
      incomplete: true
    });
  }

  // ------------------------------------------------------------------- string

  fromString(ctx) {
    const token = ctx.STRING();
    if (token) {
      return this.fromStringToken(token.symbol);
    }
    return _builder.PromQLBuilder.expression.literal.string(ctx.getText(), '', this.getParserFields(ctx));
  }
  fromStringToken(token) {
    var _token$text;
    const text = (_token$text = token.text) !== null && _token$text !== void 0 ? _token$text : '';
    const valueUnquoted = this.unquoteString(text);
    return _builder.PromQLBuilder.expression.literal.string(text, valueUnquoted, this.createParserFieldsFromToken(token));
  }
  unquoteString(text) {
    if (text.length < 2) return text;
    const firstChar = text[0];
    const lastChar = text[text.length - 1];

    // Single or double quoted
    if ((firstChar === '"' || firstChar === "'") && firstChar === lastChar) {
      return text.slice(1, -1).replace(/\\([abfnrtv\\'""])/g, (_, char) => {
        switch (char) {
          case 'a':
            return '\x07';
          case 'b':
            return '\b';
          case 'f':
            return '\f';
          case 'n':
            return '\n';
          case 'r':
            return '\r';
          case 't':
            return '\t';
          case 'v':
            return '\v';
          default:
            return char;
        }
      });
    }

    // Backtick quoted (raw string)
    if (firstChar === '`' && lastChar === '`') {
      return text.slice(1, -1);
    }
    return text;
  }

  // --------------------------------------------------------------- time value

  fromTimeValue(ctx) {
    const text = ctx.getText();
    const parserFields = this.getParserFields(ctx);
    const timeWithColonToken = ctx.TIME_VALUE_WITH_COLON();
    if (timeWithColonToken) {
      return _builder.PromQLBuilder.expression.literal.time(text, parserFields);
    }
    const timeToken = ctx.TIME_VALUE();
    if (timeToken) {
      return _builder.PromQLBuilder.expression.literal.time(text, parserFields);
    }

    // Number as time value
    const numberCtx = ctx.number_();
    if (numberCtx) {
      return _builder.PromQLBuilder.expression.literal.time(text, parserFields);
    }
    return _builder.PromQLBuilder.expression.literal.time(text, {
      ...parserFields,
      incomplete: true
    });
  }

  // --------------------------------------------------------------- identifier

  fromIdentifier(ctx) {
    const identToken = ctx.IDENTIFIER();
    if (identToken) {
      return _builder.PromQLBuilder.identifier(identToken.getText(), this.createParserFieldsFromToken(identToken.symbol));
    }

    // Non-reserved keywords can be used as identifiers
    const nonReservedCtx = ctx.nonReserved();
    if (nonReservedCtx) {
      return _builder.PromQLBuilder.identifier(nonReservedCtx.getText(), this.getParserFields(ctx));
    }
    return _builder.PromQLBuilder.identifier(ctx.getText(), this.getParserFields(ctx));
  }
}
exports.PromQLCstToAstConverter = PromQLCstToAstConverter;