"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WrappingPrettyPrinter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _grouping = require("../ast/grouping");
var _is = require("../ast/is");
var _visitor = require("../ast/visitor");
var _utils = require("../ast/visitor/utils");
var _basic_pretty_printer = require("./basic_pretty_printer");
var _constants = require("./constants");
var _helpers = require("./helpers");
var _leaf_printer = require("./leaf_printer");
var _WrappingPrettyPrinter;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
class WrappingPrettyPrinter {
  constructor(_opts = {}) {
    var _opts$indent, _opts$tab, _opts$pipeTab2, _opts$skipHeader, _opts$commandTab, _opts$multiline2, _opts$wrap, _opts$lowercase, _ref2, _opts$lowercaseComman, _ref3, _opts$lowercaseOption, _ref4, _opts$lowercaseFuncti, _ref5, _opts$lowercaseKeywor;
    (0, _defineProperty2.default)(this, "opts", void 0);
    (0, _defineProperty2.default)(this, "visitor", new _visitor.Visitor().on('visitExpression', (ctx, inp) => {
      var _ctx$node$text;
      const txt = (_ctx$node$text = ctx.node.text) !== null && _ctx$node$text !== void 0 ? _ctx$node$text : '<EXPRESSION>';
      // TODO: decorate with comments
      return {
        txt
      };
    }).on('visitHeaderCommand', (ctx, inp) => {
      const opts = this.opts;
      const cmd = opts.lowercaseCommands ? ctx.node.name : ctx.node.name.toUpperCase();
      let args = '';
      for (const arg of ctx.visitArgs(inp)) {
        args += (args ? ', ' : '') + arg.txt;
      }
      const argsFormatted = args ? ` ${args}` : '';
      const formatted = `${cmd}${argsFormatted};`;
      const formatting = ctx.node.formatting;
      let txt = formatted;
      let indented = false;
      if (formatting) {
        if (formatting.left) {
          const comments = _leaf_printer.LeafPrinter.commentList(formatting.left);
          if (comments) {
            indented = true;
            txt = `${inp.indent}${comments} ${txt}`;
          }
        }
        if (formatting.top) {
          const top = formatting.top;
          const length = top.length;
          for (let i = length - 1; i >= 0; i--) {
            const decoration = top[i];
            if (decoration.type === 'comment') {
              if (!indented) {
                txt = inp.indent + txt;
                indented = true;
              }
              txt = inp.indent + _leaf_printer.LeafPrinter.comment(decoration) + '\n' + txt;
              indented = true;
            }
          }
        }
        if (formatting.right) {
          const comments = _leaf_printer.LeafPrinter.commentList(formatting.right);
          if (comments) {
            txt = `${txt} ${comments}`;
          }
        }

        // For header commands, rightSingleLine comments should appear on the same line
        // but we need to ensure there's a newline after the comment for the next statement
        if (formatting.rightSingleLine) {
          const comment = _leaf_printer.LeafPrinter.comment(formatting.rightSingleLine);
          txt = `${txt} ${comment}`;
        }
        if (formatting.bottom) {
          for (const decoration of formatting.bottom) {
            if (decoration.type === 'comment') {
              indented = true;
              txt = txt + '\n' + inp.indent + _leaf_printer.LeafPrinter.comment(decoration);
            }
          }
        }
      }
      return {
        txt,
        indented
      };
    }).on('visitIdentifierExpression', (ctx, inp) => {
      const formatted = _leaf_printer.LeafPrinter.identifier(ctx.node);
      const {
        txt,
        indented
      } = this.decorateWithComments(inp, ctx.node, formatted);
      return {
        txt,
        indented
      };
    }).on('visitSourceExpression', (ctx, inp) => {
      const formatted = _leaf_printer.LeafPrinter.source(ctx.node);
      const {
        txt,
        indented
      } = this.decorateWithComments(inp, ctx.node, formatted);
      return {
        txt,
        indented
      };
    }).on('visitColumnExpression', (ctx, inp) => {
      const formatted = _leaf_printer.LeafPrinter.column(ctx.node);
      const {
        txt,
        indented
      } = this.decorateWithComments(inp, ctx.node, formatted);
      return {
        txt,
        indented
      };
    }).on('visitLiteralExpression', (ctx, inp) => {
      const formatted = _leaf_printer.LeafPrinter.literal(ctx.node);
      const {
        txt,
        indented
      } = this.decorateWithComments(inp, ctx.node, formatted);
      return {
        txt,
        indented
      };
    }).on('visitInlineCastExpression', (ctx, inp) => {
      const value = ctx.value();
      const wrapInBrackets = value.type !== 'literal' && value.type !== 'column' && !(value.type === 'function' && value.subtype === 'variadic-call');
      const castType = ctx.node.castType;
      const valueResult = ctx.visitValue({
        indent: inp.indent,
        remaining: inp.remaining - castType.length - 2
      });
      let {
        txt: valueFormatted
      } = valueResult;
      if (wrapInBrackets) {
        valueFormatted = `(${valueFormatted})`;
      }
      const formatted = `${valueFormatted}::${ctx.node.castType}`;
      const {
        txt,
        indented
      } = this.decorateWithComments(inp, ctx.node, formatted, valueResult.indented);
      return {
        txt,
        indented
      };
    }).on('visitListLiteralExpression', (ctx, inp) => {
      const args = this.printChildrenList(ctx, {
        indent: inp.indent,
        remaining: inp.remaining - 1
      });
      const node = ctx.node;
      const isTuple = node.subtype === 'tuple';
      const leftParenthesis = isTuple ? '(' : '[';
      const rightParenthesis = isTuple ? ')' : ']';
      const rightParenthesisIndent = args.oneArgumentPerLine ? '\n' + inp.indent : '';
      const formatted = leftParenthesis + args.txt + rightParenthesisIndent + rightParenthesis;
      const {
        txt,
        indented
      } = this.decorateWithComments(inp, ctx.node, formatted);
      return {
        txt,
        indented
      };
    }).on('visitMapEntryExpression', (ctx, inp) => {
      const operator = this.keyword(':');
      const expression = this.printBinaryOperatorExpression(ctx, operator, inp, '');
      return this.decorateWithComments({
        ...inp,
        suffix: ''
      }, ctx.node, expression.txt, expression.indented);
    }).on('visitMapExpression', (ctx, inp) => {
      const {
        txt,
        oneArgumentPerLine
      } = this.printChildrenList(ctx, inp);
      let formatted = txt;
      if (oneArgumentPerLine) {
        formatted = '{' + txt + '\n' + inp.indent + '}';
      } else {
        formatted = '{' + txt + '}';
      }
      return this.decorateWithComments(inp, ctx.node, formatted);
    }).on('visitParensExpression', (ctx, inp) => {
      var _ctx$parent, _ctx$node$child;
      // Check if parent is FORK command
      const parent = (_ctx$parent = ctx.parent) === null || _ctx$parent === void 0 ? void 0 : _ctx$parent.node;
      const isForkBranch = !Array.isArray(parent) && (parent === null || parent === void 0 ? void 0 : parent.type) === 'command' && parent.name === 'fork' && ((_ctx$node$child = ctx.node.child) === null || _ctx$node$child === void 0 ? void 0 : _ctx$node$child.type) === 'query';
      let formatted;
      if (isForkBranch) {
        const baseIndent = inp.indent + this.opts.tab;
        const childText = this.visitor.visitQuery(ctx.node.child, {
          indent: baseIndent,
          remaining: this.opts.wrap - baseIndent.length
        });
        const lines = childText.txt.split('\n');
        for (let i = 0; i < lines.length; i++) {
          if (i === 0) {
            lines[i] = '  ' + lines[i];
          } else if (lines[i].startsWith('  ')) {
            lines[i] = lines[i].slice(2);
          }
        }
        formatted = `(\n${lines.join('\n')}\n${inp.indent})`;
      } else {
        const child = ctx.visitChild(inp);
        formatted = `(${child.txt.trimStart()})`;
      }
      return this.decorateWithComments(inp, ctx.node, formatted);
    }).on('visitFunctionCallExpression', (ctx, inp) => {
      const node = ctx.node;
      let operator = ctx.operator();
      let txt = '';

      // Check if function name is a parameter stored in node.operator
      if (ctx.node.operator && (0, _is.isParamLiteral)(ctx.node.operator)) {
        operator = _leaf_printer.LeafPrinter.param(ctx.node.operator);
      } else {
        var _this$opts$lowercaseF;
        if (ctx.node.operator && (0, _is.isIdentifier)(ctx.node.operator)) {
          operator = ctx.node.operator.name;
        }
        operator = ((_this$opts$lowercaseF = this.opts.lowercaseFunctions) !== null && _this$opts$lowercaseF !== void 0 ? _this$opts$lowercaseF : this.opts.lowercase) ? operator.toLowerCase() : operator.toUpperCase();
      }
      switch (node.subtype) {
        case 'unary-expression':
          {
            operator = this.keyword(operator);
            const separator = operator === '-' || operator === '+' ? '' : ' ';
            const argument = ctx.arguments()[0];
            const argumentFormatted = ctx.visitArgument(0, inp);
            const operatorPrecedence = (0, _grouping.unaryExpressionGroup)(ctx.node);
            const argumentPrecedence = (0, _grouping.binaryExpressionGroup)(argument);
            if (argumentPrecedence !== _grouping.BinaryExpressionGroup.none && argumentPrecedence < operatorPrecedence) {
              argumentFormatted.txt = `(${argumentFormatted.txt})`;
            }
            txt = `${operator}${separator}${argumentFormatted.txt}`;
            break;
          }
        case 'postfix-unary-expression':
          {
            var _inp$suffix;
            operator = this.keyword(operator);
            const suffix = (_inp$suffix = inp.suffix) !== null && _inp$suffix !== void 0 ? _inp$suffix : '';
            txt = `${ctx.visitArgument(0, {
              ...inp,
              suffix: ''
            }).txt} ${operator}${suffix}`;
            break;
          }
        case 'binary-expression':
          {
            operator = this.keyword(operator);
            return this.printBinaryOperatorExpression(ctx, operator, inp);
          }
        default:
          {
            var _inp$suffix2;
            const args = this.printChildrenList(ctx, {
              indent: inp.indent,
              remaining: inp.remaining - operator.length - 1
            });
            let breakClosingParenthesis = false;
            if ((0, _helpers.getPrettyPrintStats)(ctx.node.args).hasRightSingleLineComments) {
              breakClosingParenthesis = true;
            }
            let closingParenthesisFormatted = ')';
            if (breakClosingParenthesis) {
              closingParenthesisFormatted = '\n' + inp.indent + ')';
            }
            txt = `${operator}(${args.txt}${closingParenthesisFormatted}${(_inp$suffix2 = inp.suffix) !== null && _inp$suffix2 !== void 0 ? _inp$suffix2 : ''}`;
          }
      }
      return this.decorateWithComments({
        ...inp,
        suffix: ''
      }, ctx.node, txt);
    }).on('visitCommandOption', (ctx, inp) => {
      const option = this.opts.lowercaseOptions ? ctx.node.name : ctx.node.name.toUpperCase();
      const args = this.printChildrenList(ctx, {
        indent: inp.indent,
        remaining: inp.remaining - option.length - 1
      });
      const argsFormatted = args.txt ? `${args.txt[0] === '\n' ? '' : ' '}${args.txt}` : '';
      const separator = _constants.commandOptionsWithEqualsSeparator.has(ctx.node.name) ? ' =' : '';
      const txt = `${option}${separator}${argsFormatted}`;

      // TODO: decorate with comments

      return {
        txt,
        lines: args.lines
      };
    }).on('visitCommand', (ctx, inp) => {
      const opts = this.opts;
      const node = ctx.node;
      let cmd = opts.lowercaseCommands ? node.name : node.name.toUpperCase();
      if (node.commandType) {
        const type = opts.lowercaseCommands ? node.commandType : node.commandType.toUpperCase();
        cmd = `${type} ${cmd}`;
      }
      const args = this.printChildrenList(ctx, {
        indent: inp.indent,
        remaining: inp.remaining - cmd.length - 1
      });
      const optionIndent = args.indent + opts.pipeTab;
      const optionsTxt = [];
      let options = '';
      let optionsLines = 0;
      let breakOptions = false;
      for (const out of ctx.visitOptions({
        indent: optionIndent,
        remaining: opts.wrap - optionIndent.length
      })) {
        var _out$lines;
        optionsLines += (_out$lines = out.lines) !== null && _out$lines !== void 0 ? _out$lines : 1;
        optionsTxt.push(out.txt);
        options += (options ? ' ' : '') + out.txt;
      }
      breakOptions = breakOptions || args.lines > 1 || optionsLines > 1 || options.length > opts.wrap - inp.remaining - cmd.length - 1 - args.txt.length;
      if (breakOptions) {
        options = optionsTxt.join('\n' + optionIndent);
      }
      const argsWithWhitespace = args.txt ? `${args.oneArgumentPerLine ? '\n' : ' '}${args.txt}` : '';
      const optionsWithWhitespace = options ? `${breakOptions ? '\n' + optionIndent : ' '}${options}` : '';
      let txt = `${cmd}${argsWithWhitespace}${optionsWithWhitespace}`;
      const formatting = node.formatting;
      if (formatting) {
        if (formatting.left) {
          const comments = _leaf_printer.LeafPrinter.commentList(formatting.left);
          if (comments) {
            txt = `${comments} ${txt}`;
          }
        }
      }
      return {
        txt,
        lines: args.lines /* add options lines count */
      };
    }).on('visitQuery', (ctx, inp) => {
      var _ref, _inp$indent, _inp$remaining, _opts$multiline, _opts$pipeTab;
      const opts = this.opts;
      const indent = (_ref = (_inp$indent = inp === null || inp === void 0 ? void 0 : inp.indent) !== null && _inp$indent !== void 0 ? _inp$indent : opts.indent) !== null && _ref !== void 0 ? _ref : '';
      const remaining = (_inp$remaining = inp === null || inp === void 0 ? void 0 : inp.remaining) !== null && _inp$remaining !== void 0 ? _inp$remaining : opts.wrap;
      const commands = ctx.node.commands;
      const commandCount = commands.length;
      const hasHeaderCommands = !opts.skipHeader && ctx.node.header && ctx.node.header.length > 0;
      let multiline = (_opts$multiline = opts.multiline) !== null && _opts$multiline !== void 0 ? _opts$multiline : commandCount > 3;
      if (!multiline) {
        const stats = (0, _helpers.getPrettyPrintStats)(ctx.node);
        if (stats.hasLineBreakingDecorations) {
          multiline = true;
        }
      }
      if (!multiline && !hasHeaderCommands) {
        const oneLine = indent + _basic_pretty_printer.BasicPrettyPrinter.print(ctx.node, opts);
        if (oneLine.length <= remaining) {
          return {
            txt: oneLine
          };
        } else {
          multiline = true;
        }
      }

      // Special handling for queries with header commands: we want to try
      // to keep the main query on a single line if possible.
      if (hasHeaderCommands && !multiline && commandCount < 4) {
        // Use skipHeader option to format just the main query
        const mainQueryOneLine = _basic_pretty_printer.BasicPrettyPrinter.print(ctx.node, {
          ...opts,
          skipHeader: true
        });
        if (mainQueryOneLine.length <= remaining) {
          // Main query fits on one line, print headers separately then main query
          let text = indent;
          let hasHeaderCommandsOutput = false;
          for (const headerOut of ctx.visitHeaderCommands({
            indent,
            remaining: remaining - indent.length
          })) {
            if (hasHeaderCommandsOutput) {
              text += '\n' + indent;
            }
            text += headerOut.txt;
            hasHeaderCommandsOutput = true;
          }
          if (hasHeaderCommandsOutput) {
            text += '\n' + indent;
          }
          text += mainQueryOneLine;
          return {
            txt: text
          };
        } else {
          // Main query doesn't fit, use multiline formatting
          multiline = true;
        }
      }

      // Regular top-level query formatting
      let text = indent;
      const pipedCommandIndent = `${indent}${(_opts$pipeTab = opts.pipeTab) !== null && _opts$pipeTab !== void 0 ? _opts$pipeTab : '  '}`;
      const cmdSeparator = multiline ? `${pipedCommandIndent}| ` : ' | ';

      // Print header pseudo-commands first (e.g., SET instructions).
      // Each header command goes on its own line.
      let hasHeaderCommandsOutput = false;
      if (hasHeaderCommands) {
        for (const headerOut of ctx.visitHeaderCommands({
          indent,
          remaining: remaining - indent.length
        })) {
          if (hasHeaderCommandsOutput) {
            text += '\n' + indent;
          }
          text += headerOut.txt;
          hasHeaderCommandsOutput = true;
        }
      }
      let i = 0;
      let hasCommands = false;
      for (const out of ctx.visitCommands({
        indent,
        remaining: remaining - indent.length
      })) {
        const isFirstCommand = i === 0;
        const commandIndent = isFirstCommand ? indent : pipedCommandIndent;
        const topDecorations = this.printTopDecorations(commandIndent, commands[i]);
        if (topDecorations) {
          if (!isFirstCommand) {
            text += '\n';
          }
          text += topDecorations;
        }
        if (hasCommands) {
          // Separate main commands with pipe `|`
          if (multiline && !topDecorations) {
            text += '\n';
          }
          text += cmdSeparator;
        } else if (hasHeaderCommandsOutput) {
          // Separate header commands from main commands with a newline
          text += '\n' + indent;
        }
        text += out.txt;
        i++;
        hasCommands = true;
      }
      return {
        txt: text
      };
    }));
    this.opts = {
      indent: (_opts$indent = _opts.indent) !== null && _opts$indent !== void 0 ? _opts$indent : '',
      tab: (_opts$tab = _opts.tab) !== null && _opts$tab !== void 0 ? _opts$tab : '  ',
      pipeTab: (_opts$pipeTab2 = _opts.pipeTab) !== null && _opts$pipeTab2 !== void 0 ? _opts$pipeTab2 : '  ',
      skipHeader: (_opts$skipHeader = _opts.skipHeader) !== null && _opts$skipHeader !== void 0 ? _opts$skipHeader : false,
      commandTab: (_opts$commandTab = _opts.commandTab) !== null && _opts$commandTab !== void 0 ? _opts$commandTab : '    ',
      multiline: (_opts$multiline2 = _opts.multiline) !== null && _opts$multiline2 !== void 0 ? _opts$multiline2 : false,
      wrap: (_opts$wrap = _opts.wrap) !== null && _opts$wrap !== void 0 ? _opts$wrap : 80,
      lowercase: (_opts$lowercase = _opts.lowercase) !== null && _opts$lowercase !== void 0 ? _opts$lowercase : false,
      lowercaseCommands: (_ref2 = (_opts$lowercaseComman = _opts.lowercaseCommands) !== null && _opts$lowercaseComman !== void 0 ? _opts$lowercaseComman : _opts.lowercase) !== null && _ref2 !== void 0 ? _ref2 : false,
      lowercaseOptions: (_ref3 = (_opts$lowercaseOption = _opts.lowercaseOptions) !== null && _opts$lowercaseOption !== void 0 ? _opts$lowercaseOption : _opts.lowercase) !== null && _ref3 !== void 0 ? _ref3 : false,
      lowercaseFunctions: (_ref4 = (_opts$lowercaseFuncti = _opts.lowercaseFunctions) !== null && _opts$lowercaseFuncti !== void 0 ? _opts$lowercaseFuncti : _opts.lowercase) !== null && _ref4 !== void 0 ? _ref4 : false,
      lowercaseKeywords: (_ref5 = (_opts$lowercaseKeywor = _opts.lowercaseKeywords) !== null && _opts$lowercaseKeywor !== void 0 ? _opts$lowercaseKeywor : _opts.lowercase) !== null && _ref5 !== void 0 ? _ref5 : false
    };
  }
  keyword(word) {
    var _this$opts$lowercaseK;
    return ((_this$opts$lowercaseK = this.opts.lowercaseKeywords) !== null && _this$opts$lowercaseK !== void 0 ? _this$opts$lowercaseK : this.opts.lowercase) ? word.toLowerCase() : word.toUpperCase();
  }
  printBinaryOperatorExpression(ctx, operator, inp, operatorLeadingWhitespace = ' ') {
    var _inp$suffix3;
    const node = ctx.node;
    const group = (0, _grouping.binaryExpressionGroup)(node);
    const [left, right] = ctx.arguments();
    const groupLeft = (0, _grouping.binaryExpressionGroup)(left);
    const groupRight = (0, _grouping.binaryExpressionGroup)(right);
    const doGroupLeft = groupLeft && groupLeft < group;
    const doGroupRight = groupRight && groupRight < group;
    const continueVerticalFlattening = group && inp.flattenBinExpOfType === group;
    const suffix = (_inp$suffix3 = inp.suffix) !== null && _inp$suffix3 !== void 0 ? _inp$suffix3 : '';
    const oneArgumentPerLine = (0, _helpers.getPrettyPrintStats)(left).hasLineBreakingDecorations || (0, _helpers.getPrettyPrintStats)(right).hasLineBreakingDecorations;
    if (continueVerticalFlattening || oneArgumentPerLine) {
      var _ctx$parent2;
      const parent = (_ctx$parent2 = ctx.parent) === null || _ctx$parent2 === void 0 ? void 0 : _ctx$parent2.node;
      const isLeftChild = (0, _is.isBinaryExpression)(parent) && parent.args[0] === node;
      const leftInput = {
        indent: inp.indent,
        remaining: inp.remaining,
        flattenBinExpOfType: group,
        suffix: operatorLeadingWhitespace + operator
      };
      const rightTab = isLeftChild ? this.opts.tab : '';
      const rightIndent = inp.indent + rightTab + (oneArgumentPerLine ? this.opts.tab : '');
      const rightInput = {
        indent: rightIndent,
        remaining: inp.remaining - this.opts.tab.length,
        flattenBinExpOfType: group,
        suffix
      };
      const leftOut = ctx.visitArgument(0, leftInput);
      const rightOut = ctx.visitArgument(1, rightInput);
      let txt = leftOut.txt + '\n';
      if (!rightOut.indented) {
        txt += rightIndent;
      }
      txt += rightOut.txt;
      return {
        txt,
        indented: leftOut.indented
      };
    }
    let txt = '';
    let leftFormatted = _basic_pretty_printer.BasicPrettyPrinter.expression(left, this.opts);
    let rightFormatted = _basic_pretty_printer.BasicPrettyPrinter.expression(right, this.opts);
    if (doGroupLeft) {
      leftFormatted = `(${leftFormatted})`;
    }
    if (doGroupRight) {
      rightFormatted = `(${rightFormatted})`;
    }
    const length = leftFormatted.length + rightFormatted.length + operatorLeadingWhitespace.length + operator.length + 2;
    const fitsOnOneLine = length <= inp.remaining;
    let indented = false;
    if (fitsOnOneLine) {
      txt = `${leftFormatted}${operatorLeadingWhitespace}${operator} ${rightFormatted}${suffix}`;
    } else {
      const flattenVertically = group === groupLeft || group === groupRight;
      const flattenBinExpOfType = flattenVertically ? group : undefined;
      const leftInput = {
        indent: inp.indent,
        remaining: inp.remaining,
        flattenBinExpOfType
      };
      const rightInput = {
        indent: inp.indent + this.opts.tab,
        remaining: inp.remaining - this.opts.tab.length,
        flattenBinExpOfType,
        suffix
      };
      const leftOut = ctx.visitArgument(0, leftInput);
      const rightOut = ctx.visitArgument(1, rightInput);
      if (doGroupLeft) {
        leftOut.txt = `(${leftOut.txt})`;
      }
      if (doGroupRight) {
        rightOut.txt = `(${rightOut.txt})`;
      }
      txt = `${leftOut.txt}${operatorLeadingWhitespace}${operator}\n`;
      if (!rightOut.indented) {
        txt += `${inp.indent}${this.opts.tab}`;
      }
      txt += rightOut.txt;
      indented = leftOut.indented;
    }
    return {
      txt,
      indented
    };
  }

  /**
   * Prints node children as a list, separated by commas. If the list is too
   * long, it will be broken into multiple lines, otherwise it will be all
   * printed on a single line.
   *
   * The breaking into two lines happens in two stages, first the list is simply
   * wrapped into multiple lines, where the wrapping happens just before the
   * list element that is too long to fit into the remaining space.
   *
   * Alternatively, if the first ("wrapping") approach results in some line
   * still exceeding the maximum line length, or if some line in the middle of
   * the list (not the last line) contains only a single element, then the list
   * is "broken" into multiple lines, where each line contains a single
   * element.
   *
   * To summarize:
   *
   * 1. First try to print the list in a single line, if it fits.
   * 2. If it doesn't fit, try to WRAP the list into multiple lines.
   * 3. If the WRAP doesn't succeed, then BREAK the list into element-per-line
   *    format.
   *
   * @param ctx The node which contains the children to be printed.
   * @param inp Expression visitor input arguments.
   * @returns Expression visitor output.
   */
  printChildrenList(ctx, inp) {
    let txt = '';
    let lines = 1;
    let largestArg = 0;
    let argsPerLine = 0;
    let minArgsPerLine = 1e6;
    let maxArgsPerLine = 0;
    let remainingCurrentLine = inp.remaining;
    let oneArgumentPerLine = false;
    if (ctx.node.name === 'fork') {
      oneArgumentPerLine = true;
    }
    for (const child of (0, _utils.children)(ctx.node)) {
      if ((0, _helpers.getPrettyPrintStats)(child).hasLineBreakingDecorations) {
        oneArgumentPerLine = true;
        break;
      }
    }
    const commaBetweenArgs = !_constants.commandsWithNoCommaArgSeparator.has(ctx.node.name);
    if (!oneArgumentPerLine) {
      let argIndex = 0;
      ARGS: for (const arg of (0, _utils.singleItems)(ctx.arguments())) {
        if (arg.type === 'option') {
          continue;
        }
        const formattedArg = _basic_pretty_printer.BasicPrettyPrinter.expression(arg, this.opts);
        const formattedArgLength = formattedArg.length;
        const needsWrap = remainingCurrentLine < formattedArgLength;
        if (formattedArgLength > largestArg) {
          largestArg = formattedArgLength;
        }

        // Check if this command has special comma rules
        const specialRule = _constants.commandsWithSpecialCommaRules.get(ctx.node.name);
        const needsComma = specialRule ? specialRule(argIndex) : commaBetweenArgs;
        let separator = txt ? needsComma ? ',' : '' : '';
        argIndex++;
        let fragment = '';
        if (needsWrap) {
          separator += '\n' + inp.indent + this.opts.tab + (ctx instanceof _visitor.CommandVisitorContext ? this.opts.commandTab : '');
          fragment = separator + formattedArg;
          lines++;
          if (argsPerLine > maxArgsPerLine) {
            maxArgsPerLine = argsPerLine;
          }
          if (argsPerLine < minArgsPerLine) {
            minArgsPerLine = argsPerLine;
            if (minArgsPerLine < 2) {
              oneArgumentPerLine = true;
              break ARGS;
            }
          }
          remainingCurrentLine = inp.remaining - formattedArgLength - this.opts.tab.length - this.opts.commandTab.length;
          argsPerLine = 1;
        } else {
          argsPerLine++;
          fragment = separator + (separator ? ' ' : '') + formattedArg;
          remainingCurrentLine -= fragment.length;
        }
        txt += fragment;
      }
    }
    let indent = inp.indent + this.opts.tab;
    if (ctx instanceof _visitor.CommandVisitorContext) {
      var _ctx$parent3, _ctx$parent3$node, _ctx$parent3$node$com;
      const isFirstCommand = ((_ctx$parent3 = ctx.parent) === null || _ctx$parent3 === void 0 ? void 0 : (_ctx$parent3$node = _ctx$parent3.node) === null || _ctx$parent3$node === void 0 ? void 0 : (_ctx$parent3$node$com = _ctx$parent3$node.commands) === null || _ctx$parent3$node$com === void 0 ? void 0 : _ctx$parent3$node$com[0]) === ctx.node;
      if (!isFirstCommand) {
        indent += this.opts.commandTab;
      }
    }
    if (oneArgumentPerLine) {
      lines = 1;
      txt = ctx instanceof _visitor.CommandVisitorContext ? '' : '\n';
      const args = [...ctx.arguments()].filter(arg => {
        if (!arg) return false;
        if (arg.type === 'option') return arg.name === 'as';
        return true;
      });
      const length = args.length;
      const last = length - 1;
      for (let i = 0; i <= last; i++) {
        const isFirstArg = i === 0;
        const isLastArg = i === last;
        const specialRule = _constants.commandsWithSpecialCommaRules.get(ctx.node.name);
        const needsComma = specialRule ? specialRule(i) : commaBetweenArgs;
        const arg = ctx.visitExpression(args[i], {
          indent,
          remaining: this.opts.wrap - indent.length,
          suffix: isLastArg ? '' : needsComma ? ',' : ''
        });
        const indentation = arg.indented ? '' : indent;
        const formattedArg = arg.txt;
        const separator = isFirstArg ? '' : '\n';
        txt += separator + indentation + formattedArg;
        lines++;
      }
    }
    return {
      txt,
      lines,
      indent,
      oneArgumentPerLine
    };
  }
  printTopDecorations(indent, node) {
    const formatting = node.formatting;
    if (!formatting || !formatting.top || !formatting.top.length) {
      return '';
    }
    let txt = '';
    for (const decoration of formatting.top) {
      if (decoration.type === 'comment') {
        txt += indent + _leaf_printer.LeafPrinter.comment(decoration) + '\n';
      }
    }
    return txt;
  }
  decorateWithComments({
    indent,
    suffix
  }, node, txt, indented = false) {
    const formatting = node.formatting;
    if (!formatting) {
      return {
        txt: txt + (suffix !== null && suffix !== void 0 ? suffix : ''),
        indented
      };
    }
    if (formatting.left) {
      const comments = _leaf_printer.LeafPrinter.commentList(formatting.left);
      if (comments) {
        indented = true;
        txt = `${indent}${comments} ${txt}`;
      }
    }
    if (formatting.top) {
      const top = formatting.top;
      const length = top.length;
      for (let i = length - 1; i >= 0; i--) {
        const decoration = top[i];
        if (decoration.type === 'comment') {
          if (!indented) {
            txt = indent + txt;
            indented = true;
          }
          txt = indent + _leaf_printer.LeafPrinter.comment(decoration) + '\n' + txt;
          indented = true;
        }
      }
    }
    if (formatting.right) {
      const comments = _leaf_printer.LeafPrinter.commentList(formatting.right);
      if (comments) {
        txt = `${txt} ${comments}`;
      }
    }
    if (suffix) {
      txt += suffix;
    }
    if (formatting.rightSingleLine) {
      const comment = _leaf_printer.LeafPrinter.comment(formatting.rightSingleLine);
      txt += ` ${comment}`;
    }
    if (formatting.bottom) {
      for (const decoration of formatting.bottom) {
        if (decoration.type === 'comment') {
          indented = true;
          txt = txt + '\n' + indent + _leaf_printer.LeafPrinter.comment(decoration);
        }
      }
    }
    return {
      txt,
      indented
    };
  }
  print(query) {
    return this.visitor.visitQuery(query, undefined).txt;
  }
}
exports.WrappingPrettyPrinter = WrappingPrettyPrinter;
_WrappingPrettyPrinter = WrappingPrettyPrinter;
(0, _defineProperty2.default)(WrappingPrettyPrinter, "print", (query, opts) => {
  const printer = new _WrappingPrettyPrinter(opts);
  return printer.print(query);
});