"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BasicPrettyPrinter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _is = require("../ast/is");
var _grouping = require("../ast/grouping");
var _visitor = require("../ast/visitor");
var _utils = require("../ast/visitor/utils");
var _constants = require("./constants");
var _leaf_printer = require("./leaf_printer");
var _BasicPrettyPrinter;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
class BasicPrettyPrinter {
  constructor(_opts = {}) {
    var _opts$pipeTab2, _opts$multiline, _opts$skipHeader, _opts$lowercase, _ref, _opts$lowercaseComman, _ref2, _opts$lowercaseOption, _ref3, _opts$lowercaseFuncti, _ref4, _opts$lowercaseKeywor;
    (0, _defineProperty2.default)(this, "opts", void 0);
    (0, _defineProperty2.default)(this, "visitor", new _visitor.Visitor().on('visitExpression', ctx => {
      return '<EXPRESSION>';
    }).on('visitHeaderCommand', ctx => {
      const opts = this.opts;
      const cmd = opts.lowercaseCommands ? ctx.node.name : ctx.node.name.toUpperCase();
      let args = '';
      for (const arg of ctx.visitArgs()) {
        args += (args ? ', ' : '') + arg;
      }
      const argsFormatted = args ? ` ${args}` : '';
      const cmdFormatted = `${cmd}${argsFormatted};`;
      return this.decorateWithComments(ctx.node, cmdFormatted);
    }).on('visitIdentifierExpression', ctx => {
      const formatted = _leaf_printer.LeafPrinter.identifier(ctx.node);
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitSourceExpression', ctx => {
      const formatted = _leaf_printer.LeafPrinter.source(ctx.node);
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitColumnExpression', ctx => {
      const formatted = _leaf_printer.LeafPrinter.column(ctx.node);
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitLiteralExpression', ctx => {
      const formatted = _leaf_printer.LeafPrinter.literal(ctx.node);
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitInlineCastExpression', ctx => {
      const value = ctx.value();
      const wrapInBrackets = value.type !== 'literal' && value.type !== 'column' && !(value.type === 'function' && value.subtype === 'variadic-call');
      let valueFormatted = ctx.visitValue();
      if (wrapInBrackets) {
        valueFormatted = `(${valueFormatted})`;
      }
      const typeName = this.keyword(ctx.node.castType);
      const formatted = `${valueFormatted}::${typeName}`;
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitListLiteralExpression', ctx => {
      let elements = '';
      for (const arg of ctx.visitElements()) {
        elements += (elements ? ', ' : '') + arg;
      }
      const formatted = ctx.node.subtype === 'tuple' ? '(' + elements + ')' : '[' + elements + ']';
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitMapEntryExpression', ctx => {
      const key = ctx.visitKey();
      const value = ctx.visitValue();
      const formatted = key + ': ' + value;
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitMapExpression', ctx => {
      let entriesFormatted = '';
      for (const entry of ctx.visitEntries()) {
        entriesFormatted += (entriesFormatted ? ', ' : '') + entry;
      }
      const formatted = '{' + entriesFormatted + '}';
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitParensExpression', ctx => {
      const child = ctx.visitChild();
      const formatted = `(${child})`;
      return this.decorateWithComments(ctx.node, formatted);
    }).on('visitFunctionCallExpression', ctx => {
      const opts = this.opts;
      const node = ctx.node;
      let operator = ctx.operator();
      switch (node.subtype) {
        case 'unary-expression':
          {
            operator = this.keyword(operator);
            const separator = operator === '-' || operator === '+' ? '' : ' ';
            const argument = ctx.arguments()[0];
            let argumentFormatted = ctx.visitArgument(0, undefined);
            const operatorPrecedence = (0, _grouping.unaryExpressionGroup)(ctx.node);
            const argumentPrecedence = (0, _grouping.binaryExpressionGroup)(argument);
            if (argumentPrecedence !== _grouping.BinaryExpressionGroup.none && argumentPrecedence < operatorPrecedence) {
              argumentFormatted = `(${argumentFormatted})`;
            }
            const formatted = `${operator}${separator}${argumentFormatted}`;
            return this.decorateWithComments(ctx.node, formatted);
          }
        case 'postfix-unary-expression':
          {
            operator = this.keyword(operator);
            const formatted = `${ctx.visitArgument(0)} ${operator}`;
            return this.decorateWithComments(ctx.node, formatted);
          }
        case 'binary-expression':
          {
            operator = this.keyword(operator);
            const group = (0, _grouping.binaryExpressionGroup)(ctx.node);
            // Note: right operand may be undefined for incomplete expressions.
            // For assignments (=), left is the target name, right is the expression to assign.
            const [left, right] = ctx.arguments();
            const formatOperand = (operand, index) => {
              const operandGroup = (0, _grouping.binaryExpressionGroup)(operand);
              let formatted = ctx.visitArgument(index);
              const shouldGroup = operandGroup && (operandGroup === _grouping.BinaryExpressionGroup.unknown || operandGroup < group);
              if (shouldGroup) {
                formatted = `(${formatted})`;
              }
              return formatted;
            };
            if (node.name === '*' && ((0, _is.isIntegerLiteral)(left) && Math.abs(left.value) === 1 || right && (0, _is.isIntegerLiteral)(right) && Math.abs(right.value) === 1)) {
              const formatted = this.simplifyMultiplicationByOne(node);
              if (formatted) {
                return formatted;
              }
            }
            const leftFormatted = formatOperand(left, 0);
            const rightFormatted = right ? formatOperand(right, 1) : '';
            const formatted = `${leftFormatted} ${operator} ${rightFormatted}`;
            return this.decorateWithComments(ctx.node, formatted);
          }
        default:
          {
            // Check if function name is a parameter stored in node.operator
            if (ctx.node.operator && (0, _is.isParamLiteral)(ctx.node.operator)) {
              operator = _leaf_printer.LeafPrinter.param(ctx.node.operator);
            } else {
              if (ctx.node.operator && (0, _is.isIdentifier)(ctx.node.operator)) {
                operator = ctx.node.operator.name;
              }
              operator = opts.lowercaseFunctions ? operator.toLowerCase() : operator.toUpperCase();
            }
            let args = '';
            for (const arg of ctx.visitArguments()) {
              args += (args ? ', ' : '') + arg;
            }
            const formatted = `${operator}(${args})`;
            return this.decorateWithComments(ctx.node, formatted);
          }
      }
    }).on('visitOrderExpression', ctx => {
      const node = ctx.node;
      let text = ctx.visitArgument(0);
      if (node.order) {
        text += ` ${node.order}`;
      }
      if (node.nulls) {
        text += ` ${node.nulls}`;
      }
      return text;
    }).on('visitCommandOption', ctx => {
      const opts = this.opts;
      const option = opts.lowercaseOptions ? ctx.node.name : ctx.node.name.toUpperCase();
      let args = '';
      for (const arg of ctx.visitArguments()) {
        args += (args ? ', ' : '') + arg;
      }
      const separator = _constants.commandOptionsWithEqualsSeparator.has(ctx.node.name) ? ' = ' : ' ';
      const argsFormatted = args ? `${separator}${args}` : '';
      const optionFormatted = `${option}${argsFormatted}`;
      return this.decorateWithComments(ctx.node, optionFormatted);
    }).on('visitCommand', ctx => {
      const opts = this.opts;
      const node = ctx.node;
      const cmd = opts.lowercaseCommands ? node.name : node.name.toUpperCase();
      const cmdType = !node.commandType ? '' : (opts.lowercaseCommands ? node.commandType : node.commandType.toUpperCase()) + ' ';
      if (cmd === 'FORK') {
        const branches = node.args.map(branch => {
          if (Array.isArray(branch)) {
            return undefined;
          }
          if (branch.type === 'parens' && branch.child.type === 'query') {
            return ctx.visitSubQuery(branch.child);
          }
          return undefined;
        }).filter(Boolean);
        const spaces = n => ' '.repeat(n);
        const branchSeparator = opts.multiline ? `)\n${spaces(4)}(` : `) (`;
        return this.decorateWithComments(ctx.node, `FORK${opts.multiline ? `\n${spaces(4)}` : ' '}(${branches.join(branchSeparator)})`);
      }
      let args = '';
      let options = '';
      let argIndex = 0;
      for (const source of ctx.visitArguments()) {
        const needsSeparator = !!args;

        // Check if this command has special comma rules
        const specialRule = _constants.commandsWithSpecialCommaRules.get(ctx.node.name);
        const needsComma = specialRule ? specialRule(argIndex) : !_constants.commandsWithNoCommaArgSeparator.has(ctx.node.name);
        const separator = needsSeparator ? (needsComma ? ',' : '') + ' ' : '';
        args += separator + source;
        argIndex++;
      }
      for (const option of ctx.visitOptions()) {
        options += (options ? ' ' : '') + option;
      }
      const argsFormatted = args ? ` ${args}` : '';
      const optionsFormatted = options ? ` ${options}` : '';
      const cmdFormatted = `${cmdType}${cmd}${argsFormatted}${optionsFormatted}`;
      return this.decorateWithComments(ctx.node, cmdFormatted);
    }).on('visitQuery', ctx => {
      var _ctx$parent, _parentNode, _opts$pipeTab;
      const opts = this.opts;
      let parentNode;
      if ((_ctx$parent = ctx.parent) !== null && _ctx$parent !== void 0 && _ctx$parent.node && !Array.isArray(ctx.parent.node)) {
        parentNode = ctx.parent.node;
      }
      const useMultiLine = opts.multiline && !Array.isArray(parentNode) && ((_parentNode = parentNode) === null || _parentNode === void 0 ? void 0 : _parentNode.name) !== 'fork';
      const cmdSeparator = useMultiLine ? `\n${(_opts$pipeTab = opts.pipeTab) !== null && _opts$pipeTab !== void 0 ? _opts$pipeTab : '  '}| ` : ' | ';
      let text = '';

      // Print header commands first (e.g., SET instructions)
      if (!opts.skipHeader) {
        for (const headerCmd of ctx.visitHeaderCommands()) {
          if (text) text += ' ';
          text += headerCmd;
        }
      }
      let hasCommands = false;
      for (const cmd of ctx.visitCommands()) {
        if (hasCommands) {
          // Separate main commands with pipe `|`
          text += cmdSeparator;
        } else if (text) {
          // Separate header commands from main commands with just a space
          text += ' ';
        }
        text += cmd;
        hasCommands = true;
      }
      return text;
    }));
    this.opts = {
      pipeTab: (_opts$pipeTab2 = _opts.pipeTab) !== null && _opts$pipeTab2 !== void 0 ? _opts$pipeTab2 : '  ',
      multiline: (_opts$multiline = _opts.multiline) !== null && _opts$multiline !== void 0 ? _opts$multiline : false,
      skipHeader: (_opts$skipHeader = _opts.skipHeader) !== null && _opts$skipHeader !== void 0 ? _opts$skipHeader : false,
      lowercase: (_opts$lowercase = _opts.lowercase) !== null && _opts$lowercase !== void 0 ? _opts$lowercase : false,
      lowercaseCommands: (_ref = (_opts$lowercaseComman = _opts.lowercaseCommands) !== null && _opts$lowercaseComman !== void 0 ? _opts$lowercaseComman : _opts.lowercase) !== null && _ref !== void 0 ? _ref : false,
      lowercaseOptions: (_ref2 = (_opts$lowercaseOption = _opts.lowercaseOptions) !== null && _opts$lowercaseOption !== void 0 ? _opts$lowercaseOption : _opts.lowercase) !== null && _ref2 !== void 0 ? _ref2 : false,
      lowercaseFunctions: (_ref3 = (_opts$lowercaseFuncti = _opts.lowercaseFunctions) !== null && _opts$lowercaseFuncti !== void 0 ? _opts$lowercaseFuncti : _opts.lowercase) !== null && _ref3 !== void 0 ? _ref3 : false,
      lowercaseKeywords: (_ref4 = (_opts$lowercaseKeywor = _opts.lowercaseKeywords) !== null && _opts$lowercaseKeywor !== void 0 ? _opts$lowercaseKeywor : _opts.lowercase) !== null && _ref4 !== void 0 ? _ref4 : false
    };
  }
  keyword(word) {
    var _this$opts$lowercaseK;
    return ((_this$opts$lowercaseK = this.opts.lowercaseKeywords) !== null && _this$opts$lowercaseK !== void 0 ? _this$opts$lowercaseK : this.opts.lowercase) ? word.toLowerCase() : word.toUpperCase();
  }
  decorateWithComments(node, formatted) {
    const formatting = node.formatting;
    if (!formatting) {
      return formatted;
    }
    if (formatting.left) {
      const comments = _leaf_printer.LeafPrinter.commentList(formatting.left);
      if (comments) {
        formatted = `${comments} ${formatted}`;
      }
    }
    if (formatting.right) {
      const comments = _leaf_printer.LeafPrinter.commentList(formatting.right);
      if (comments) {
        formatted = `${formatted} ${comments}`;
      }
    }
    return formatted;
  }
  simplifyMultiplicationByOne(node, minusCount = 0) {
    if ((0, _is.isBinaryExpression)(node) && node.name === '*') {
      let [left, right] = node.args;
      left = (0, _utils.resolveItem)(left);
      right = (0, _utils.resolveItem)(right);
      if ((0, _is.isProperNode)(left) && (0, _is.isProperNode)(right)) {
        if (!!left.formatting || !!right.formatting) {
          return undefined;
        }
        if ((0, _is.isIntegerLiteral)(left)) {
          if (left.value === 1) {
            return this.simplifyMultiplicationByOne(right, minusCount);
          } else if (left.value === -1) {
            return this.simplifyMultiplicationByOne(right, minusCount + 1);
          }
        }
        if ((0, _is.isIntegerLiteral)(right)) {
          if (right.value === 1) {
            return this.simplifyMultiplicationByOne(left, minusCount);
          } else if (right.value === -1) {
            return this.simplifyMultiplicationByOne(left, minusCount + 1);
          }
        }
        return undefined;
      } else {
        return undefined;
      }
    }
    const isNegative = minusCount % 2 === 1;
    if (isNegative && ((0, _is.isIntegerLiteral)(node) || (0, _is.isDoubleLiteral)(node)) && node.value < 0) {
      return BasicPrettyPrinter.expression({
        ...node,
        value: Math.abs(node.value)
      }, this.opts);
    }
    let expression = BasicPrettyPrinter.expression(node, this.opts);
    const sign = isNegative ? '-' : '';
    const needsBrackets = !!sign && !(0, _is.isColumn)(node) && !(0, _is.isLiteral)(node);
    if (needsBrackets) {
      expression = `(${expression})`;
    }
    return sign ? `${sign}${expression}` : expression;
  }
  print(query) {
    return this.visitor.visitQuery(query, undefined);
  }
  printCommand(command) {
    return this.visitor.visitCommand(command, undefined);
  }
  printExpression(expression) {
    return this.visitor.visitExpression(expression, undefined);
  }
}
exports.BasicPrettyPrinter = BasicPrettyPrinter;
_BasicPrettyPrinter = BasicPrettyPrinter;
/**
 * @param query ES|QL query AST to print.
 * @returns A single-line string representation of the query.
 */
(0, _defineProperty2.default)(BasicPrettyPrinter, "print", (node, opts) => {
  return node.type === 'query' ? _BasicPrettyPrinter.query(node, opts) : node.type === 'command' ? _BasicPrettyPrinter.command(node, opts) : node.type === 'header-command' ? _BasicPrettyPrinter.command(node, opts) : _BasicPrettyPrinter.expression(node, opts);
});
/**
 * Print a query with each command on a separate line. It is also possible to
 * specify a tabbing option for the pipe character.
 *
 * @param query ES|QL query AST to print.
 * @param opts Options for pretty-printing.
 * @returns A multi-line string representation of the query.
 */
(0, _defineProperty2.default)(BasicPrettyPrinter, "multiline", (query, opts) => _BasicPrettyPrinter.print(query, {
  ...opts,
  multiline: true
}));
/**
 * @param query ES|QL query AST to print.
 * @returns A single-line string representation of the query.
 */
(0, _defineProperty2.default)(BasicPrettyPrinter, "query", (query, opts) => {
  const printer = new _BasicPrettyPrinter(opts);
  return printer.print(query);
});
/**
 * @param command ES|QL command AST node to print.
 * @returns Prints a single-line string representation of the command.
 */
(0, _defineProperty2.default)(BasicPrettyPrinter, "command", (command, opts) => {
  const printer = new _BasicPrettyPrinter(opts);
  return printer.printCommand(command);
});
/**
 * @param expression ES|QL expression AST node to print.
 * @returns Prints a single-line string representation of the expression.
 */
(0, _defineProperty2.default)(BasicPrettyPrinter, "expression", (expression, opts) => {
  const printer = new _BasicPrettyPrinter(opts);
  return printer.printExpression(expression);
});