"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateQuery = validateQuery;
var _composer = require("../../composer");
var _registry = require("../../commands/registry");
var _ast = require("../../ast");
var _utils = require("../../commands/definitions/utils");
var _query_columns_service = require("../../query_columns_service");
var _resources = require("./resources");
var _subqueries = require("./subqueries");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * ES|QL validation public API
 * It takes a query string and returns a list of messages (errors and warnings) after validate
 * The astProvider is optional, but if not provided the default one will be used.
 * This is useful for async loading the ES|QL parser and reduce the bundle size, or to swap grammar version.
 * As for the callbacks, while optional, the validation function will selectively ignore some errors types based on each callback missing.
 *
 * @param queryString - The query string to validate
 * @param callbacks - Optional callbacks for resource retrieval.
 * @param options.invalidateColumnsCache - Invalidates the columns metadata cache before validation. Has no effect if 'getColumnsFor' callback is not provided.
 *
 */
async function validateQuery(queryString, callbacks, options) {
  return validateAst(queryString, callbacks, options);
}
function shouldValidateCallback(callbacks, name) {
  return (callbacks === null || callbacks === void 0 ? void 0 : callbacks[name]) !== undefined;
}

/**
 * This function will perform an high level validation of the
 * query AST. An initial syntax validation is already performed by the parser
 * while here it can detect things like function names, types correctness and potential warnings
 * @param ast A valid AST data structure
 */
async function validateAst(queryString, callbacks, options) {
  var _parsingResult$ast$he, _callbacks$getJoinInd, _callbacks$getLicense;
  const messages = [];
  const parsingResult = _composer.EsqlQuery.fromSrc(queryString);
  const headerCommands = (_parsingResult$ast$he = parsingResult.ast.header) !== null && _parsingResult$ast$he !== void 0 ? _parsingResult$ast$he : [];
  const rootCommands = parsingResult.ast.commands;
  const [sources, availablePolicies, joinIndices] = await Promise.all([shouldValidateCallback(callbacks, 'getSources') ? (0, _resources.retrieveSources)(rootCommands, callbacks) : new Set(), shouldValidateCallback(callbacks, 'getPolicies') ? (0, _resources.retrievePolicies)(rootCommands, callbacks) : new Map(), shouldValidateCallback(callbacks, 'getJoinIndices') ? callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getJoinInd = callbacks.getJoinIndices) === null || _callbacks$getJoinInd === void 0 ? void 0 : _callbacks$getJoinInd.call(callbacks) : undefined]);
  const sourceQuery = queryString.split('|')[0];
  const sourceFields = shouldValidateCallback(callbacks, 'getColumnsFor') ? await new _query_columns_service.QueryColumns(_composer.EsqlQuery.fromSrc(sourceQuery).ast, sourceQuery, callbacks, options).asMap() : new Map();
  if (shouldValidateCallback(callbacks, 'getColumnsFor') && sourceFields.size > 0) {
    messages.push(...validateUnsupportedTypeFields(sourceFields, rootCommands));
  }
  const license = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getLicense = callbacks.getLicense) === null || _callbacks$getLicense === void 0 ? void 0 : _callbacks$getLicense.call(callbacks));
  const hasMinimumLicenseRequired = license === null || license === void 0 ? void 0 : license.hasAtLeast;

  // Validate the header commands
  for (const command of headerCommands) {
    const references = {
      sources,
      columns: new Map(),
      // no columns available in header
      policies: availablePolicies,
      query: queryString,
      joinIndices: (joinIndices === null || joinIndices === void 0 ? void 0 : joinIndices.indices) || []
    };
    const commandMessages = validateCommand(command, references, rootCommands, {
      ...callbacks,
      hasMinimumLicenseRequired
    });
    messages.push(...commandMessages);
  }

  /**
   * Even though we are validating single commands, we work with subqueries.
   *
   * The reason is that building the list of columns available in each command requires
   * the full command subsequence that precedes that command.
   */
  const subqueries = (0, _subqueries.getSubqueriesToValidate)(rootCommands);
  for (const subquery of subqueries) {
    const currentCommand = subquery.commands[subquery.commands.length - 1];
    const subqueryForColumns = currentCommand.name === 'join' ? subquery : {
      ...subquery,
      commands: subquery.commands.slice(0, -1)
    };
    const columns = shouldValidateCallback(callbacks, 'getColumnsFor') ? await new _query_columns_service.QueryColumns(subqueryForColumns, queryString, callbacks, options).asMap() : new Map();
    const references = {
      sources,
      columns,
      policies: availablePolicies,
      query: queryString,
      joinIndices: (joinIndices === null || joinIndices === void 0 ? void 0 : joinIndices.indices) || []
    };
    const commandMessages = validateCommand(currentCommand, references, rootCommands, {
      ...callbacks,
      hasMinimumLicenseRequired
    });
    messages.push(...commandMessages);
  }
  const parserErrors = parsingResult.errors;

  /**
   * Some changes to the grammar deleted the literal names for some tokens.
   * This is a workaround to restore the literals that were lost.
   *
   * See https://github.com/elastic/elasticsearch/pull/124177 for context.
   */
  for (const error of parserErrors) {
    error.message = error.message.replace(/\bLP\b/, "'('");
    error.message = error.message.replace(/\bOPENING_BRACKET\b/, "'['");
  }
  return {
    errors: [...parserErrors, ...messages.filter(({
      type
    }) => type === 'error')],
    warnings: messages.filter(({
      type
    }) => type === 'warning')
  };
}
function validateCommand(command, references, rootCommands, callbacks) {
  const messages = [];
  if (command.incomplete) {
    return messages;
  }
  // do not check the command exists, the grammar is already picking that up
  const commandDefinition = _registry.esqlCommandRegistry.getCommandByName(command.name);
  if (!commandDefinition) {
    return messages;
  }

  // Check license requirements for the command
  if (callbacks !== null && callbacks !== void 0 && callbacks.hasMinimumLicenseRequired) {
    const license = commandDefinition.metadata.license;
    if (license && !callbacks.hasMinimumLicenseRequired(license.toLowerCase())) {
      messages.push((0, _utils.getMessageFromId)({
        messageId: 'licenseRequired',
        values: {
          name: command.name.toUpperCase(),
          requiredLicense: license.toUpperCase()
        },
        locations: command.location
      }));
    }
  }
  const context = {
    columns: references.columns,
    policies: references.policies,
    sources: [...references.sources].map(source => ({
      name: source
    })),
    joinSources: references.joinIndices
  };
  if (commandDefinition.methods.validate) {
    const allErrors = commandDefinition.methods.validate(command, rootCommands, context, callbacks);
    const filteredErrors = allErrors.filter(error => {
      if (error.errorType === 'semantic' && error.requiresCallback) {
        return shouldValidateCallback(callbacks, error.requiresCallback);
      }

      // All other errors pass through (syntax errors, untagged errors, etc.)
      return true;
    });
    messages.push(...filteredErrors);
  }

  // no need to check for mandatory options passed
  // as they are already validated at syntax level
  return messages;
}
function validateUnsupportedTypeFields(fields, commands) {
  const usedColumnsInQuery = [];
  (0, _ast.walk)(commands, {
    visitColumn: node => usedColumnsInQuery.push(node.name)
  });
  const messages = [];
  for (const column of usedColumnsInQuery) {
    if (fields.has(column) && fields.get(column).type === 'unsupported') {
      messages.push((0, _utils.getMessageFromId)({
        messageId: 'unsupportedFieldType',
        values: {
          field: column
        },
        locations: {
          min: 1,
          max: 1
        }
      }));
    }
  }
  return messages;
}