"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getLocationFromCommandOrOptionName = void 0;
exports.getLocationInfo = getLocationInfo;
var _is = require("../../ast/is");
var _location = require("../../ast/location");
var _walker = require("../../ast/walker");
var _types = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const commandOptionNameToLocation = {
  eval: _types.Location.EVAL,
  where: _types.Location.WHERE,
  row: _types.Location.ROW,
  sort: _types.Location.SORT,
  stats: _types.Location.STATS,
  'inline stats': _types.Location.STATS,
  by: _types.Location.STATS_BY,
  enrich: _types.Location.ENRICH,
  with: _types.Location.ENRICH_WITH,
  dissect: _types.Location.DISSECT,
  rename: _types.Location.RENAME,
  join: _types.Location.JOIN,
  show: _types.Location.SHOW,
  completion: _types.Location.COMPLETION,
  rerank: _types.Location.RERANK,
  'join:on': _types.Location.JOIN,
  'rerank:on': _types.Location.RERANK
};

/**
 * Configuration for function-based locations.
 * Maps command name -> function name -> location config.
 * If argIndex is specified, the position must be within that specific argument.
 */
const functionBasedLocations = {
  stats: {
    where: {
      location: _types.Location.STATS_WHERE,
      displayName: 'stats_where',
      argIndex: 1
    }
  },
  'inline stats': {
    where: {
      location: _types.Location.STATS_WHERE,
      displayName: 'inline stats_where',
      argIndex: 1
    }
  }
};

/**
 * Pause before using this in new places. Where possible, use the Location enum directly.
 *
 * This is primarily around for backwards compatibility with the old system of command and option names.
 */
const getLocationFromCommandOrOptionName = name => commandOptionNameToLocation[name];

/**
 * Identifies the location ID at the given position
 */
exports.getLocationFromCommandOrOptionName = getLocationFromCommandOrOptionName;
function getLocationInfo(position, parentCommand, ast, withinAggFunction) {
  if (withinAggFunction && ast[0].name === 'ts') {
    return {
      id: _types.Location.STATS_TIMESERIES,
      displayName: 'agg_function_in_timeseries_context'
    };
  }
  const option = _walker.Walker.find(parentCommand, node => (0, _is.isOptionNode)(node) && (0, _location.within)(position, node));
  if (option) {
    var _commandOptionNameToL;
    const displayName = option.name;
    const parentCommandName = parentCommand.name;
    const contextualKey = `${parentCommandName}:${displayName}`;
    const id = (_commandOptionNameToL = commandOptionNameToLocation[contextualKey]) !== null && _commandOptionNameToL !== void 0 ? _commandOptionNameToL : getLocationFromCommandOrOptionName(displayName);
    return {
      id,
      displayName
    };
  }

  // If not in an option node, try to find a function that defines a location
  // We need to find ALL functions containing the position, then check if any have a location config
  const funcs = _walker.Walker.findAll(parentCommand, node => (0, _is.isFunctionExpression)(node) && (0, _location.within)(position, node));

  // Iterate through all matching functions to find one with a location config
  for (const func of funcs) {
    var _functionBasedLocatio;
    if (!(0, _is.isFunctionExpression)(func)) {
      continue;
    }
    const locationConfig = (_functionBasedLocatio = functionBasedLocations[parentCommand.name]) === null || _functionBasedLocatio === void 0 ? void 0 : _functionBasedLocatio[func.name];
    if (locationConfig) {
      // If argIndex is specified, position must be within that specific argument
      if (locationConfig.argIndex !== undefined) {
        const targetArg = func.args[locationConfig.argIndex];
        const arg = Array.isArray(targetArg) ? targetArg[0] : targetArg;
        if (arg && !Array.isArray(arg) && 'location' in arg && (0, _location.within)(position, arg)) {
          return {
            id: locationConfig.location,
            displayName: locationConfig.displayName
          };
        }
      } else {
        // No argIndex constraint, return immediately
        return {
          id: locationConfig.location,
          displayName: locationConfig.displayName
        };
      }
    }
  }
  const displayName = (option !== null && option !== void 0 ? option : parentCommand).name;
  const id = getLocationFromCommandOrOptionName(displayName);
  return {
    id,
    displayName
  };
}