"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestForExpression = suggestForExpression;
var _ = require("../..");
var _is = require("../../../../../ast/is");
var _location = require("../../../../../ast/location");
var _all_operators = require("../../../all_operators");
var _expressions = require("../../expressions");
var _shared = require("../../shared");
var _dispatcher = require("./operators/partial/dispatcher");
var _utils = require("./operators/partial/utils");
var _position = require("./position");
var _dispatcher2 = require("./positions/dispatcher");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const WHITESPACE_REGEX = /\s/;
const LAST_WORD_BOUNDARY_REGEX = /\b\w(?=\w*$)/;

/** Coordinates position detection, handler selection, and range attachment */
async function suggestForExpression(params) {
  const baseCtx = buildContext(params);
  const computed = computeDerivedState(baseCtx);
  const partialSuggestions = await trySuggestForPartialOperators(baseCtx);
  if (partialSuggestions !== null) {
    return {
      suggestions: attachRanges(baseCtx, partialSuggestions),
      computed
    };
  }
  const suggestions = await (0, _dispatcher2.dispatchStates)(baseCtx, computed.position);
  return {
    suggestions: attachRanges(baseCtx, suggestions),
    computed
  };
}

/**
 * Handles IN/NOT IN, LIKE/RLIKE, and IS/IS NOT operators.
 *
 * Use cases:
 * 1. Partial operator typing: "field IS ", "field IS N", "field LIKE "
 * 2. Complete operator without AST: "CASE(field IN(" - operator complete but no AST node
 *
 * - If valid AST node exists: uses it directly (preserves parser information)
 * - If AST missing/incomplete: creates synthetic node or suggests directly
 *
 * Skips when AST has a complete operator node - normal flow handles it.
 */
async function trySuggestForPartialOperators(ctx) {
  const {
    innerText,
    expressionRoot
  } = ctx;
  const detection = (0, _utils.detectNullCheck)(innerText) || (0, _utils.detectLike)(innerText) || (0, _utils.detectIn)(innerText);
  if (!detection) {
    return null;
  }

  // If we have an AST operator node, check if it's complete
  if ((expressionRoot === null || expressionRoot === void 0 ? void 0 : expressionRoot.type) === 'function') {
    var _expressionRoot$name;
    const astOperatorName = (_expressionRoot$name = expressionRoot.name) === null || _expressionRoot$name === void 0 ? void 0 : _expressionRoot$name.toLowerCase();
    const isIncomplete = expressionRoot.incomplete;

    // Build list of all operator names that we handle in the partial system
    const managedPartialOperators = [..._all_operators.nullCheckOperators.map(op => op.name), ..._all_operators.inOperators.map(op => op.name), ..._all_operators.patternMatchOperators.map(op => op.name)];

    // If AST has a complete operator node (not incomplete), let normal flow handle it
    if (managedPartialOperators.includes(astOperatorName) && !isIncomplete) {
      return null;
    }
  }
  return (0, _dispatcher.dispatchPartialOperators)(detection.operatorName, detection, ctx);
}

/** Derives innerText and option flags from the incoming params.*/
function buildContext(params) {
  var _params$options;
  const {
    query,
    cursorPosition
  } = params;
  const innerText = query.slice(0, cursorPosition);
  const isCursorFollowedByComma = query.slice(cursorPosition).trimStart().startsWith(',');
  const baseOptions = (_params$options = params.options) !== null && _params$options !== void 0 ? _params$options : {};
  const options = {
    ...baseOptions,
    isCursorFollowedByComma
  };
  return {
    query,
    cursorPosition,
    innerText,
    expressionRoot: params.expressionRoot,
    location: params.location,
    command: params.command,
    context: params.context,
    callbacks: params.callbacks,
    options
  };
}

/** Computes derived state from the expression context */
function computeDerivedState(ctx) {
  const {
    expressionRoot,
    innerText,
    cursorPosition,
    context
  } = ctx;
  const position = (0, _position.getPosition)(innerText, expressionRoot);
  const expressionType = (0, _.getExpressionType)(expressionRoot, context === null || context === void 0 ? void 0 : context.columns);
  const isComplete = (0, _expressions.isExpressionComplete)(expressionType, innerText);
  const insideFunction = expressionRoot !== undefined && (0, _is.isFunctionExpression)(expressionRoot) && (0, _location.within)(cursorPosition, expressionRoot);
  return {
    innerText,
    position,
    expressionType,
    isComplete,
    insideFunction
  };
}

/** Returns new suggestions array with range information */
function attachRanges(ctx, suggestions) {
  const {
    innerText
  } = ctx;
  const lastChar = innerText[innerText.length - 1];
  const hasNonWhitespacePrefix = !WHITESPACE_REGEX.test(lastChar);
  return suggestions.map(suggestion => {
    if ((0, _utils2.isNullCheckOperator)(suggestion.text)) {
      return {
        ...suggestion,
        rangeToReplace: (0, _shared.getOverlapRange)(innerText, suggestion.text)
      };
    }
    if (hasNonWhitespacePrefix) {
      return {
        ...suggestion,
        rangeToReplace: {
          start: innerText.search(LAST_WORD_BOUNDARY_REGEX),
          end: innerText.length
        }
      };
    }
    return {
      ...suggestion
    };
  });
}