"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestAfterComplete = suggestAfterComplete;
var _types = require("../../../../types");
var _literals = require("../../../literals");
var _is = require("../../../../../../ast/is");
var _complete_items = require("../../../../../registry/complete_items");
var _expressions = require("../../../expressions");
var _signature_analyzer = require("../signature_analyzer");
var _utils = require("../operators/utils");
var _should_suggest_operators = require("./after_complete/should_suggest_operators");
var _suggestion_builder = require("../suggestion_builder");
var _all_operators = require("../../../../all_operators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Handler for autocomplete suggestions after complete expressions.
 * Handles after_complete position for all complete expression types:
 * - Columns (e.g., "field /")
 * - Functions (e.g., "ABS(x) /")
 * - Literals (e.g., "123 /" or "true /")
 * - Postfix operators (e.g., "field IS NULL /")
 *
 * Boolean literals (true/false) have special operator filtering logic.
 */
async function suggestAfterComplete(ctx) {
  const {
    expressionRoot,
    context,
    options
  } = ctx;
  const {
    functionParameterContext
  } = options;

  // Postfix unary operators (IS NULL, IS NOT NULL) are complete boolean expressions
  // They should only suggest logical operators (AND, OR) and comma in function contexts
  if (expressionRoot && (0, _is.isFunctionExpression)(expressionRoot) && expressionRoot.subtype === 'postfix-unary-expression') {
    const suggestions = (0, _utils.getLogicalContinuationSuggestions)();

    // Check if cursor is already followed by comma to avoid duplicates
    if (options.isCursorFollowedByComma) {
      return suggestions;
    }

    // Only suggest comma if we're inside a function (indicated by functionParameterContext)
    // Top-level expressions (e.g., WHERE field IS NULL) should not suggest comma
    if (!functionParameterContext) {
      return suggestions;
    }
    const analyzer = _signature_analyzer.SignatureAnalyzer.from(functionParameterContext);

    // For repeating signatures (CASE): boolean at position 2,4,6... is a condition → suggest comma
    if (analyzer !== null && analyzer !== void 0 && analyzer.isAmbiguousPosition) {
      return [...suggestions, _complete_items.commaCompleteItem];
    }
    if (analyzer !== null && analyzer !== void 0 && analyzer.hasMoreParams) {
      return [...suggestions, _complete_items.commaCompleteItem];
    }
    return suggestions;
  }

  // Enrich context with expression type and signature analysis
  const expressionType = (0, _expressions.getExpressionType)(expressionRoot, context === null || context === void 0 ? void 0 : context.columns);
  const signatureAnalysis = functionParameterContext ? _signature_analyzer.SignatureAnalyzer.from(functionParameterContext) : null;
  const enrichedCtx = {
    ...ctx,
    expressionType,
    signatureAnalysis
  };
  if (!(0, _types.isParameterType)(expressionType) && !functionParameterContext) {
    return [];
  }

  // Case 1: Unknown type handling within function parameter context
  if (functionParameterContext && expressionType === 'unknown') {
    return [];
  }

  // Case 2: Time interval completions or standard operators
  const timeUnitItems = (_paramCtx$functionDef => {
    const paramCtx = functionParameterContext;
    if (!paramCtx || !expressionRoot) {
      return [];
    }
    if (!(0, _is.isLiteral)(expressionRoot) || !(0, _types.isNumericType)(expressionRoot.literalType)) {
      return [];
    }

    // Check if function accepts time_duration or date_period parameters
    const acceptedTypes = paramCtx.paramDefinitions.map(({
      type
    }) => type);
    const acceptsTimeParam = acceptedTypes.includes('time_duration') || acceptedTypes.includes('date_period');
    if (!acceptsTimeParam) {
      return [];
    }
    const hasMoreMandatory = Boolean(paramCtx.hasMoreMandatoryArgs);
    const fnType = (_paramCtx$functionDef = paramCtx.functionDefinition) === null || _paramCtx$functionDef === void 0 ? void 0 : _paramCtx$functionDef.type;
    const shouldAddComma = hasMoreMandatory && fnType !== _types.FunctionDefinitionTypes.OPERATOR;
    return (0, _literals.getTimeUnitLiterals)(shouldAddComma, hasMoreMandatory);
  })();

  // If time units are available, return them early (they already include comma if needed)
  if (timeUnitItems.length) {
    return timeUnitItems;
  }

  // Use SignatureAnalyzer for parameter state analysis if available
  const paramState = signatureAnalysis ? signatureAnalysis.getParameterState(expressionType, (0, _is.isLiteral)(expressionRoot)) : {
    typeMatches: false,
    isLiteral: false,
    hasMoreParams: false,
    isVariadic: false
  };

  // Check if operators make sense in this context (intelligent filtering)
  const operatorDecision = (0, _should_suggest_operators.shouldSuggestOperators)({
    expressionType,
    functionParameterContext,
    ctx: enrichedCtx
  });

  // Build suggestions using SuggestionBuilder
  const builder = new _suggestion_builder.SuggestionBuilder(enrichedCtx);

  // Special case: string/IP/version literals can't be extended with operators, only comma
  // Numeric and boolean literals can have operators (arithmetic or logical)
  const isNonExtendableLiteral = paramState.isLiteral && !(0, _types.isNumericType)(expressionType) && expressionType !== 'boolean';

  // Add operator suggestions if they make sense
  if (operatorDecision.shouldSuggest && !isNonExtendableLiteral) {
    const isBooleanLiteral = expressionType === 'boolean' && expressionRoot && (0, _is.isLiteral)(expressionRoot);

    // Boolean literals (TRUE/FALSE) should only suggest logical operators (AND, OR)
    // Other types use the decision engine's allowed list
    let allowedOperators = operatorDecision.allowedOperators;
    if (isBooleanLiteral) {
      const {
        location
      } = ctx;
      allowedOperators = _all_operators.logicalOperators.filter(({
        locationsAvailable
      }) => locationsAvailable.includes(location)).map(({
        name
      }) => name);
    }
    builder.addOperators({
      leftParamType: expressionType === 'param' ? undefined : expressionType,
      allowed: allowedOperators,
      ignored: ['=', ':']
    });
  }

  // Add comma if needed (only in function context)
  if (functionParameterContext) {
    builder.addCommaIfNeeded({
      position: 'after_complete',
      typeMatches: paramState.typeMatches,
      isLiteral: paramState.isLiteral,
      hasMoreParams: paramState.hasMoreParams,
      isVariadic: paramState.isVariadic,
      isAmbiguousPosition: signatureAnalysis === null || signatureAnalysis === void 0 ? void 0 : signatureAnalysis.isAmbiguousPosition,
      functionSignatures: signatureAnalysis === null || signatureAnalysis === void 0 ? void 0 : signatureAnalysis.getValidSignatures(),
      expressionType,
      isCursorFollowedByComma: options.isCursorFollowedByComma
    });
  }
  return builder.build();
}