"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.asCodeGroupFilterSchema = exports.asCodeFilterSchema = exports.asCodeDSLFilterSchema = exports.asCodeConditionFilterSchema = void 0;
var _configSchema = require("@kbn/config-schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Validation Schemas for As Code Filter Interface
 *
 * These schemas are used for server validation of API requests and responses
 * in * as Code APIs.
 */

// ====================================================================
// CORE FILTER OPERATOR AND VALUE SCHEMAS
// ====================================================================

/**
 * Schema for range values used in numeric and date filters
 */
const rangeSchema = _configSchema.schema.object({
  gte: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.number(), _configSchema.schema.string()], {
    meta: {
      description: 'Greater than or equal to'
    }
  })),
  lte: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.number(), _configSchema.schema.string()], {
    meta: {
      description: 'Less than or equal to'
    }
  })),
  gt: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.number(), _configSchema.schema.string()], {
    meta: {
      description: 'Greater than'
    }
  })),
  lt: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.number(), _configSchema.schema.string()], {
    meta: {
      description: 'Less than'
    }
  }))
});

// ====================================================================
// BASE PROPERTIES (SHARED BY ALL FILTERS)
// ====================================================================

/**
 * Base properties shared by all simplified filters
 */
const basePropertiesSchema = _configSchema.schema.object({
  pinned: _configSchema.schema.maybe(_configSchema.schema.boolean({
    meta: {
      description: 'Whether the filter is pinned'
    }
  })),
  disabled: _configSchema.schema.maybe(_configSchema.schema.boolean({
    meta: {
      description: 'Whether the filter is disabled'
    }
  })),
  controlledBy: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Optional identifier for the component/plugin managing this filter'
    }
  })),
  dataViewId: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Data view ID that this filter applies to'
    }
  })),
  negate: _configSchema.schema.maybe(_configSchema.schema.boolean({
    meta: {
      description: 'Whether to negate the filter condition'
    }
  })),
  label: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Human-readable label for the filter'
    }
  })),
  isMultiIndex: _configSchema.schema.maybe(_configSchema.schema.boolean({
    meta: {
      description: 'Whether this filter can be applied to multiple indices'
    }
  })),
  filterType: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Filter type from legacy filters (e.g., "spatial_filter", "query_string") for backwards compatibility'
    }
  })),
  key: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Field name metadata from legacy filters for backwards compatibility'
    }
  })),
  value: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Value metadata from legacy filters for backwards compatibility'
    }
  }))
});

// ====================================================================
// FILTER CONDITION SCHEMAS
// ====================================================================

/**
 * Common field property for all filter conditions
 */
const conditionFieldSchema = _configSchema.schema.object({
  field: _configSchema.schema.string({
    meta: {
      description: 'Field the filter applies to'
    }
  })
});

/**
 * Schema for 'is' and 'is_not' operators with single value
 */
const singleConditionSchema = conditionFieldSchema.extends({
  operator: _configSchema.schema.oneOf([_configSchema.schema.literal('is'), _configSchema.schema.literal('is_not')], {
    meta: {
      description: 'Single value comparison operators'
    }
  }),
  value: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.number(), _configSchema.schema.boolean()], {
    meta: {
      description: 'Single value for comparison'
    }
  })
});

/**
 * Schema for 'is_one_of' and 'is_not_one_of' operators with array values
 */
const oneOfConditionSchema = conditionFieldSchema.extends({
  operator: _configSchema.schema.oneOf([_configSchema.schema.literal('is_one_of'), _configSchema.schema.literal('is_not_one_of')], {
    meta: {
      description: 'Array value comparison operators'
    }
  }),
  value: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.arrayOf(_configSchema.schema.number()), _configSchema.schema.arrayOf(_configSchema.schema.boolean())], {
    meta: {
      description: 'Homogeneous array of values'
    }
  })
});

/**
 * Schema for 'range' operator with range value
 */
const rangeConditionSchema = conditionFieldSchema.extends({
  operator: _configSchema.schema.literal('range'),
  value: rangeSchema
});

/**
 * Schema for 'exists' and 'not_exists' operators without value
 */
const existsConditionSchema = conditionFieldSchema.extends({
  operator: _configSchema.schema.oneOf([_configSchema.schema.literal('exists'), _configSchema.schema.literal('not_exists')], {
    meta: {
      description: 'Field existence check operators'
    }
  })
  // value is intentionally omitted for exists/not_exists operators
});

/**
 * Discriminated union schema for simple filter conditions with proper operator/value type combinations
 */
const conditionSchema = _configSchema.schema.oneOf([singleConditionSchema, oneOfConditionSchema, rangeConditionSchema, existsConditionSchema], {
  meta: {
    description: 'A filter condition with strict operator/value type matching'
  }
});

// ====================================================================
// FILTER DISCRIMINATED UNION SCHEMA
// ====================================================================

/**
 * Schema for condition filters
 */
const asCodeConditionFilterSchema = exports.asCodeConditionFilterSchema = basePropertiesSchema.extends({
  condition: conditionSchema
}, {
  meta: {
    description: 'Condition filter'
  }
});

/**
 * Schema for logical filter groups with recursive structure
 * Uses lazy schema to handle recursive references
 */
const GROUP_FILTER_ID = '@kbn/es-query-server_groupFilter'; // package prefix for global uniqueness in OAS specs
const asCodeGroupFilterSchema = exports.asCodeGroupFilterSchema = basePropertiesSchema.extends({
  group: _configSchema.schema.object({
    type: _configSchema.schema.oneOf([_configSchema.schema.literal('and'), _configSchema.schema.literal('or')]),
    conditions: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([conditionSchema, _configSchema.schema.lazy(GROUP_FILTER_ID) // Recursive reference for nested groups
    ]))
  }, {
    meta: {
      description: 'Condition or nested group filter',
      id: GROUP_FILTER_ID
    }
  })
}, {
  meta: {
    description: 'Grouped condition filter'
  }
});

/**
 * Schema for DSL filters
 */
const asCodeDSLFilterSchema = exports.asCodeDSLFilterSchema = basePropertiesSchema.extends({
  dsl: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any(), {
    meta: {
      description: 'Elasticsearch Query DSL object'
    }
  })
});

/**
 * Main discriminated union schema for Filter
 * Ensures exactly one of: condition, group, or dsl is present
 */
const asCodeFilterSchema = exports.asCodeFilterSchema = _configSchema.schema.oneOf([asCodeConditionFilterSchema, asCodeGroupFilterSchema, asCodeDSLFilterSchema], {
  meta: {
    description: 'A filter which can be a condition, group, or raw DSL'
  }
});