"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postActionsConnectorExecuteRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _server = require("@kbn/data-plugin/server");
var _configSchema = require("@kbn/config-schema");
var _uuid = require("uuid");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _common = require("@kbn/elastic-assistant-common/impl/schemas/common");
var _inferenceCommon = require("@kbn/inference-common");
var _prompt = require("../lib/prompt");
var _event_based_telemetry = require("../lib/telemetry/event_based_telemetry");
var _build_response = require("../lib/build_response");
var _helpers = require("./helpers");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const postActionsConnectorExecuteRoute = (router, config) => {
  const RESPONSE_TIMEOUT = config === null || config === void 0 ? void 0 : config.responseTimeout;
  router.versioned.post({
    access: 'internal',
    path: _elasticAssistantCommon.POST_ACTIONS_CONNECTOR_EXECUTE,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    },
    options: {
      timeout: {
        // Add extra time to the timeout to account for the time it takes to process the request
        idleSocket: RESPONSE_TIMEOUT + 30 * 1000
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        body: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ExecuteConnectorRequestBody),
        params: _configSchema.schema.object({
          connectorId: _configSchema.schema.string()
        }),
        query: (0, _common.buildRouteValidationWithZod)(_elasticAssistantCommon.ExecuteConnectorRequestQuery)
      }
    }
  }, async (context, request, response) => {
    var _await$coreContext$fe, _coreContext$featureF, _await$coreContext$fe2, _coreContext$featureF2;
    const abortSignal = (0, _server.getRequestAbortedSignal)(request.events.aborted$);
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const assistantContext = ctx.elasticAssistant;
    const logger = assistantContext.logger;
    const telemetry = assistantContext.telemetry;
    let onLlmResponse;
    const coreContext = await context.core;
    const inferenceChatModelDisabled = (_await$coreContext$fe = await (coreContext === null || coreContext === void 0 ? void 0 : (_coreContext$featureF = coreContext.featureFlags) === null || _coreContext$featureF === void 0 ? void 0 : _coreContext$featureF.getBooleanValue(_elasticAssistantCommon.INFERENCE_CHAT_MODEL_DISABLED_FEATURE_FLAG, false))) !== null && _await$coreContext$fe !== void 0 ? _await$coreContext$fe : false;
    const assistantInterruptsEnabled = (_await$coreContext$fe2 = await (coreContext === null || coreContext === void 0 ? void 0 : (_coreContext$featureF2 = coreContext.featureFlags) === null || _coreContext$featureF2 === void 0 ? void 0 : _coreContext$featureF2.getBooleanValue(_elasticAssistantCommon.ASSISTANT_INTERRUPTS_ENABLED_FEATURE_FLAG, false))) !== null && _await$coreContext$fe2 !== void 0 ? _await$coreContext$fe2 : false;
    try {
      var _await$ctx$elasticAss;
      const checkResponse = await (0, _helpers.performChecks)({
        context: ctx,
        request,
        response
      });
      if (!checkResponse.isSuccess) {
        return checkResponse.response;
      }
      let latestReplacements = request.body.replacements;
      const onNewReplacements = newReplacements => {
        latestReplacements = {
          ...latestReplacements,
          ...newReplacements
        };
      };
      const threadId = (0, _uuid.v4)();
      let newMessage;
      const conversationId = request.body.conversationId;
      const actionTypeId = request.body.actionTypeId;
      const screenContext = request.body.screenContext;
      const connectorId = decodeURIComponent(request.params.connectorId);

      // if message is undefined, it means the user is regenerating a message from the stored conversation
      if (request.body.message) {
        newMessage = {
          content: request.body.message,
          role: 'user'
        };
      }

      // get the actions plugin start contract from the request context:
      const actions = ctx.elasticAssistant.actions;
      const inference = ctx.elasticAssistant.inference;
      const savedObjectsClient = ctx.elasticAssistant.savedObjectsClient;
      const productDocsAvailable = (_await$ctx$elasticAss = await ctx.elasticAssistant.llmTasks.retrieveDocumentationAvailable({
        inferenceId: _inferenceCommon.defaultInferenceEndpoints.ELSER
      })) !== null && _await$ctx$elasticAss !== void 0 ? _await$ctx$elasticAss : false;
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const connectors = await actionsClient.getBulk({
        ids: [connectorId]
      });
      const connector = connectors.length > 0 ? connectors[0] : undefined;
      const isOssModel = (0, _utils.isOpenSourceModel)(connector);
      const conversationsDataClient = await assistantContext.getAIAssistantConversationsDataClient({
        assistantInterruptsEnabled
      });
      if (conversationId) {
        var _checkResponse$curren, _checkResponse$curren2;
        const conversation = await (conversationsDataClient === null || conversationsDataClient === void 0 ? void 0 : conversationsDataClient.getConversation({
          id: conversationId
        }));
        if (conversation && !(0, _elasticAssistantCommon.getIsConversationOwner)(conversation, {
          name: (_checkResponse$curren = checkResponse.currentUser) === null || _checkResponse$curren === void 0 ? void 0 : _checkResponse$curren.username,
          id: (_checkResponse$curren2 = checkResponse.currentUser) === null || _checkResponse$curren2 === void 0 ? void 0 : _checkResponse$curren2.profile_uid
        })) {
          return resp.error({
            body: `Updating a conversation is only allowed for the owner of the conversation.`,
            statusCode: 403
          });
        }
      }
      const promptsDataClient = await assistantContext.getAIAssistantPromptsDataClient();
      const contentReferencesStore = (0, _elasticAssistantCommon.newContentReferencesStore)({
        disabled: request.query.content_references_disabled
      });
      onLlmResponse = async ({
        content,
        refusal,
        traceData,
        isError,
        interruptValue
      }) => {
        if (conversationsDataClient && conversationId) {
          const {
            prunedContent,
            prunedContentReferencesStore
          } = (0, _elasticAssistantCommon.pruneContentReferences)(content, contentReferencesStore);
          await (0, _helpers.appendAssistantMessageToConversation)({
            conversationId,
            conversationsDataClient,
            messageContent: prunedContent,
            messageRefusal: refusal,
            replacements: latestReplacements,
            isError,
            traceData,
            contentReferences: prunedContentReferencesStore,
            interruptValue
          });
        }
      };
      const promptIds = request.body.promptIds;
      let systemPrompt;
      if (conversationsDataClient && promptsDataClient && conversationId) {
        systemPrompt = await (0, _helpers.getSystemPromptFromUserConversation)({
          conversationsDataClient,
          conversationId,
          promptsDataClient
        });
      }
      if (promptIds) {
        const additionalSystemPrompt = await (0, _prompt.getPrompt)({
          actionsClient,
          connectorId,
          // promptIds is promptId and promptGroupId
          ...promptIds,
          savedObjectsClient
        });
        systemPrompt = systemPrompt && systemPrompt.length ? `${systemPrompt}\n\n${additionalSystemPrompt}` : additionalSystemPrompt;
      }
      const timeout = new Promise((_, reject) => {
        setTimeout(() => {
          reject(new Error('Request timed out, increase xpack.elasticAssistant.responseTimeout'));
        }, config === null || config === void 0 ? void 0 : config.responseTimeout);
      });
      return await Promise.race([(0, _helpers.langChainExecute)({
        abortSignal,
        isStream: request.body.subAction !== 'invokeAI',
        actionsClient,
        actionTypeId,
        connectorId,
        threadId,
        contentReferencesStore,
        isOssModel,
        inferenceChatModelDisabled,
        conversationId,
        context: ctx,
        logger,
        inference,
        messages: newMessage ? [newMessage] : [],
        onLlmResponse,
        onNewReplacements,
        replacements: latestReplacements,
        request,
        response,
        telemetry,
        savedObjectsClient,
        screenContext,
        systemPrompt,
        ...(productDocsAvailable ? {
          llmTasks: ctx.elasticAssistant.llmTasks
        } : {})
      }), timeout]);
    } catch (err) {
      var _await$assistantConte;
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      if (onLlmResponse) {
        await onLlmResponse({
          content: error.message,
          traceData: {},
          isError: true
        });
      }
      const kbDataClient = (_await$assistantConte = await assistantContext.getAIAssistantKnowledgeBaseDataClient()) !== null && _await$assistantConte !== void 0 ? _await$assistantConte : undefined;
      const isKnowledgeBaseInstalled = await (0, _helpers.getIsKnowledgeBaseInstalled)(kbDataClient);
      telemetry.reportEvent(_event_based_telemetry.INVOKE_ASSISTANT_ERROR_EVENT.eventType, {
        actionTypeId: request.body.actionTypeId,
        model: request.body.model,
        errorMessage: error.message,
        assistantStreamingEnabled: request.body.subAction !== 'invokeAI',
        isEnabledKnowledgeBase: isKnowledgeBaseInstalled,
        errorLocation: 'postActionsConnectorExecuteRoute'
      });
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.postActionsConnectorExecuteRoute = postActionsConnectorExecuteRoute;