"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeSingleTab = void 0;
var _esQuery = require("@kbn/es-query");
var _lodash = require("lodash");
var _internal_state = require("../internal_state");
var _get_initial_app_state = require("../../utils/get_initial_app_state");
var _data_views = require("./data_views");
var _cleanup_url_state = require("../../utils/cleanup_url_state");
var _get_esql_data_view = require("../../utils/get_esql_data_view");
var _resolve_data_view = require("../../utils/resolve_data_view");
var _data_sources = require("../../../../../../common/data_sources");
var _discover_saved_search_container = require("../../discover_saved_search_container");
var _validate_time = require("../../../../../utils/validate_time");
var _get_valid_filters = require("../../../../../utils/get_valid_filters");
var _update_saved_search = require("../../utils/update_saved_search");
var _common = require("../../../../../../common");
var _runtime_state = require("../runtime_state");
var _tabs = require("./tabs");
var _selectors = require("../selectors");
var _constants = require("../../../../../../common/constants");
var _tab_mapping_utils = require("../tab_mapping_utils");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const initializeSingleTab = exports.initializeSingleTab = (0, _utils.createInternalStateAsyncThunk)('internalState/initializeSingleTab', async function initializeSingleTabThunkFn({
  tabId,
  initializeSingleTabParams: {
    stateContainer,
    customizationService,
    dataViewSpec,
    esqlControls,
    defaultUrlState
  }
}, {
  dispatch,
  getState,
  extra: {
    services,
    runtimeStateManager,
    urlStateStorage,
    searchSessionManager
  }
}) {
  var _urlAppState$query, _tabInitialInternalSt, _tabInitialInternalSt2;
  dispatch((0, _tabs.disconnectTab)({
    tabId
  }));
  dispatch(_internal_state.internalStateSlice.actions.resetOnSavedSearchChange({
    tabId
  }));
  const {
    currentDataView$,
    stateContainer$,
    customizationService$,
    scopedEbtManager$
  } = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, tabId);

  /**
   * New tab initialization with the restored data if available
   */

  let tabInitialGlobalState;
  let tabInitialAppState;
  let tabInitialInternalState;
  const tabState = (0, _selectors.selectTab)(getState(), tabId);
  if (tabState.globalState) {
    tabInitialGlobalState = (0, _lodash.cloneDeep)(tabState.globalState);
  }
  if (tabState.appState) {
    tabInitialAppState = (0, _lodash.cloneDeep)(tabState.appState);
  }
  if (tabState.initialInternalState) {
    tabInitialInternalState = (0, _lodash.cloneDeep)(tabState.initialInternalState);
  }
  if (esqlControls) {
    tabInitialInternalState = {
      ...tabInitialInternalState,
      controlGroupJson: JSON.stringify(esqlControls)
    };
    dispatch(_internal_state.internalStateSlice.actions.setControlGroupState({
      tabId,
      controlGroupState: esqlControls
    }));
    dispatch(_internal_state.internalStateSlice.actions.setEsqlVariables({
      tabId,
      esqlVariables: (0, _utils.extractEsqlVariables)(esqlControls)
    }));
  }

  // Get a snapshot of the current URL state before any async work is done
  // to avoid race conditions if the URL changes during tab initialization,
  // e.g. if the user quickly switches tabs
  const urlGlobalState = urlStateStorage.get(_constants.GLOBAL_STATE_URL_KEY);
  const urlAppState = {
    ...tabInitialAppState,
    ...(defaultUrlState !== null && defaultUrlState !== void 0 ? defaultUrlState : (0, _cleanup_url_state.cleanupUrlState)(urlStateStorage.get(_common.APP_STATE_URL_KEY), services.uiSettings))
  };
  const discoverTabLoadTracker = scopedEbtManager$.getValue().trackPerformanceEvent('discoverLoadSavedSearch');
  const {
    persistedDiscoverSession
  } = getState();
  const persistedTab = persistedDiscoverSession === null || persistedDiscoverSession === void 0 ? void 0 : persistedDiscoverSession.tabs.find(tab => tab.id === tabId);
  const persistedTabSavedSearch = persistedDiscoverSession && persistedTab ? await (0, _tab_mapping_utils.fromSavedObjectTabToSavedSearch)({
    tab: persistedTab,
    discoverSession: persistedDiscoverSession,
    services
  }) : undefined;
  const initialQuery = (_urlAppState$query = urlAppState === null || urlAppState === void 0 ? void 0 : urlAppState.query) !== null && _urlAppState$query !== void 0 ? _urlAppState$query : persistedTabSavedSearch === null || persistedTabSavedSearch === void 0 ? void 0 : persistedTabSavedSearch.searchSource.getField('query');
  const isEsqlMode = (0, _esQuery.isOfAggregateQueryType)(initialQuery);
  const initialDataViewIdOrSpec = (_tabInitialInternalSt = tabInitialInternalState) === null || _tabInitialInternalSt === void 0 ? void 0 : (_tabInitialInternalSt2 = _tabInitialInternalSt.serializedSearchSource) === null || _tabInitialInternalSt2 === void 0 ? void 0 : _tabInitialInternalSt2.index;
  const initialAdHocDataViewSpec = (0, _lodash.isObject)(initialDataViewIdOrSpec) ? initialDataViewIdOrSpec : undefined;
  const persistedTabDataView = persistedTabSavedSearch === null || persistedTabSavedSearch === void 0 ? void 0 : persistedTabSavedSearch.searchSource.getField('index');
  const dataViewId = (0, _data_sources.isDataViewSource)(urlAppState === null || urlAppState === void 0 ? void 0 : urlAppState.dataSource) ? urlAppState === null || urlAppState === void 0 ? void 0 : urlAppState.dataSource.dataViewId : persistedTabDataView === null || persistedTabDataView === void 0 ? void 0 : persistedTabDataView.id;
  const tabHasInitialAdHocDataViewSpec = dataViewId && (initialAdHocDataViewSpec === null || initialAdHocDataViewSpec === void 0 ? void 0 : initialAdHocDataViewSpec.id) === dataViewId;
  const peristedTabHasAdHocDataView = Boolean(persistedTabDataView && !persistedTabDataView.isPersisted());
  const {
    initializationState,
    defaultProfileAdHocDataViewIds
  } = getState();
  const profileDataViews = runtimeStateManager.adHocDataViews$.getValue().filter(({
    id
  }) => id && defaultProfileAdHocDataViewIds.includes(id));
  const profileDataViewsExist = profileDataViews.length > 0;
  const locationStateHasDataViewSpec = Boolean(dataViewSpec);
  const canAccessWithoutPersistedDataView = isEsqlMode || tabHasInitialAdHocDataViewSpec || peristedTabHasAdHocDataView || profileDataViewsExist || locationStateHasDataViewSpec;
  if (!initializationState.hasUserDataView && !canAccessWithoutPersistedDataView) {
    return {
      showNoDataPage: true
    };
  }

  /**
   * Tab initialization
   */

  let dataView;
  if ((0, _esQuery.isOfAggregateQueryType)(initialQuery)) {
    // Regardless of what was requested, we always use ad hoc data views for ES|QL
    dataView = await (0, _get_esql_data_view.getEsqlDataView)(initialQuery, persistedTabDataView !== null && persistedTabDataView !== void 0 ? persistedTabDataView : currentDataView$.getValue(), services);
  } else {
    // Load the requested data view if one exists, or a fallback otherwise
    const result = await (0, _resolve_data_view.loadAndResolveDataView)({
      dataViewId,
      locationDataViewSpec: dataViewSpec,
      initialAdHocDataViewSpec,
      savedSearch: persistedTabSavedSearch,
      isEsqlMode,
      services,
      internalState: stateContainer.internalState,
      runtimeStateManager
    });
    dataView = result.dataView;
  }
  dispatch((0, _data_views.setDataView)({
    tabId,
    dataView
  }));
  if (!dataView.isPersisted()) {
    dispatch((0, _data_views.appendAdHocDataViews)(dataView));
  }
  const initialGlobalState = {
    ...(persistedTabSavedSearch !== null && persistedTabSavedSearch !== void 0 && persistedTabSavedSearch.timeRestore && dataView.isTimeBased() ? (0, _lodash.pick)(persistedTabSavedSearch, 'timeRange', 'refreshInterval') : undefined),
    ...tabInitialGlobalState
  };
  if (urlGlobalState !== null && urlGlobalState !== void 0 && urlGlobalState.time) {
    initialGlobalState.timeRange = urlGlobalState.time;
  }
  if (urlGlobalState !== null && urlGlobalState !== void 0 && urlGlobalState.refreshInterval) {
    initialGlobalState.refreshInterval = urlGlobalState.refreshInterval;
  }
  if (urlGlobalState !== null && urlGlobalState !== void 0 && urlGlobalState.filters) {
    initialGlobalState.filters = urlGlobalState.filters;
  }

  // Get the initial app state based on a combo of the URL and persisted tab saved search,
  // then get an updated copy of the saved search with the applied initial state
  const initialAppState = (0, _get_initial_app_state.getInitialAppState)({
    initialUrlState: urlAppState,
    persistedTab,
    dataView,
    services
  });
  const savedSearch = (0, _update_saved_search.updateSavedSearch)({
    savedSearch: persistedTabSavedSearch ? (0, _discover_saved_search_container.copySavedSearch)(persistedTabSavedSearch) : services.savedSearch.getNew(),
    dataView,
    initialInternalState: tabInitialInternalState,
    appState: initialAppState,
    globalState: initialGlobalState,
    services
  });

  /**
   * Sync global services
   */

  // Only update global services if this is still the current tab
  if (getState().tabs.unsafeCurrentId === tabId) {
    var _tabInitialInternalSt3;
    // Push the tab's initial search session ID to the URL if one exists,
    // unless it should be overridden by a search session ID already in the URL
    if ((_tabInitialInternalSt3 = tabInitialInternalState) !== null && _tabInitialInternalSt3 !== void 0 && _tabInitialInternalSt3.searchSessionId && !searchSessionManager.hasSearchSessionIdInURL()) {
      searchSessionManager.pushSearchSessionIdToURL(tabInitialInternalState.searchSessionId, {
        replace: true
      });
    }

    // Cleaning up the previous state
    services.filterManager.setAppFilters([]);
    services.data.query.queryString.clearQuery();
    if (initialGlobalState.timeRange && (0, _validate_time.isTimeRangeValid)(initialGlobalState.timeRange)) {
      services.timefilter.setTime(initialGlobalState.timeRange);
    }
    if (initialGlobalState.refreshInterval && (0, _validate_time.isRefreshIntervalValid)(initialGlobalState.refreshInterval)) {
      services.timefilter.setRefreshInterval(initialGlobalState.refreshInterval);
    }
    if (initialGlobalState.filters) {
      services.filterManager.setGlobalFilters((0, _lodash.cloneDeep)(initialGlobalState.filters));
    }
    if (initialAppState.filters) {
      services.filterManager.setAppFilters((0, _lodash.cloneDeep)(initialAppState.filters));
    }

    // some filters may not be valid for this context, so update
    // the filter manager with a modified list of valid filters
    const currentFilters = services.filterManager.getFilters();
    const validFilters = (0, _get_valid_filters.getValidFilters)(dataView, currentFilters);
    if (!(0, _lodash.isEqual)(currentFilters, validFilters)) {
      services.filterManager.setFilters(validFilters);
    }
    if (initialAppState.query) {
      services.data.query.queryString.setQuery(initialAppState.query);
    }
  }

  /**
   * Update state containers
   */

  if (persistedTabSavedSearch) {
    // Set the persisted tab saved search first, then assign the
    // updated saved search to ensure unsaved changes are detected
    stateContainer.savedSearchState.set(persistedTabSavedSearch);
    stateContainer.savedSearchState.assignNextSavedSearch(savedSearch);
  } else {
    stateContainer.savedSearchState.set(savedSearch);
  }

  // Make sure app state is completely reset
  dispatch(_internal_state.internalStateSlice.actions.resetAppState({
    tabId,
    appState: initialAppState
  }));

  // Set runtime state
  stateContainer$.next(stateContainer);
  customizationService$.next(customizationService);

  // Begin syncing the state and trigger the initial fetch
  // if this is still the current tab, otherwise mark the
  // tab to fetch when selected
  if (getState().tabs.unsafeCurrentId === tabId) {
    stateContainer.actions.initializeAndSync();
    stateContainer.actions.fetchData(true);
  } else {
    dispatch(_internal_state.internalStateSlice.actions.setForceFetchOnSelect({
      tabId,
      forceFetchOnSelect: true
    }));
  }
  discoverTabLoadTracker.reportEvent();
  return {
    showNoDataPage: false
  };
});