"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSearchSessionRestorationDataProvider = createSearchSessionRestorationDataProvider;
exports.getDiscoverStateContainer = getDiscoverStateContainer;
var _i18n = require("@kbn/i18n");
var _public = require("@kbn/kibana-utils-plugin/public");
var _public2 = require("@kbn/data-plugin/public");
var _public3 = require("@kbn/data-views-plugin/public");
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _esqlUtils = require("@kbn/esql-utils");
var _esQuery = require("@kbn/es-query");
var _lodash = require("lodash");
var _types = require("../../types");
var _change_data_view = require("./utils/change_data_view");
var _build_state_subscribe = require("./utils/build_state_subscribe");
var _add_log = require("../../../utils/add_log");
var _discover_data_state_container = require("./discover_data_state_container");
var _common = require("../../../../common");
var _cleanup_url_state = require("./utils/cleanup_url_state");
var _update_filter_references = require("./utils/update_filter_references");
var _data_sources = require("../../../../common/data_sources");
var _redux = require("./redux");
var _discover_saved_search_container = require("./discover_saved_search_container");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Builds and returns appState and globalState containers and helper functions
 * Used to sync URL with UI state
 */
function getDiscoverStateContainer({
  tabId,
  services,
  customizationContext,
  stateStorageContainer: stateStorage,
  internalState,
  runtimeStateManager,
  searchSessionManager
}) {
  const injectCurrentTab = (0, _redux.createTabActionInjector)(tabId);
  const getCurrentTab = () => (0, _redux.selectTab)(internalState.getState(), tabId);

  /**
   * Saved Search State Container, the persisted saved object of Discover
   */
  const savedSearchContainer = (0, _discover_saved_search_container.getSavedSearchContainer)({
    services,
    internalState,
    getCurrentTab
  });
  const pauseAutoRefreshInterval = async dataView => {
    if (dataView && (!dataView.isTimeBased() || dataView.type === _public3.DataViewType.ROLLUP)) {
      const state = (0, _redux.selectTab)(internalState.getState(), tabId).globalState;
      if (state !== null && state !== void 0 && state.refreshInterval && !state.refreshInterval.pause) {
        internalState.dispatch(injectCurrentTab(_redux.internalStateActions.updateGlobalState)({
          globalState: {
            refreshInterval: {
              ...state.refreshInterval,
              pause: true
            }
          }
        }));
      }
    }
  };
  const setDataView = dataView => {
    internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setDataView)({
      dataView
    }));
    pauseAutoRefreshInterval(dataView);
    savedSearchContainer.getState().searchSource.setField('index', dataView);
  };
  const dataStateContainer = (0, _discover_data_state_container.getDataStateContainer)({
    services,
    searchSessionManager,
    internalState,
    runtimeStateManager,
    savedSearchContainer,
    setDataView,
    injectCurrentTab,
    getCurrentTab
  });

  /**
   * When editing an ad hoc data view, a new id needs to be generated for the data view
   * This is to prevent duplicate ids messing with our system
   */
  const updateAdHocDataViewId = async editedDataView => {
    const {
      currentDataView$
    } = (0, _redux.selectTabRuntimeState)(runtimeStateManager, tabId);
    const prevDataView = currentDataView$.getValue();
    if (!prevDataView || prevDataView.isPersisted()) return;
    const isUsedInMultipleTabs = (0, _redux.selectIsDataViewUsedInMultipleRuntimeTabStates)(runtimeStateManager, prevDataView.id);
    const nextDataView = await services.dataViews.create({
      ...editedDataView.toSpec(),
      id: (0, _uuid.v4)()
    });
    if (!isUsedInMultipleTabs) {
      services.dataViews.clearInstanceCache(prevDataView.id);
    }
    await (0, _update_filter_references.updateFiltersReferences)({
      prevDataView,
      nextDataView,
      services
    });
    if (isUsedInMultipleTabs) {
      internalState.dispatch(_redux.internalStateActions.appendAdHocDataViews(nextDataView));
    } else {
      internalState.dispatch(_redux.internalStateActions.replaceAdHocDataViewWithId(prevDataView.id, nextDataView));
    }
    if ((0, _data_sources.isDataSourceType)(getCurrentTab().appState.dataSource, _data_sources.DataSourceType.DataView)) {
      await internalState.dispatch(injectCurrentTab(_redux.internalStateActions.updateAppStateAndReplaceUrl)({
        appState: {
          dataSource: nextDataView.id ? (0, _data_sources.createDataViewDataSource)({
            dataViewId: nextDataView.id
          }) : undefined
        }
      }));
    }
    const trackingEnabled = Boolean(nextDataView.isPersisted() || savedSearchContainer.getId());
    services.urlTracker.setTrackingEnabled(trackingEnabled);
    return nextDataView;
  };
  const onOpenSavedSearch = async newSavedSearchId => {
    (0, _add_log.addLog)('[discoverState] onOpenSavedSearch', newSavedSearchId);
    const {
      persistedDiscoverSession
    } = internalState.getState();
    if ((persistedDiscoverSession === null || persistedDiscoverSession === void 0 ? void 0 : persistedDiscoverSession.id) === newSavedSearchId) {
      (0, _add_log.addLog)('[discoverState] undo changes since saved search did not change');
      await internalState.dispatch(_redux.internalStateActions.resetDiscoverSession()).unwrap();
    } else {
      (0, _add_log.addLog)('[discoverState] onOpenSavedSearch open view URL');
      services.locator.navigate({
        savedSearchId: newSavedSearchId
      });
    }
  };
  const transitionFromESQLToDataView = dataViewId => {
    internalState.dispatch(injectCurrentTab(_redux.internalStateActions.updateAppState)({
      appState: {
        query: {
          language: 'kuery',
          query: ''
        },
        columns: [],
        dataSource: {
          type: _data_sources.DataSourceType.DataView,
          dataViewId
        }
      }
    }));
  };
  const clearTimeFieldFromSort = (sort, timeFieldName) => {
    if (!Array.isArray(sort) || !timeFieldName) return sort;
    const filteredSort = sort.filter(([field]) => field !== timeFieldName);
    return filteredSort;
  };
  const transitionFromDataViewToESQL = dataView => {
    const appState = getCurrentTab().appState;
    const {
      query,
      sort
    } = appState;
    const filterQuery = query && (0, _esQuery.isOfQueryType)(query) ? query : undefined;
    const queryString = (0, _esqlUtils.getInitialESQLQuery)(dataView, true, filterQuery);
    const clearedSort = clearTimeFieldFromSort(sort, dataView === null || dataView === void 0 ? void 0 : dataView.timeFieldName);
    internalState.dispatch(injectCurrentTab(_redux.internalStateActions.updateAppState)({
      appState: {
        query: {
          esql: queryString
        },
        filters: [],
        dataSource: {
          type: _data_sources.DataSourceType.Esql
        },
        columns: [],
        sort: clearedSort
      }
    }));

    // clears pinned filters
    internalState.dispatch(injectCurrentTab(_redux.internalStateActions.updateGlobalState)({
      globalState: {
        filters: []
      }
    }));
  };
  const onDataViewCreated = async nextDataView => {
    if (!nextDataView.isPersisted()) {
      internalState.dispatch(_redux.internalStateActions.appendAdHocDataViews(nextDataView));
    } else {
      await internalState.dispatch(_redux.internalStateActions.loadDataViewList());
    }
    if (nextDataView.id) {
      await onChangeDataView(nextDataView);
    }
  };
  const onDataViewEdited = async editedDataView => {
    if (editedDataView.isPersisted()) {
      // Clear the current data view from the cache and create a new instance
      // of it, ensuring we have a new object reference to trigger a re-render
      services.dataViews.clearInstanceCache(editedDataView.id);
      setDataView(await services.dataViews.create(editedDataView.toSpec(), true));
    } else {
      await updateAdHocDataViewId(editedDataView);
    }
    void internalState.dispatch(_redux.internalStateActions.loadDataViewList());
    (0, _add_log.addLog)('[getDiscoverStateContainer] onDataViewEdited triggers data fetching');
    fetchData();
  };
  const getAppState = state => {
    return (0, _redux.selectTab)(state, tabId).appState;
  };
  const appState$ = (0, _rxjs.from)(internalState).pipe((0, _rxjs.map)(getAppState), (0, _rxjs.distinctUntilChanged)(_lodash.isEqual), (0, _rxjs.skip)(1));
  const appStateContainer = {
    get: () => getAppState(internalState.getState()),
    set: appState => {
      if (!appState) {
        return;
      }
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setAppState)({
        appState
      }));
    },
    state$: appState$
  };
  const getGlobalState = state => {
    const tabState = (0, _redux.selectTab)(state, tabId);
    const {
      timeRange: time,
      refreshInterval,
      filters
    } = tabState.globalState;
    return {
      time,
      refreshInterval,
      filters
    };
  };
  const globalState$ = (0, _rxjs.from)(internalState).pipe((0, _rxjs.map)(getGlobalState), (0, _rxjs.distinctUntilChanged)(_lodash.isEqual), (0, _rxjs.skip)(1));
  const globalStateContainer = {
    get: () => getGlobalState(internalState.getState()),
    set: state => {
      if (!state) {
        return;
      }
      const {
        time: timeRange,
        refreshInterval,
        filters
      } = state;
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setGlobalState)({
        globalState: {
          timeRange,
          refreshInterval,
          filters
        }
      }));
    },
    state$: globalState$
  };
  const initializeAndSyncUrlState = () => {
    const currentSavedSearch = savedSearchContainer.getState();
    (0, _add_log.addLog)('[appState] initialize state and sync with URL', currentSavedSearch);

    // Set the default profile state only if not loading a saved search,
    // to avoid overwriting saved search state
    if (!currentSavedSearch.id) {
      const {
        breakdownField,
        columns,
        rowHeight,
        hideChart
      } = (0, _cleanup_url_state.getCurrentUrlState)(stateStorage, services);

      // Only set default state which is not already set in the URL
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.setResetDefaultProfileState)({
        resetDefaultProfileState: {
          columns: columns === undefined,
          rowHeight: rowHeight === undefined,
          breakdownField: breakdownField === undefined,
          hideChart: hideChart === undefined
        }
      }));
    }
    const {
      data
    } = services;
    const savedSearchDataView = currentSavedSearch.searchSource.getField('index');
    const appState = appStateContainer.get();
    const setDataViewFromSavedSearch = !appState.dataSource || (0, _data_sources.isDataSourceType)(appState.dataSource, _data_sources.DataSourceType.DataView) && appState.dataSource.dataViewId !== (savedSearchDataView === null || savedSearchDataView === void 0 ? void 0 : savedSearchDataView.id);
    if (setDataViewFromSavedSearch) {
      // used data view is different from the given by url/state which is invalid
      internalState.dispatch(injectCurrentTab(_redux.internalStateActions.updateAppState)({
        appState: {
          dataSource: savedSearchDataView !== null && savedSearchDataView !== void 0 && savedSearchDataView.id ? (0, _data_sources.createDataViewDataSource)({
            dataViewId: savedSearchDataView.id
          }) : undefined
        }
      }));
    }

    // syncs `_a` portion of url with query services
    const stopSyncingQueryAppStateWithStateContainer = (0, _public2.connectToQueryState)(data.query, appStateContainer, {
      filters: _esQuery.FilterStateStore.APP_STATE,
      query: true
    });
    const {
      start: startSyncingAppStateWithUrl,
      stop: stopSyncingAppStateWithUrl
    } = (0, _public.syncState)({
      storageKey: _common.APP_STATE_URL_KEY,
      stateContainer: appStateContainer,
      stateStorage
    });

    // syncs `_g` portion of url with query services
    const stopSyncingQueryGlobalStateWithStateContainer = (0, _public2.connectToQueryState)(data.query, globalStateContainer, {
      refreshInterval: true,
      time: true,
      filters: _esQuery.FilterStateStore.GLOBAL_STATE
    });
    const {
      start: startSyncingGlobalStateWithUrl,
      stop: stopSyncingGlobalStateWithUrl
    } = (0, _public.syncState)({
      storageKey: _constants.GLOBAL_STATE_URL_KEY,
      stateContainer: globalStateContainer,
      stateStorage
    });

    // current state needs to be pushed to url
    internalState.dispatch(injectCurrentTab(_redux.internalStateActions.pushCurrentTabStateToUrl)()).then(() => {
      startSyncingAppStateWithUrl();
      startSyncingGlobalStateWithUrl();
    });
    return () => {
      stopSyncingQueryAppStateWithStateContainer();
      stopSyncingQueryGlobalStateWithStateContainer();
      stopSyncingAppStateWithUrl();
      stopSyncingGlobalStateWithUrl();
    };
  };
  let internalStopSyncing = () => {};
  const stopSyncing = () => {
    internalStopSyncing();
    internalStopSyncing = () => {};
  };

  /**
   * state containers initializing and subscribing to changes triggering e.g. data fetching
   */
  const initializeAndSync = () => {
    const syncLocallyPersistedTabState = () => internalState.dispatch(injectCurrentTab(_redux.internalStateActions.syncLocallyPersistedTabState)());

    // This needs to be the first thing that's wired up because initializeAndSyncUrlState is pulling the current state from the URL which
    // might change the time filter and thus needs to re-check whether the saved search has changed.
    const timefilerUnsubscribe = (0, _rxjs.merge)(services.timefilter.getTimeUpdate$(), services.timefilter.getRefreshIntervalUpdate$()).subscribe(() => {
      savedSearchContainer.updateTimeRange();
      syncLocallyPersistedTabState();
    });

    // Enable/disable kbn url tracking (That's the URL used when selecting Discover in the side menu)
    const unsubscribeSavedSearchUrlTracking = savedSearchContainer.initUrlTracking();

    // initialize syncing with _g and _a part of the URL
    const unsubscribeUrlState = initializeAndSyncUrlState();

    // subscribing to state changes of appStateContainer, triggering data fetching
    const appStateSubscription = appStateContainer.state$.subscribe((0, _build_state_subscribe.buildStateSubscribe)({
      savedSearchState: savedSearchContainer,
      dataState: dataStateContainer,
      internalState,
      runtimeStateManager,
      services,
      setDataView,
      getCurrentTab
    }));
    const savedSearchChangesSubscription = savedSearchContainer.getCurrent$().subscribe(syncLocallyPersistedTabState);

    // start subscribing to dataStateContainer, triggering data fetching
    const unsubscribeData = dataStateContainer.subscribe();

    // updates saved search when query or filters change, triggers data fetching
    const filterUnsubscribe = (0, _rxjs.merge)(services.filterManager.getFetches$()).subscribe(() => {
      const {
        currentDataView$
      } = (0, _redux.selectTabRuntimeState)(runtimeStateManager, tabId);
      savedSearchContainer.update({
        nextDataView: currentDataView$.getValue(),
        nextState: getCurrentTab().appState,
        useFilterAndQueryServices: true
      });
      (0, _add_log.addLog)('[getDiscoverStateContainer] filter changes triggers data fetching');
      fetchData();
    });
    services.data.search.session.enableStorage(createSearchSessionRestorationDataProvider({
      data: services.data,
      getCurrentTab,
      getSavedSearch: () => savedSearchContainer.getState()
    }), {
      isDisabled: () => services.capabilities.discover_v2.storeSearchSession ? {
        disabled: false
      } : {
        disabled: true,
        reasonText: _public2.noSearchSessionStorageCapabilityMessage
      }
    });
    internalStopSyncing = () => {
      savedSearchChangesSubscription.unsubscribe();
      unsubscribeData();
      appStateSubscription.unsubscribe();
      unsubscribeUrlState();
      unsubscribeSavedSearchUrlTracking();
      filterUnsubscribe.unsubscribe();
      timefilerUnsubscribe.unsubscribe();
    };
  };
  const createAndAppendAdHocDataView = async dataViewSpec => {
    var _newDataView$fields$g;
    const newDataView = await services.dataViews.create(dataViewSpec);
    if (((_newDataView$fields$g = newDataView.fields.getByName('@timestamp')) === null || _newDataView$fields$g === void 0 ? void 0 : _newDataView$fields$g.type) === 'date') {
      newDataView.timeFieldName = '@timestamp';
    }
    internalState.dispatch(_redux.internalStateActions.appendAdHocDataViews(newDataView));
    await onChangeDataView(newDataView);
    return newDataView;
  };
  const trackQueryFields = query => {
    const {
      scopedEbtManager$
    } = (0, _redux.selectTabRuntimeState)(runtimeStateManager, tabId);
    const scopedEbtManager = scopedEbtManager$.getValue();
    const {
      fieldsMetadata
    } = services;
    scopedEbtManager.trackSubmittingQuery({
      query,
      fieldsMetadata
    });
  };

  /**
   * Triggered when a user submits a query in the search bar
   */
  const onUpdateQuery = (payload, isUpdate) => {
    trackQueryFields(payload.query);
    if (isUpdate === false) {
      // remove the search session if the given query is not just updated
      searchSessionManager.removeSearchSessionIdFromURL({
        replace: false
      });
      (0, _add_log.addLog)('[getDiscoverStateContainer] onUpdateQuery triggers data fetching');
      dataStateContainer.fetch();
    }
  };

  /**
   * Function e.g. triggered when user changes data view in the sidebar
   */
  const onChangeDataView = async dataViewId => {
    await (0, _change_data_view.changeDataView)({
      dataViewId,
      services,
      internalState,
      runtimeStateManager,
      injectCurrentTab,
      getCurrentTab
    });
  };
  const fetchData = (initial = false) => {
    (0, _add_log.addLog)('fetchData', {
      initial
    });
    if (!initial || dataStateContainer.getInitialFetchStatus() === _types.FetchStatus.LOADING) {
      dataStateContainer.fetch();
    }
  };
  const updateESQLQuery = queryOrUpdater => {
    (0, _add_log.addLog)('updateESQLQuery');
    const {
      query: currentQuery
    } = getCurrentTab().appState;
    if (!(0, _esQuery.isOfAggregateQueryType)(currentQuery)) {
      throw new Error('Cannot update a non-ES|QL query. Make sure this function is only called once in ES|QL mode.');
    }
    const queryUpdater = (0, _lodash.isFunction)(queryOrUpdater) ? queryOrUpdater : () => queryOrUpdater;
    const query = {
      esql: queryUpdater(currentQuery.esql)
    };
    internalState.dispatch(injectCurrentTab(_redux.internalStateActions.updateAppState)({
      appState: {
        query
      }
    }));
  };
  return {
    appState$,
    internalState,
    internalStateActions: _redux.internalStateActions,
    injectCurrentTab,
    getCurrentTab,
    runtimeStateManager,
    dataState: dataStateContainer,
    savedSearchState: savedSearchContainer,
    stateStorage,
    searchSessionManager,
    customizationContext,
    actions: {
      initializeAndSync,
      stopSyncing,
      fetchData,
      onChangeDataView,
      createAndAppendAdHocDataView,
      onDataViewCreated,
      onDataViewEdited,
      onOpenSavedSearch,
      transitionFromESQLToDataView,
      transitionFromDataViewToESQL,
      onUpdateQuery,
      setDataView,
      updateAdHocDataViewId,
      updateESQLQuery
    }
  };
}
function createSearchSessionRestorationDataProvider(deps) {
  const getSavedSearch = () => deps.getSavedSearch();
  return {
    getName: async () => {
      const savedSearch = deps.getSavedSearch();
      return savedSearch.id && savedSearch.title || _i18n.i18n.translate('discover.discoverDefaultSearchSessionName', {
        defaultMessage: 'Discover'
      });
    },
    getLocatorData: async () => {
      return {
        id: _common.DISCOVER_APP_LOCATOR,
        initialState: createUrlGeneratorState({
          ...deps,
          getSavedSearch,
          shouldRestoreSearchSession: false
        }),
        restoreState: createUrlGeneratorState({
          ...deps,
          getSavedSearch,
          shouldRestoreSearchSession: true
        })
      };
    }
  };
}
function createUrlGeneratorState({
  data,
  getCurrentTab,
  getSavedSearch,
  shouldRestoreSearchSession
}) {
  var _appState$hideChart;
  const appState = getCurrentTab().appState;
  const dataView = getSavedSearch().searchSource.getField('index');
  return {
    filters: data.query.filterManager.getFilters(),
    dataViewId: dataView === null || dataView === void 0 ? void 0 : dataView.id,
    query: appState.query,
    savedSearchId: getSavedSearch().id,
    timeRange: shouldRestoreSearchSession ? data.query.timefilter.timefilter.getAbsoluteTime() : data.query.timefilter.timefilter.getTime(),
    searchSessionId: shouldRestoreSearchSession ? data.search.session.getSessionId() : undefined,
    columns: appState.columns,
    grid: appState.grid,
    sort: appState.sort,
    savedQuery: appState.savedQuery,
    interval: appState.interval,
    refreshInterval: shouldRestoreSearchSession ? {
      pause: true,
      // force pause refresh interval when restoring a session
      value: 0
    } : undefined,
    useHash: false,
    viewMode: appState.viewMode,
    hideAggregatedPreview: appState.hideAggregatedPreview,
    breakdownField: appState.breakdownField,
    dataViewSpec: !(dataView !== null && dataView !== void 0 && dataView.isPersisted()) ? dataView === null || dataView === void 0 ? void 0 : dataView.toMinimalSpec() : undefined,
    ...(shouldRestoreSearchSession ? {
      hideChart: (_appState$hideChart = appState.hideChart) !== null && _appState$hideChart !== void 0 ? _appState$hideChart : false,
      sampleSize: appState.sampleSize
    } : {})
  };
}