"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStreamsHandler = void 0;
var _error_handler = require("../error_handler");
var _get_metering_stats = require("../../utils/get_metering_stats");
var _errors = require("../../errors");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getDataStreamsHandler = dataUsageContext => {
  const logger = dataUsageContext.logFactory.get('dataStreamsRoute');
  return async (context, request, response) => {
    const {
      includeZeroStorage
    } = request.query;
    logger.debug('Retrieving user data streams');
    try {
      const core = await context.core;
      const {
        datastreams: meteringStatsDataStreams
      } = await (0, _get_metering_stats.getMeteringStats)(core.elasticsearch.client.asSecondaryAuthUser);
      if (!meteringStatsDataStreams || !meteringStatsDataStreams.length) {
        return (0, _error_handler.errorHandler)(logger, response, new _utils.CustomHttpRequestError('No data streams found', 404));
      }
      const nonSystemDataStreams = meteringStatsDataStreams.filter(dataStream => {
        var _dataStream$name;
        return !((_dataStream$name = dataStream.name) !== null && _dataStream$name !== void 0 && _dataStream$name.startsWith('.'));
      });
      if (!nonSystemDataStreams || !nonSystemDataStreams.length) {
        return (0, _error_handler.errorHandler)(logger, response, new _utils.CustomHttpRequestError('No user defined data streams found', 404));
      }
      const body = nonSystemDataStreams.reduce((acc, stat) => {
        if (includeZeroStorage || stat.size_in_bytes > 0) {
          acc.push({
            name: stat.name,
            storageSizeBytes: stat.size_in_bytes
          });
        }
        return acc;
      }, []).sort((a, b) => b.storageSizeBytes - a.storageSizeBytes);
      if (!body || !body.length) {
        return (0, _error_handler.errorHandler)(logger, response, new _utils.CustomHttpRequestError('No relevant user defined data streams found with storage size greater than zero', 404));
      }
      return response.ok({
        body
      });
    } catch (error) {
      if (error.message.includes('security_exception')) {
        return (0, _error_handler.errorHandler)(logger, response, new _errors.NoPrivilegeMeteringError());
      } else if (error.message.includes('index_not_found_exception')) {
        return (0, _error_handler.errorHandler)(logger, response, new _errors.NoIndicesMeteringError());
      }
      return (0, _error_handler.errorHandler)(logger, response, error);
    }
  };
};
exports.getDataStreamsHandler = getDataStreamsHandler;