"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeControlGroupStates = mergeControlGroupStates;
var _esqlTypes = require("@kbn/esql-types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Merges initial control group state with incoming control group state from embeddable packages.
 * Only adds controls from the incoming state that don't already exist based on variable names.
 *
 * @param initialControlGroupInput - The existing control group state from the dashboard
 * @param incomingControlGroup - The incoming embeddable package containing control group state
 * @returns The merged control group state, or undefined if no state is available
 */
function mergeControlGroupStates(initialControlGroupInput, incomingControlGroup) {
  var _incomingControlGroup;
  const incomingControlGroupState = incomingControlGroup === null || incomingControlGroup === void 0 ? void 0 : (_incomingControlGroup = incomingControlGroup.serializedState) === null || _incomingControlGroup === void 0 ? void 0 : _incomingControlGroup.rawState;
  let mergedControlGroupState = initialControlGroupInput;
  if (mergedControlGroupState && incomingControlGroupState) {
    // check if the control exists already
    const uniqueControls = [];
    const existingControlVariableNames = new Set(mergedControlGroupState.controls.map(control => {
      if ((0, _esqlTypes.controlHasVariableName)(control.controlConfig)) {
        return control.controlConfig.variableName;
      }
    }));

    // Checks each incoming control's variable name against existing controls to avoid duplicates
    incomingControlGroupState.controls.forEach(control => {
      var _control$controlConfi;
      if ((0, _esqlTypes.controlHasVariableName)(control.controlConfig) && !existingControlVariableNames.has((_control$controlConfi = control.controlConfig) === null || _control$controlConfi === void 0 ? void 0 : _control$controlConfi.variableName)) {
        uniqueControls.push(control);
      }
    });
    mergedControlGroupState = {
      ...mergedControlGroupState,
      controls: [...uniqueControls, ...mergedControlGroupState.controls]
    };
  } else if (!mergedControlGroupState && incomingControlGroupState) {
    mergedControlGroupState = incomingControlGroupState;
  }
  return mergedControlGroupState;
}