"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateDashboardTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _onechatServer = require("@kbn/onechat-server");
var _common = require("../../../common");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateDashboardSchema = _zod.z.object({
  id: _zod.z.string().describe('The ID of the dashboard to update.'),
  title: _zod.z.string().optional().describe('The updated title of the dashboard.'),
  description: _zod.z.string().optional().describe('The updated description of the dashboard.'),
  panels: _zod.z.array(_zod.z.unknown()).describe('An array of panel configurations (PanelJSON or lens_tool_artifact) to update.'),
  markdownContent: _zod.z.string().describe('Markdown content for a summary panel displayed at the top of the dashboard. This tool replaces the existing markdown summary with this content.')
});
const updateDashboardTool = (dashboard, savedObjects, {
  dashboardLocator,
  spaces
}) => {
  return {
    id: _common.dashboardTools.updateDashboard,
    type: _onechatCommon.ToolType.builtin,
    availability: {
      cacheMode: 'space',
      handler: _utils.checkDashboardToolsAvailability
    },
    description: `Update an existing dashboard by replacing its markdown summary and visualization panels.

This tool will:
1. Accept a dashboard ID, required fields (panels, markdownContent), and optional fields (title, description)
2. Build a markdown summary panel at the top
3. Replace the dashboard's panels with the markdown summary panel followed by the provided visualization panels
4. Return the updated dashboard information`,
    schema: updateDashboardSchema,
    tags: [],
    handler: async ({
      id,
      title,
      description,
      panels,
      markdownContent
    }, {
      logger,
      request,
      esClient
    }) => {
      try {
        var _spaces$spacesService, _updateData$descripti, _updateData$panels$le, _updateData$panels;
        const coreContext = {
          savedObjects: {
            client: savedObjects.getScopedClient(request)
          }
        };

        // Create a minimal request handler context
        const requestHandlerContext = {
          core: Promise.resolve(coreContext),
          resolve: async () => ({
            core: coreContext
          })
        };

        // First, read the existing dashboard to get current values
        const existingDashboard = await dashboard.client.read(requestHandlerContext, id);
        const markdownPanel = (0, _utils.buildMarkdownPanel)(markdownContent);
        const yOffset = markdownPanel.grid.h;
        const normalizedPanels = (0, _utils.normalizePanels)(panels, yOffset);
        const updatedPanels = [markdownPanel, ...normalizedPanels];

        // Merge existing data with provided updates. Dashboard update is a full replace, so we
        // must start from the existing dashboard state and then apply changes.
        const updateData = {
          ...existingDashboard.data,
          title: title !== null && title !== void 0 ? title : existingDashboard.data.title,
          description: description !== null && description !== void 0 ? description : existingDashboard.data.description,
          panels: updatedPanels
        };

        // Update dashboard using the Dashboard plugin's client
        const dashboardUpdateResponse = await dashboard.client.update(requestHandlerContext, id, {
          data: updateData
        });
        logger.info(`Dashboard updated successfully: ${dashboardUpdateResponse.id}`);
        const spaceId = spaces === null || spaces === void 0 ? void 0 : (_spaces$spacesService = spaces.spacesService) === null || _spaces$spacesService === void 0 ? void 0 : _spaces$spacesService.getSpaceId(request);
        const dashboardUrl = await (dashboardLocator === null || dashboardLocator === void 0 ? void 0 : dashboardLocator.getRedirectUrl({
          dashboardId: dashboardUpdateResponse.id
        }, {
          spaceId
        }));
        return {
          results: [{
            type: _tool_result.ToolResultType.dashboard,
            tool_result_id: (0, _onechatServer.getToolResultId)(),
            data: {
              id: dashboardUpdateResponse.id,
              title: updateData.title,
              content: {
                url: dashboardUrl,
                description: (_updateData$descripti = updateData.description) !== null && _updateData$descripti !== void 0 ? _updateData$descripti : '',
                panelCount: (_updateData$panels$le = (_updateData$panels = updateData.panels) === null || _updateData$panels === void 0 ? void 0 : _updateData$panels.length) !== null && _updateData$panels$le !== void 0 ? _updateData$panels$le : 0
              }
            }
          }]
        };
      } catch (error) {
        logger.error(`Error in update_dashboard tool: ${error.message}`);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to update dashboard: ${error.message}`,
              metadata: {
                id,
                title,
                description
              }
            }
          }]
        };
      }
    }
  };
};
exports.updateDashboardTool = updateDashboardTool;