"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useNavigation = void 0;
var _react = require("react");
var _get_initial_active_items = require("../utils/get_initial_active_items");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Hook for managing the main navigation state.
 *
 * @param isCollapsed - whether the side nav is collapsed.
 * @param items - the navigation structure including primary, secondary, and footer items.
 * @param logoId - the logo ID, used for highlighting the logo.
 * @param activeItemId - the active item ID, used for highlighting the active item.
 * @returns the navigation state including:
 * - `actualActiveItemId` - the actual active item ID. There can only be one `aria-current=page` link on the page.
 * - `visuallyActivePageId` - the visually active page ID. The link does not have to be `aria-current=page`, it can be a parent of an active page.
 * - `visuallyActiveSubpageId` - the visually active subpage ID.
 * - `openerNode` - the primary menu item whose submenu is shown in the side panel.
 * - `isCollapsed` - whether the side nav is collapsed.
 * - `isSidePanelOpen` - whether the side panel is open.
 */
const useNavigation = (isCollapsed, items, logoId, activeItemId) => {
  const {
    primaryItem,
    secondaryItem,
    isLogoActive
  } = (0, _react.useMemo)(() => (0, _get_initial_active_items.getActiveItems)(items, activeItemId, logoId), [items, activeItemId, logoId]);
  const actualActiveItemId = activeItemId;
  const visuallyActivePageId = isLogoActive ? logoId : primaryItem === null || primaryItem === void 0 ? void 0 : primaryItem.id;
  const visuallyActiveSubpageId = secondaryItem === null || secondaryItem === void 0 ? void 0 : secondaryItem.id;
  const openerNode = primaryItem;
  const isSidePanelOpen = !isCollapsed && !!(openerNode !== null && openerNode !== void 0 && openerNode.sections);
  const state = {
    actualActiveItemId,
    visuallyActivePageId,
    visuallyActiveSubpageId,
    openerNode,
    isCollapsed,
    isSidePanelOpen
  };
  return state;
};
exports.useNavigation = useNavigation;