"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchGraph = void 0;
var _v = require("@kbn/cloud-security-posture-common/types/graph/v1");
var _v2 = require("@kbn/cloud-security-posture-common/schema/graph/v1");
var _helpers = require("@kbn/cloud-security-posture-common/utils/helpers");
var _constants = require("@kbn/cloud-security-posture-common/constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fetchGraph = async ({
  esClient,
  logger,
  start,
  end,
  originEventIds,
  showUnknownTarget,
  indexPatterns,
  spaceId,
  esQuery
}) => {
  const originAlertIds = originEventIds.filter(originEventId => originEventId.isAlert);

  // FROM clause currently doesn't support parameters, Therefore, we validate the index patterns to prevent injection attacks.
  // Regex to match invalid characters in index patterns: upper case characters, \, /, ?, ", <, >, |, (space), #, or ,
  indexPatterns.forEach((indexPattern, idx) => {
    if (!_v2.INDEX_PATTERN_REGEX.test(indexPattern)) {
      throw new Error(`Invalid index pattern [${indexPattern}] at index ${idx}. Cannot contain characters \\, /, ?, ", <, >, |, (space character), #, or ,`);
    }
  });
  const isEnrichPolicyExists = await checkEnrichPolicyExists(esClient, logger, spaceId);
  const SECURITY_ALERTS_PARTIAL_IDENTIFIER = '.alerts-security.alerts-';
  const alertsMappingsIncluded = indexPatterns.some(indexPattern => indexPattern.includes(SECURITY_ALERTS_PARTIAL_IDENTIFIER));
  const query = buildEsqlQuery({
    indexPatterns,
    originEventIds,
    originAlertIds,
    isEnrichPolicyExists,
    spaceId,
    alertsMappingsIncluded
  });
  logger.trace(`Executing query [${query}]`);
  const eventIds = originEventIds.map(originEventId => originEventId.id);
  return await esClient.asCurrentUser.helpers.esql({
    columnar: false,
    filter: buildDslFilter(eventIds, showUnknownTarget, start, end, esQuery),
    query,
    // @ts-ignore - types are not up to date
    params: [...originEventIds.map((originEventId, idx) => ({
      [`og_id${idx}`]: originEventId.id
    })), ...originEventIds.filter(originEventId => originEventId.isAlert).map((originEventId, idx) => ({
      [`og_alrt_id${idx}`]: originEventId.id
    }))]
  }).toRecords();
};
exports.fetchGraph = fetchGraph;
const buildDslFilter = (eventIds, showUnknownTarget, start, end, esQuery) => {
  var _esQuery$bool$filter, _esQuery$bool$must, _esQuery$bool$should, _esQuery$bool$must_no;
  return {
    bool: {
      filter: [{
        range: {
          '@timestamp': {
            gte: start,
            lte: end
          }
        }
      }, ...(showUnknownTarget ? [] : [{
        bool: {
          should: _constants.GRAPH_TARGET_ENTITY_FIELDS.map(field => ({
            exists: {
              field
            }
          })),
          minimum_should_match: 1
        }
      }]), {
        bool: {
          should: [...(esQuery !== null && esQuery !== void 0 && (_esQuery$bool$filter = esQuery.bool.filter) !== null && _esQuery$bool$filter !== void 0 && _esQuery$bool$filter.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$must = esQuery.bool.must) !== null && _esQuery$bool$must !== void 0 && _esQuery$bool$must.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$should = esQuery.bool.should) !== null && _esQuery$bool$should !== void 0 && _esQuery$bool$should.length || esQuery !== null && esQuery !== void 0 && (_esQuery$bool$must_no = esQuery.bool.must_not) !== null && _esQuery$bool$must_no !== void 0 && _esQuery$bool$must_no.length ? [esQuery] : []), {
            terms: {
              'event.id': eventIds
            }
          }],
          minimum_should_match: 1
        }
      }]
    }
  };
};
const checkEnrichPolicyExists = async (esClient, logger, spaceId) => {
  try {
    const {
      policies
    } = await esClient.asInternalUser.enrich.getPolicy({
      name: (0, _helpers.getEnrichPolicyId)(spaceId)
    });
    return policies.some(policy => {
      var _policy$config$match;
      return ((_policy$config$match = policy.config.match) === null || _policy$config$match === void 0 ? void 0 : _policy$config$match.name) === (0, _helpers.getEnrichPolicyId)(spaceId);
    });
  } catch (error) {
    logger.error(`Error fetching enrich policy ${error.message}`);
    logger.error(error);
    return false;
  }
};
const buildEsqlQuery = ({
  indexPatterns,
  originEventIds,
  originAlertIds,
  isEnrichPolicyExists,
  spaceId,
  alertsMappingsIncluded
}) => {
  const SECURITY_ALERTS_PARTIAL_IDENTIFIER = '.alerts-security.alerts-';
  const enrichPolicyName = (0, _helpers.getEnrichPolicyId)(spaceId);
  const actorFieldsCoalesce = _constants.GRAPH_ACTOR_ENTITY_FIELDS.join(',\n    ');

  // Generate target entity ID collection logic
  // All fields use the same pattern: only append if not null
  // This ensures we filter out null values and only collect actual target entity IDs
  const targetEntityIdEvals = [
  // Initialize targetEntityId as null
  '| EVAL targetEntityId = TO_STRING(null)',
  // For each target field, append if not null
  ..._constants.GRAPH_TARGET_ENTITY_FIELDS.map(field => {
    return `| EVAL targetEntityId = CASE(
    ${field} IS NULL,
    targetEntityId,
    CASE(
      targetEntityId IS NULL,
      ${field},
      MV_DEDUPE(MV_APPEND(targetEntityId, ${field}))
    )
  )`;
  })].join('\n');

  // Generate actor and target field hint CASE statements
  const actorFieldHintCases = (0, _utils.generateFieldHintCases)(_constants.GRAPH_ACTOR_ENTITY_FIELDS, 'actorEntityId');
  const targetFieldHintCases = (0, _utils.generateFieldHintCases)(_constants.GRAPH_TARGET_ENTITY_FIELDS, 'targetEntityId');
  const query = `FROM ${indexPatterns.filter(indexPattern => indexPattern.length > 0).join(',')} METADATA _id, _index
| EVAL actorEntityId = COALESCE(
    ${actorFieldsCoalesce}
  )
| WHERE event.action IS NOT NULL AND actorEntityId IS NOT NULL
| EVAL actorEntityId = COALESCE(
    ${actorFieldsCoalesce}
  )
${targetEntityIdEvals}
| MV_EXPAND actorEntityId
| MV_EXPAND targetEntityId
| EVAL actorEntityFieldHint = CASE(
${actorFieldHintCases},
    ""
  )
| EVAL targetEntityFieldHint = CASE(
${targetFieldHintCases},
    ""
)
${isEnrichPolicyExists ? `

| ENRICH ${enrichPolicyName} ON actorEntityId WITH actorEntityName = entity.name, actorEntityType = entity.type, actorEntitySubType = entity.sub_type, actorHostIp = host.ip
| ENRICH ${enrichPolicyName} ON targetEntityId WITH targetEntityName = entity.name, targetEntityType = entity.type, targetEntitySubType = entity.sub_type, targetHostIp = host.ip

// Construct actor and target entities data
// Build entity field conditionally - only include fields that have values
| EVAL actorEntityField = CASE(
    actorEntityName IS NOT NULL OR actorEntityType IS NOT NULL OR actorEntitySubType IS NOT NULL,
    CONCAT(",\\"entity\\":", "{",
      ${(0, _utils.concatPropIfExists)('name', 'actorEntityName', false)},
      ${(0, _utils.concatPropIfExists)('type', 'actorEntityType')},
      ${(0, _utils.concatPropIfExists)('sub_type', 'actorEntitySubType')},
      CASE(
        actorHostIp IS NOT NULL,
        CONCAT(",\\"host\\":", "{", "\\"ip\\":\\"", TO_STRING(actorHostIp), "\\"", "}"),
        ""
      ),
      ",\\"availableInEntityStore\\":true",
      ",\\"ecsParentField\\":\\"", actorEntityFieldHint, "\\"",
    "}"),
    CONCAT(",\\"entity\\":", "{",
      "\\"availableInEntityStore\\":false",
      ",\\"ecsParentField\\":\\"", actorEntityFieldHint, "\\"",
    "}")
  )
| EVAL targetEntityField = CASE(
    targetEntityName IS NOT NULL OR targetEntityType IS NOT NULL OR targetEntitySubType IS NOT NULL,
    CONCAT(",\\"entity\\":", "{",
      ${(0, _utils.concatPropIfExists)('name', 'targetEntityName', false)},
      ${(0, _utils.concatPropIfExists)('type', 'targetEntityType')},
      ${(0, _utils.concatPropIfExists)('sub_type', 'targetEntitySubType')},
      CASE(
        targetHostIp IS NOT NULL,
        CONCAT(",\\"host\\":", "{", "\\"ip\\":\\"", TO_STRING(targetHostIp), "\\"", "}"),
        ""
      ),
      ",\\"availableInEntityStore\\":true",
      ",\\"ecsParentField\\":\\"", targetEntityFieldHint, "\\"",
    "}"),
    CONCAT(",\\"entity\\":", "{",
      "\\"availableInEntityStore\\":false",
      ",\\"ecsParentField\\":\\"", targetEntityFieldHint, "\\"",
    "}")
  )
` : `
| EVAL actorEntityField = CONCAT(",\\"entity\\":", "{",
    "\\"availableInEntityStore\\":false",
    ",\\"ecsParentField\\":\\"", actorEntityFieldHint, "\\"",
  "}")
| EVAL targetEntityField = CONCAT(",\\"entity\\":", "{",
    "\\"availableInEntityStore\\":false",
    ",\\"ecsParentField\\":\\"", targetEntityFieldHint, "\\"",
  "}")
// Fallback to null string with non-enriched entity metadata
| EVAL actorEntityName = TO_STRING(null)
| EVAL actorEntityType = TO_STRING(null)
| EVAL actorEntitySubType = TO_STRING(null)
| EVAL actorHostIp = TO_STRING(null)
| EVAL targetEntityName = TO_STRING(null)
| EVAL targetEntityType = TO_STRING(null)
| EVAL targetEntitySubType = TO_STRING(null)
| EVAL targetHostIp = TO_STRING(null)
`}
// Create actor and target data with entity data

| EVAL actorDocData = CONCAT("{",
    "\\"id\\":\\"", actorEntityId, "\\"",
    ",\\"type\\":\\"", "${_v2.DOCUMENT_TYPE_ENTITY}", "\\"",
    actorEntityField,
  "}")
| EVAL targetDocData = CONCAT("{",
    "\\"id\\":\\"", COALESCE(targetEntityId, ""), "\\"",
    ",\\"type\\":\\"", "${_v2.DOCUMENT_TYPE_ENTITY}", "\\"",
    targetEntityField,
  "}")

// Map host and source values to enriched contextual data
| EVAL sourceIps = source.ip
| EVAL sourceCountryCodes = source.geo.country_iso_code
// Origin event and alerts allow us to identify the start position of graph traversal
| EVAL isOrigin = ${originEventIds.length > 0 ? `event.id in (${originEventIds.map((_id, idx) => `?og_id${idx}`).join(', ')})` : 'false'}
| EVAL isOriginAlert = isOrigin AND ${originAlertIds.length > 0 ? `event.id in (${originAlertIds.map((_id, idx) => `?og_alrt_id${idx}`).join(', ')})` : 'false'}
| EVAL isAlert = _index LIKE "*${SECURITY_ALERTS_PARTIAL_IDENTIFIER}*"
// Aggregate document's data for popover expansion and metadata enhancements
// We format it as JSON string, the best alternative so far. Tried to use tuple using MV_APPEND
// but it flattens the data and we lose the structure
| EVAL docType = CASE (isAlert, "${_v.DOCUMENT_TYPE_ALERT}", "${_v.DOCUMENT_TYPE_EVENT}")
| EVAL docData = CONCAT("{",
    "\\"id\\":\\"", _id, "\\"",
    CASE (event.id IS NOT NULL AND event.id != "", CONCAT(",\\"event\\":","{","\\"id\\":\\"", event.id, "\\"","}"), ""),
    ",\\"type\\":\\"", docType, "\\"",
    ",\\"index\\":\\"", _index, "\\"",
    ${
  // ESQL complains about missing field's mapping when we don't fetch from alerts index
  alertsMappingsIncluded ? `CASE (isAlert, CONCAT(",\\"alert\\":", "{",
      "\\"ruleName\\":\\"", kibana.alert.rule.name, "\\"",
    "}"), ""),` : ''}
  "}")
| STATS badge = COUNT(*),
  uniqueEventsCount = COUNT_DISTINCT(CASE(isAlert == false, _id, null)),
  uniqueAlertsCount = COUNT_DISTINCT(CASE(isAlert == true, _id, null)),
  isAlert = MV_MAX(VALUES(isAlert)),
  docs = VALUES(docData),
  sourceIps = MV_DEDUPE(VALUES(sourceIps)),
  sourceCountryCodes = MV_DEDUPE(VALUES(sourceCountryCodes)),
  // actor attributes
  actorNodeId = CASE(
    // deterministic group IDs - use raw entity ID for single values, MD5 hash for multiple
    MV_COUNT(VALUES(actorEntityId)) == 1, TO_STRING(VALUES(actorEntityId)),
    MD5(MV_CONCAT(MV_SORT(VALUES(actorEntityId)), ","))
  ),
  actorIdsCount = COUNT_DISTINCT(actorEntityId),
  actorEntityName = VALUES(actorEntityName),
  actorHostIp = VALUES(actorHostIp),
  actorsDocData = VALUES(actorDocData),
  // target attributes
  targetNodeId = CASE(
    // deterministic group IDs - use raw entity ID for single values, MD5 hash for multiple
    COUNT_DISTINCT(targetEntityId) == 0, null,
    CASE(
      MV_COUNT(VALUES(targetEntityId)) == 1, TO_STRING(VALUES(targetEntityId)),
      MD5(MV_CONCAT(MV_SORT(VALUES(targetEntityId)), ","))
    )
  ),
  targetIdsCount = COUNT_DISTINCT(targetEntityId),
  targetEntityName = VALUES(targetEntityName),
  targetHostIp = VALUES(targetHostIp),
  targetsDocData = VALUES(targetDocData)
    BY action = event.action,
      actorEntityType,
      actorEntitySubType,
      targetEntityType,
      targetEntitySubType,
      isOrigin,
      isOriginAlert
| LIMIT 1000
| SORT action DESC, isOrigin`;
  return query;
};