"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createCloudUsageCollector = createCloudUsageCollector;
exports.registerCloudUsageCollector = registerCloudUsageCollector;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createCloudUsageCollector(usageCollection, config) {
  const {
    isCloudEnabled,
    trialEndDate,
    isElasticStaffOwned,
    organizationId,
    deploymentId,
    projectId,
    projectType,
    productTier,
    orchestratorTarget,
    organizationInTrial
  } = config;
  const trialEndDateMs = trialEndDate ? new Date(trialEndDate).getTime() : undefined;
  return usageCollection.makeUsageCollector({
    type: 'cloud',
    isReady: () => true,
    schema: {
      isCloudEnabled: {
        type: 'boolean',
        _meta: {
          description: 'Is the deployment running in Elastic Cloud (ESS or Serverless)?'
        }
      },
      trialEndDate: {
        type: 'date',
        _meta: {
          description: 'End of the trial period'
        }
      },
      inTrial: {
        type: 'boolean',
        _meta: {
          description: 'Is the organization during the trial period?'
        }
      },
      isElasticStaffOwned: {
        type: 'boolean',
        _meta: {
          description: 'Is the deploymend owned by an Elastician'
        }
      },
      organizationId: {
        type: 'keyword',
        _meta: {
          description: 'The Elastic Cloud Organization ID that owns the deployment/project'
        }
      },
      deploymentId: {
        type: 'keyword',
        _meta: {
          description: 'The ESS Deployment ID'
        }
      },
      projectId: {
        type: 'keyword',
        _meta: {
          description: 'The Serverless Project ID'
        }
      },
      projectType: {
        type: 'keyword',
        _meta: {
          description: 'The Serverless Project type'
        }
      },
      productTier: {
        type: 'keyword',
        _meta: {
          description: 'The Serverless Product Tier'
        }
      },
      orchestratorTarget: {
        type: 'keyword',
        _meta: {
          description: 'The Orchestrator Target where it is deployed (canary/non-canary)'
        }
      }
    },
    fetch: () => {
      return {
        isCloudEnabled,
        isElasticStaffOwned,
        organizationId,
        trialEndDate,
        ...(organizationInTrial ? {
          inTrial: true
        } : trialEndDateMs ? {
          inTrial: Date.now() <= trialEndDateMs
        } : {}),
        deploymentId,
        projectId,
        projectType,
        productTier,
        orchestratorTarget
      };
    }
  });
}
function registerCloudUsageCollector(usageCollection, config) {
  if (!usageCollection) {
    return;
  }
  const collector = createCloudUsageCollector(usageCollection, config);
  usageCollection.registerCollector(collector);
}