"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAuthenticateRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _axios = _interopRequireDefault(require("axios"));
var _cloud_connect_client = require("../services/cloud_connect_client");
var _cluster_info = require("../lib/cluster_info");
var _create_storage_service = require("../lib/create_storage_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bodySchema = _configSchema.schema.object({
  apiKey: _configSchema.schema.string({
    minLength: 1
  })
});
const registerAuthenticateRoute = ({
  router,
  logger,
  getStartServices,
  hasEncryptedSOEnabled,
  cloudApiUrl
}) => {
  router.get({
    path: '/internal/cloud_connect/config',
    security: {
      authz: {
        enabled: false,
        reason: 'This route returns public configuration information.'
      }
    },
    validate: false,
    options: {
      access: 'internal'
    }
  }, async (context, request, response) => {
    try {
      const coreContext = await context.core;
      const esClient = coreContext.elasticsearch.client.asCurrentUser;

      // Fetch cluster and license information from Elasticsearch
      const [licenseResponse, clusterResponse] = await Promise.all([esClient.license.get(), esClient.info()]);
      return response.ok({
        body: {
          hasEncryptedSOEnabled,
          license: {
            type: licenseResponse.license.type,
            uid: licenseResponse.license.uid
          },
          cluster: {
            id: clusterResponse.cluster_uuid,
            name: clusterResponse.cluster_name,
            version: clusterResponse.version.number
          }
        }
      });
    } catch (error) {
      logger.error('Failed to fetch cluster and license information', {
        error
      });

      // Return partial data if ES calls fail
      return response.ok({
        body: {
          hasEncryptedSOEnabled
        }
      });
    }
  });
  router.post({
    path: '/internal/cloud_connect/authenticate',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authentication to the Cloud Connect API and handles authorization there.'
      }
    },
    validate: {
      body: bodySchema
    },
    options: {
      access: 'internal'
    }
  }, async (context, request, response) => {
    const {
      apiKey
    } = request.body;
    try {
      const cloudConnectClient = new _cloud_connect_client.CloudConnectClient(logger, cloudApiUrl);
      const coreContext = await context.core;

      // Initialize storage service for saving the API key
      const storageService = await (0, _create_storage_service.createStorageService)(context, getStartServices, logger);

      // Step 1: Validate the API key scope
      const validationResult = await cloudConnectClient.validateApiKeyScope(apiKey);
      if (!validationResult.hasValidScope) {
        const errorMessage = validationResult.errorMessage || 'Invalid API key';
        logger.warn(`API key validation failed: ${errorMessage}`);
        return response.badRequest({
          body: {
            message: errorMessage
          }
        });
      }

      // Step 2: Onboard the cluster based on the key type
      let onboardingResponse;

      // Cluster-scoped key: use existing key directly
      if (validationResult.isClusterScoped) {
        const clusterDetails = await cloudConnectClient.getClusterDetails(apiKey, validationResult.clusterId);

        // Use the fetched cluster details to onboard
        const clusterData = {
          self_managed_cluster: clusterDetails.self_managed_cluster,
          license: clusterDetails.license
        };
        onboardingResponse = await cloudConnectClient.onboardCluster(apiKey, clusterData);

        // Store the API key after successful onboarding
        await storageService.saveApiKey(apiKey, onboardingResponse.id);
      } else {
        // Generate a new cluster-scoped API key
        const esClient = coreContext.elasticsearch.client.asCurrentUser;
        const clusterData = await (0, _cluster_info.getCurrentClusterData)(esClient);

        // Use cluster details to onboard
        onboardingResponse = await cloudConnectClient.onboardClusterWithKeyGeneration(apiKey, clusterData);

        // Store the generated API key after successful onboarding
        await storageService.saveApiKey(onboardingResponse.key, onboardingResponse.id);
      }
      logger.info(`Cluster onboarded successfully: ${onboardingResponse.id} (org: ${onboardingResponse.metadata.organization_id})`);
      return response.ok({
        body: {
          success: true,
          cluster_id: onboardingResponse.id,
          organization_id: onboardingResponse.metadata.organization_id,
          message: 'Cluster authenticated and onboarded successfully'
        }
      });
    } catch (error) {
      logger.error('Failed to authenticate with Cloud Connect', {
        error
      });
      if (_axios.default.isAxiosError(error)) {
        var _error$response, _error$response2;
        const status = (_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.status;
        const errorData = (_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : _error$response2.data;
        if (status === 401) {
          return response.unauthorized({
            body: {
              message: 'Invalid or expired API key'
            }
          });
        }
        if (status === 403) {
          return response.forbidden({
            body: {
              message: 'Terms and Conditions not accepted or no cloud organization found. Please complete the setup in Elastic Cloud.'
            }
          });
        }
        if (status === 400) {
          return response.badRequest({
            body: {
              message: (errorData === null || errorData === void 0 ? void 0 : errorData.message) || 'Invalid request'
            }
          });
        }
      }
      return response.customError({
        statusCode: 500,
        body: {
          message: 'An error occurred while authenticating with Cloud Connect'
        }
      });
    }
  });
};
exports.registerAuthenticateRoute = registerAuthenticateRoute;