"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getClusterInfo = getClusterInfo;
exports.getCurrentClusterData = getCurrentClusterData;
exports.getLicenseInfo = getLicenseInfo;
var _uuid = require("uuid");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches cluster information from Elasticsearch
 * @param esClient - Elasticsearch client
 * @returns Cluster info including UUID, name, and version
 */
async function getClusterInfo(esClient) {
  try {
    return await esClient.info();
  } catch (error) {
    throw new Error(`Failed to fetch cluster information: ${error instanceof Error ? error.message : 'Unknown error'}`);
  }
}

/**
 * Fetches license information from Elasticsearch
 * @param esClient - Elasticsearch client
 * @returns License info including type and UID
 */
async function getLicenseInfo(esClient) {
  try {
    return await esClient.license.get();
  } catch (error) {
    throw new Error(`Failed to fetch license information: ${error instanceof Error ? error.message : 'Unknown error'}`);
  }
}

/**
 * Builds cluster data object for Cloud Connect onboarding
 * Fetches cluster info and license from Elasticsearch and formats them for the API
 * @param esClient - Elasticsearch client
 * @returns Formatted cluster data ready for Cloud Connect API
 */
async function getCurrentClusterData(esClient) {
  var _licenseInfo$license, _licenseInfo$license2;
  const [clusterInfo, licenseInfo] = await Promise.all([getClusterInfo(esClient), getLicenseInfo(esClient)]);
  const clusterData = {
    self_managed_cluster: {
      id: clusterInfo.cluster_uuid,
      name: clusterInfo.cluster_name || (0, _uuid.v4)(),
      version: clusterInfo.version.number
    },
    license: {
      type: ((_licenseInfo$license = licenseInfo.license) === null || _licenseInfo$license === void 0 ? void 0 : _licenseInfo$license.type) || 'trial',
      uid: ((_licenseInfo$license2 = licenseInfo.license) === null || _licenseInfo$license2 === void 0 ? void 0 : _licenseInfo$license2.uid) || ''
    }
  };
  return clusterData;
}