"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AutomaticImportSavedObjectService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AutomaticImportSavedObjectService {
  constructor(logger, savedObjectsClient) {
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.logger = logger.get('savedObjectsService');
    this.savedObjectsClient = savedObjectsClient;
  }

  /**
   * Helper function to parse and increment a semantic version string (x.y.z)
   * @param currentVersion - Current semantic version string (e.g., "1.0.0")
   * @param incrementType - Optional: Which part to increment: 'major' | 'minor' | 'patch'. Defaults to 'patch'.
   * @returns Incremented semantic version string
   */
  incrementSemanticVersion(currentVersion, incrementType = 'patch') {
    if (!currentVersion) {
      return '0.0.0';
    }
    const versionParts = currentVersion.split('.');
    if (versionParts.length !== 3) {
      throw new Error('Invalid version format');
    }
    let [major, minor, patch] = versionParts.map(v => parseInt(v, 10));
    if (isNaN(major) || isNaN(minor) || isNaN(patch)) {
      return '0.0.0';
    }
    switch (incrementType) {
      case 'major':
        major += 1;
        minor = 0;
        patch = 0;
        break;
      case 'minor':
        minor += 1;
        patch = 0;
        break;
      case 'patch':
      default:
        patch += 1;
        break;
    }
    return `${major}.${minor}.${patch}`;
  }

  /**
   * Integration Operations
   */

  async insertIntegration(integrationParams, authenticatedUser) {
    const {
      integrationId
    } = integrationParams;
    if (!integrationId) {
      throw new Error('Integration ID is required');
    }
    try {
      this.logger.debug(`Creating integration: ${integrationId}`);
      const initialIntegrationData = {
        integration_id: integrationId,
        data_stream_count: 0,
        status: _constants.TASK_STATUSES.pending,
        created_by: authenticatedUser.username,
        metadata: {
          title: integrationParams.title,
          description: integrationParams.description,
          logo: integrationParams.logo,
          created_at: new Date().toISOString(),
          version: '0.0.0'
        }
      };
      return await this.savedObjectsClient.create(_constants.INTEGRATION_SAVED_OBJECT_TYPE, initialIntegrationData, {
        // overwrite id that is default generated by the saved objects service
        id: integrationId
      });
    } catch (error) {
      // Create will throw a confict if integration ID already exists.
      if (_server.SavedObjectsErrorHelpers.isConflictError(error)) {
        throw new Error(`Integration ${integrationId} already exists`);
      }
      this.logger.error(`Failed to create integration: ${error}`);
      throw error;
    }
  }

  /**
   * Create or update an integration
   * @param data - The integration data. Must include an integration_id.
   * @param expectedVersion - The expected version for optimistic concurrency control at the application layer. Required to ensure data consistency.
   * @param versionUpdate - Optional: specify which version part to increment ('major' | 'minor' | 'patch'). Defaults to incrementing 'patch'.
   * @param options - The options for the update.
   * @returns The saved object
   */
  async updateIntegration(data, expectedVersion, versionUpdate, options) {
    const {
      integration_id: integrationId,
      data_stream_count: dataStreamCount = 0,
      status
    } = data;
    if (!integrationId) {
      throw new Error('Integration ID is required');
    }
    try {
      var _existingIntegration$;
      this.logger.debug(`Updating integration: ${integrationId}`);
      const existingIntegration = await this.getIntegration(integrationId);
      if (!existingIntegration) {
        throw new Error(`Integration ${integrationId} not found`);
      }
      const currentVersion = ((_existingIntegration$ = existingIntegration.metadata) === null || _existingIntegration$ === void 0 ? void 0 : _existingIntegration$.version) || '0.0.0';
      if (currentVersion !== expectedVersion) {
        throw new Error(`Version conflict: Integration ${integrationId} has been updated. Expected version ${expectedVersion}, but current version is ${currentVersion}. Please fetch the latest version and try again.`);
      }
      const newVersion = this.incrementSemanticVersion(currentVersion, versionUpdate);
      const integrationData = {
        integration_id: integrationId,
        data_stream_count: dataStreamCount,
        created_by: existingIntegration.created_by,
        status: status || existingIntegration.status,
        metadata: {
          ...existingIntegration.metadata,
          version: newVersion,
          title: data.metadata.title,
          description: data.metadata.description
        }
      };
      return await this.savedObjectsClient.update(_constants.INTEGRATION_SAVED_OBJECT_TYPE, integrationId, integrationData, options);
    } catch (error) {
      this.logger.error(`Failed to update integration: ${error}`);
      throw error;
    }
  }

  /**
   * Get an integration by ID
   * @param integrationId - The ID of the integration
   * @returns The integration
   */
  async getIntegration(integrationId) {
    try {
      this.logger.debug(`Getting integration: ${integrationId}`);
      const integrationResponse = await this.savedObjectsClient.get(_constants.INTEGRATION_SAVED_OBJECT_TYPE, integrationId);
      return integrationResponse.attributes;
    } catch (error) {
      this.logger.error(`Failed to get integration ${integrationId}: ${error}`);
      throw error;
    }
  }

  /**
   * @returns All integrations
   */
  async getAllIntegrations() {
    try {
      this.logger.debug('Getting all integrations');
      const integrationsResponse = await this.savedObjectsClient.find({
        type: _constants.INTEGRATION_SAVED_OBJECT_TYPE
      });
      return integrationsResponse.saved_objects.map(integration => integration.attributes);
    } catch (error) {
      // Return empty array if index doesn't exist yet
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        this.logger.debug('No integrations index found, returning empty array');
        return [];
      }
      this.logger.error(`Failed to get all integrations: ${error}`);
      throw error;
    }
  }

  /**
   * Delete an integration by ID and cascade delete all associated data streams
   * @param integrationId - The ID of the integration
   * @param options - The options for the delete
   * @returns Object containing deletion results with success status and any errors
   */
  async deleteIntegration(integrationId, options) {
    this.logger.debug(`Starting cascade deletion for integration: ${integrationId}`);
    const deletionErrors = [];
    let dataStreamsDeleted = 0;
    try {
      // delete up to 100 data streams at a time
      const perPage = 100;
      let page = 1;
      let hasMore = true;
      const dataStreamsToDelete = [];
      while (hasMore) {
        try {
          const dataStreamsResponse = await this.savedObjectsClient.find({
            type: _constants.DATA_STREAM_SAVED_OBJECT_TYPE,
            filter: `${_constants.DATA_STREAM_SAVED_OBJECT_TYPE}.attributes.integration_id: ${JSON.stringify(integrationId)}`,
            page,
            perPage,
            fields: ['integration_id']
          });
          dataStreamsToDelete.push(...dataStreamsResponse.saved_objects.map(ds => ({
            id: ds.id,
            type: _constants.DATA_STREAM_SAVED_OBJECT_TYPE
          })));

          // If 100 saved objects are retrieved in a page, then we try to see if there are more
          hasMore = dataStreamsResponse.saved_objects.length === perPage;
          page++;
        } catch (error) {
          this.logger.error(`Failed to find data streams for integration ${integrationId} on page ${page}: ${error}`);
          hasMore = false;
        }
      }
      this.logger.debug(`Found ${dataStreamsToDelete.length} data streams to delete for integration ${integrationId}`);
      const CHUNK_SIZE = 50;
      if (dataStreamsToDelete.length > 0) {
        for (let i = 0; i < dataStreamsToDelete.length; i += CHUNK_SIZE) {
          const chunk = dataStreamsToDelete.slice(i, i + CHUNK_SIZE);
          try {
            const bulkDeleteResult = await this.savedObjectsClient.bulkDelete(chunk, {
              ...options,
              force: true
            });

            // Process results to track successes and failures
            bulkDeleteResult.statuses.forEach(status => {
              if (status.success) {
                dataStreamsDeleted++;
              } else {
                var _status$error;
                const errorMessage = ((_status$error = status.error) === null || _status$error === void 0 ? void 0 : _status$error.message) || 'Unknown error';
                this.logger.warn(`Failed to delete data stream ${status.id}: ${errorMessage}`);
                deletionErrors.push({
                  id: status.id,
                  error: errorMessage
                });
              }
            });
          } catch (bulkDeleteError) {
            // If bulk delete fails entirely, log and track errors for this chunk
            this.logger.error(`Bulk delete failed for chunk starting at index ${i}: ${bulkDeleteError}`);
            chunk.forEach(ds => {
              deletionErrors.push({
                id: ds.id,
                error: bulkDeleteError instanceof Error ? bulkDeleteError.message : String(bulkDeleteError)
              });
            });
          }
        }
        this.logger.debug(`Deleted ${dataStreamsDeleted} of ${dataStreamsToDelete.length} data streams for integration ${integrationId}`);
      }

      // Delete the integration if we successfully deleted all data streams OR the a force option is provided
      const allDataStreamsDeleted = deletionErrors.length === 0;
      if (!allDataStreamsDeleted && !(options !== null && options !== void 0 && options.force)) {
        throw new Error(`Cannot delete integration ${integrationId}: Failed to delete ${deletionErrors.length} data streams. ` + `Use force option to delete the integration anyway. Errors: ${JSON.stringify(deletionErrors)}`);
      }
      await this.savedObjectsClient.delete(_constants.INTEGRATION_SAVED_OBJECT_TYPE, integrationId, options);
      this.logger.info(`Successfully deleted integration ${integrationId} and ${dataStreamsDeleted} associated data streams`);
      return {
        success: true,
        dataStreamsDeleted,
        errors: deletionErrors
      };
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        this.logger.error(`Integration ${integrationId} not found`);
        throw error;
      }
      this.logger.error(`Failed to delete integration ${integrationId}: ${error}`);
      throw error;
    }
  }

  /**
   * Data Stream Operations
   */

  /**
   * Generate composite ID for data stream: integration_id-data_stream_id
   */
  getDataStreamCompositeId(integrationId, dataStreamId) {
    return `${integrationId}-${dataStreamId}`;
  }

  /**
   * Create a data stream
   * @param request - The Kibana request object
   * @param data - The data stream data. Must include an integration_id and data_stream_id.
   * @param options - The options for the create
   * @returns The created data stream
   */
  async insertDataStream(dataStreamParams, authenticatedUser, options) {
    const {
      integrationId,
      dataStreamId,
      jobInfo,
      title,
      description,
      inputTypes
    } = dataStreamParams;
    if (!integrationId) {
      throw new Error('Integration ID is required');
    }
    if (!dataStreamId) {
      throw new Error('Data stream ID is required');
    }
    try {
      var _jobInfo$jobId, _jobInfo$jobType, _jobInfo$status, _dataStreamParams$met, _dataStreamParams$met2;
      this.logger.debug(`Creating data stream: ${dataStreamId}`);
      const initialDataStreamData = {
        integration_id: integrationId,
        data_stream_id: dataStreamId,
        created_by: authenticatedUser.username,
        title,
        description,
        input_types: inputTypes.map(type => type.name),
        job_info: {
          job_id: (_jobInfo$jobId = jobInfo === null || jobInfo === void 0 ? void 0 : jobInfo.jobId) !== null && _jobInfo$jobId !== void 0 ? _jobInfo$jobId : '',
          job_type: (_jobInfo$jobType = jobInfo === null || jobInfo === void 0 ? void 0 : jobInfo.jobType) !== null && _jobInfo$jobType !== void 0 ? _jobInfo$jobType : '',
          status: (_jobInfo$status = jobInfo === null || jobInfo === void 0 ? void 0 : jobInfo.status) !== null && _jobInfo$status !== void 0 ? _jobInfo$status : _constants.TASK_STATUSES.pending
        },
        metadata: {
          sample_count: (_dataStreamParams$met = (_dataStreamParams$met2 = dataStreamParams.metadata) === null || _dataStreamParams$met2 === void 0 ? void 0 : _dataStreamParams$met2.sampleCount) !== null && _dataStreamParams$met !== void 0 ? _dataStreamParams$met : 0,
          created_at: new Date().toISOString(),
          version: '0.0.0'
        }
      };
      const compositeId = this.getDataStreamCompositeId(integrationId, dataStreamId);
      return await this.savedObjectsClient.create(_constants.DATA_STREAM_SAVED_OBJECT_TYPE, initialDataStreamData, {
        ...options,
        id: compositeId
      });
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isConflictError(error)) {
        throw new Error(`Data stream ${dataStreamId} already exists`);
      }
      this.logger.error(`Failed to create data stream: ${error}`);
      throw error;
    }
  }

  /**
   * Get a data stream by ID
   * @param dataStreamId - The ID of the data stream
   * @param integrationId - The ID of the integration
   * @returns The data stream
   */
  async getDataStream(dataStreamId, integrationId) {
    try {
      this.logger.debug(`Getting data stream: ${dataStreamId}`);
      const compositeId = this.getDataStreamCompositeId(integrationId, dataStreamId);
      return await this.savedObjectsClient.get(_constants.DATA_STREAM_SAVED_OBJECT_TYPE, compositeId);
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        throw new Error(`Data stream ${dataStreamId} not found`);
      }
      this.logger.error(`Failed to get data stream ${dataStreamId}: ${error}`);
      throw error;
    }
  }

  /**
   * Get all data streams
   * @returns All data streams
   */
  async getAllDataStreams(integrationId) {
    try {
      this.logger.debug('Getting all data streams');
      const dataStreamsResponse = await this.savedObjectsClient.find({
        type: _constants.DATA_STREAM_SAVED_OBJECT_TYPE,
        filter: `${_constants.DATA_STREAM_SAVED_OBJECT_TYPE}.attributes.integration_id: ${JSON.stringify(integrationId)}`
      });
      return dataStreamsResponse.saved_objects.map(dataStream => dataStream.attributes);
    } catch (error) {
      this.logger.error(`Failed to get all data streams: ${error}`);
      throw error;
    }
  }

  /**
   * Find all data streams by integration ID
   * @param integrationId - The ID of the integration
   * @returns All data streams for the integration
   */
  async findAllDataStreamsByIntegrationId(integrationId) {
    try {
      this.logger.debug(`Finding all data streams for integration: ${integrationId}`);
      return await this.savedObjectsClient.find({
        type: _constants.DATA_STREAM_SAVED_OBJECT_TYPE,
        filter: `${_constants.DATA_STREAM_SAVED_OBJECT_TYPE}.attributes.integration_id: ${JSON.stringify(integrationId)}`
      });
    } catch (error) {
      this.logger.error(`Failed to find all data streams for integration ${integrationId}: ${error}`);
      throw error;
    }
  }

  /**
   * Delete a data stream by ID
   * @param integrationId - The ID of the integration
   * @param dataStreamId - The ID of the data stream
   * @param authenticatedUser - The authenticated user
   * @param options - The options for the delete
   * @returns The deleted data stream
   */
  async deleteDataStream(integrationId, dataStreamId, options) {
    let parentIntegrationId;
    try {
      const dataStream = await this.getDataStream(dataStreamId, integrationId);
      parentIntegrationId = dataStream.attributes.integration_id;
      this.logger.debug(`Deleting data stream with id:${dataStreamId}`);
      const compositeId = this.getDataStreamCompositeId(integrationId, dataStreamId);
      await this.savedObjectsClient.delete(_constants.DATA_STREAM_SAVED_OBJECT_TYPE, compositeId, options);
    } catch (error) {
      this.logger.error(`Failed to delete data stream ${dataStreamId}: ${error}`);
      throw error;
    }

    // update the integration count since we are deleting a data stream
    try {
      var _parentIntegration$da, _parentIntegration$me;
      const parentIntegration = await this.getIntegration(parentIntegrationId);
      if (!parentIntegration) {
        throw new Error(`Integration associated with this data stream ${parentIntegrationId} not found`);
      }
      const updatedIntegrationData = {
        ...parentIntegration,
        data_stream_count: ((_parentIntegration$da = parentIntegration.data_stream_count) !== null && _parentIntegration$da !== void 0 ? _parentIntegration$da : 0) - 1
      };
      await this.updateIntegration(updatedIntegrationData, ((_parentIntegration$me = parentIntegration.metadata) === null || _parentIntegration$me === void 0 ? void 0 : _parentIntegration$me.version) || '0.0.0');
    } catch (integrationError) {
      this.logger.error(`Failed to update integration ${parentIntegrationId} after deleting data stream ${dataStreamId}: ${integrationError}`);
    }
  }
}
exports.AutomaticImportSavedObjectService = AutomaticImportSavedObjectService;