"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTaskTool = void 0;
var _tools = require("@langchain/core/tools");
var _messages = require("@langchain/core/messages");
var _langgraph = require("@langchain/langgraph");
var _prebuilt = require("@langchain/langgraph/prebuilt");
var _zod = require("@kbn/zod");
var _prompts = require("../prompts");
var _state = require("../state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createTaskTool = params => {
  const {
    subagents,
    model,
    recursionLimit
  } = params;
  const agentsMap = new Map();
  for (const subagent of subagents) {
    // Create ReAct agent for the subagent
    const baseSubAgent = (0, _prebuilt.createReactAgent)({
      llm: model,
      tools: subagent.tools || [],
      messageModifier: subagent.prompt,
      stateSchema: _state.AutomaticImportAgentState
    });
    const ReActSubAgent = recursionLimit != null ? baseSubAgent.withConfig({
      recursionLimit
    }) : baseSubAgent;
    agentsMap.set(subagent.name, ReActSubAgent);
  }
  return (0, _tools.tool)(async (input, config) => {
    var _config$toolCall;
    const state = _state.AutomaticImportAgentState;
    const toolCallId = config === null || config === void 0 ? void 0 : (_config$toolCall = config.toolCall) === null || _config$toolCall === void 0 ? void 0 : _config$toolCall.id;
    const subAgent = agentsMap.get(input.subagent_name);
    const modifiedState = {
      ...state,
      messages: [{
        role: 'user',
        content: input.description
      }]
    };
    try {
      var _result$messages, _result$messages$slic;
      const result = await subAgent.invoke(modifiedState);
      return new _langgraph.Command({
        update: {
          ...result,
          messages: [new _messages.ToolMessage({
            content: (_result$messages = result.messages) === null || _result$messages === void 0 ? void 0 : (_result$messages$slic = _result$messages.slice(-1)[0]) === null || _result$messages$slic === void 0 ? void 0 : _result$messages$slic.content,
            tool_call_id: toolCallId
          })]
        }
      });
    } catch (e) {
      return new _langgraph.Command({
        update: {
          messages: [new _messages.ToolMessage({
            content: `Error executing task with ${input.subagent_name}: ${String(e)}`,
            tool_call_id: toolCallId
          })]
        }
      });
    }
  }, {
    name: 'task',
    description: _prompts.TASK_TOOL_DESCRIPTION.replace('{available_agents}', subagents.map(a => `- ${a.name}: ${a.description}`).join('\n')),
    schema: _zod.z.object({
      description: _zod.z.string().describe('The task to execute with the selected agent'),
      subagent_name: _zod.z.string().describe(`Name of the agent to use. Available: ${subagents.map(a => a.name).join(', ')}`)
    })
  });
};
exports.createTaskTool = createTaskTool;