"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_ITEMS_PER_PAGE = void 0;
exports.getApmTraceError = getApmTraceError;
exports.getTraceItems = getTraceItems;
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _lodash = require("lodash");
var _compact_map = require("../../utils/compact_map");
var _get_linked_children = require("../span_links/get_linked_children");
var _get_apm_trace_error_query = require("./get_apm_trace_error_query");
var _get_trace_docs_per_page = require("./get_trace_docs_per_page");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getTraceItems({
  traceId,
  config,
  apmEventClient,
  start,
  end,
  maxTraceItemsFromUrlParam,
  logger
}) {
  const maxTraceItems = maxTraceItemsFromUrlParam !== null && maxTraceItemsFromUrlParam !== void 0 ? maxTraceItemsFromUrlParam : config.ui.maxTraceItems;
  const errorResponsePromise = getApmTraceError({
    apmEventClient,
    traceId,
    start,
    end
  });
  const traceResponsePromise = getTraceDocsPaginated({
    apmEventClient,
    maxTraceItems,
    traceId,
    start,
    end,
    logger
  });
  const [errorDocs, traceResponse, spanLinksCountById] = await Promise.all([errorResponsePromise, traceResponsePromise, (0, _get_linked_children.getSpanLinksCountById)({
    traceId,
    apmEventClient,
    start,
    end
  })]);
  const traceDocsTotal = traceResponse.total;
  const exceedsMax = traceDocsTotal > maxTraceItems;
  const traceDocs = traceResponse.hits.map(({
    hit
  }) => hit);
  return {
    exceedsMax,
    traceDocs,
    errorDocs,
    spanLinksCountById,
    traceDocsTotal,
    maxTraceItems
  };
}
const MAX_ITEMS_PER_PAGE = exports.MAX_ITEMS_PER_PAGE = 10000; // 10000 is the max allowed by ES

async function getApmTraceError(params) {
  const response = await (0, _get_apm_trace_error_query.getApmTraceErrorQuery)(params);
  return (0, _compact_map.compactMap)(response.hits.hits, hit => {
    var _parent$id, _unflattened$span, _errorSource$error$ex, _errorSource$error$ex2, _errorSource$error, _errorSource$error$ex3;
    const errorSource = 'error' in hit._source ? hit._source : undefined;
    const event = hit.fields ? (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(_get_apm_trace_error_query.requiredFields) : undefined;
    if (!event) {
      return undefined;
    }
    const {
      _id: id,
      parent,
      error,
      ...unflattened
    } = event.unflatten();
    return {
      id,
      parent: {
        id: (_parent$id = parent === null || parent === void 0 ? void 0 : parent.id) !== null && _parent$id !== void 0 ? _parent$id : (_unflattened$span = unflattened.span) === null || _unflattened$span === void 0 ? void 0 : _unflattened$span.id
      },
      trace: unflattened.trace,
      span: unflattened.span,
      transaction: unflattened.transaction,
      timestamp: unflattened.timestamp,
      service: {
        name: unflattened.service.name
      },
      error: {
        exception: ((_errorSource$error$ex = errorSource === null || errorSource === void 0 ? void 0 : (_errorSource$error$ex2 = errorSource.error.exception) === null || _errorSource$error$ex2 === void 0 ? void 0 : _errorSource$error$ex2.length) !== null && _errorSource$error$ex !== void 0 ? _errorSource$error$ex : 0) > 0 ? errorSource === null || errorSource === void 0 ? void 0 : (_errorSource$error = errorSource.error) === null || _errorSource$error === void 0 ? void 0 : (_errorSource$error$ex3 = _errorSource$error.exception) === null || _errorSource$error$ex3 === void 0 ? void 0 : _errorSource$error$ex3[0] : error.exception,
        grouping_key: error === null || error === void 0 ? void 0 : error.grouping_key,
        culprit: error === null || error === void 0 ? void 0 : error.culprit,
        id: error === null || error === void 0 ? void 0 : error.id,
        log: errorSource === null || errorSource === void 0 ? void 0 : errorSource.error.log
      }
    };
  });
}
async function getTraceDocsPaginated({
  apmEventClient,
  maxTraceItems,
  traceId,
  start,
  end,
  hits = [],
  searchAfter,
  logger
}) {
  var _last;
  const response = await (0, _get_trace_docs_per_page.getTraceDocsPerPage)({
    apmEventClient,
    maxTraceItems,
    traceId,
    start,
    end,
    searchAfter
  });
  const mergedHits = [...hits, ...response.hits];
  logger.debug(`Paginating traces: retrieved: ${response.hits.length}, (total: ${mergedHits.length} of ${response.total}), maxTraceItems: ${maxTraceItems}`);
  if (mergedHits.length >= maxTraceItems || mergedHits.length >= response.total || mergedHits.length === 0 || response.hits.length < MAX_ITEMS_PER_PAGE) {
    return {
      hits: mergedHits,
      total: response.total
    };
  }
  return getTraceDocsPaginated({
    apmEventClient,
    maxTraceItems,
    traceId,
    start,
    end,
    hits: mergedHits,
    searchAfter: (_last = (0, _lodash.last)(response.hits)) === null || _last === void 0 ? void 0 : _last.sort,
    logger
  });
}