"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TraceDataState = void 0;
exports.createColorLookupMap = createColorLookupMap;
exports.getClockSkew = getClockSkew;
exports.getColorByType = getColorByType;
exports.getLegends = getLegends;
exports.getRootItemOrFallback = getRootItemOrFallback;
exports.getTraceParentChildrenMap = getTraceParentChildrenMap;
exports.getTraceWaterfall = getTraceWaterfall;
exports.getTraceWaterfallDuration = getTraceWaterfallDuration;
exports.useTraceWaterfall = useTraceWaterfall;
var _eui = require("@elastic/eui");
var _i18n = require("@kbn/i18n");
var _react = require("react");
var _legend = require("../../../../common/waterfall/legend");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FALLBACK_WARNING = _i18n.i18n.translate('xpack.apm.traceWaterfallItem.warningMessage.fallbackWarning', {
  defaultMessage: 'The trace document is incomplete and not all spans have arrived yet. Try refreshing the page or adjusting the time range.'
});
const INSTRUMENTATION_WARNING = _i18n.i18n.translate('xpack.apm.traceWaterfallItem.euiCallOut.aDuplicatedSpanWasLabel', {
  defaultMessage: 'A duplicated span was detected. This indicates a problem with how your services have been instrumented, as span IDs are meant to be unique.'
});
function useTraceWaterfall({
  traceItems,
  isFiltered = false,
  errors,
  onErrorClick
}) {
  const waterfall = (0, _react.useMemo)(() => {
    try {
      const legends = getLegends(traceItems);
      const colorBy = getColorByType(legends);
      const colorMap = createColorLookupMap(legends);
      const traceParentChildrenMap = getTraceParentChildrenMap(traceItems, isFiltered);
      const {
        rootItem,
        traceState,
        orphans
      } = getRootItemOrFallback(traceParentChildrenMap, traceItems);
      const traceWaterfall = rootItem ? getTraceWaterfall({
        rootItem,
        parentChildMap: traceParentChildrenMap,
        orphans,
        colorMap,
        colorBy
      }) : [];
      const errorMarks = rootItem && errors ? getWaterfallErrorsMarks({
        errors,
        traceItems: traceWaterfall,
        rootItem,
        onErrorClick
      }) : [];
      return {
        rootItem,
        traceState,
        message: traceState !== TraceDataState.Full ? FALLBACK_WARNING : undefined,
        traceWaterfall,
        duration: getTraceWaterfallDuration(traceWaterfall),
        maxDepth: Math.max(...traceWaterfall.map(item => item.depth)),
        legends,
        colorBy,
        errorMarks
      };
    } catch (e) {
      return {
        traceState: TraceDataState.Invalid,
        message: INSTRUMENTATION_WARNING,
        traceWaterfall: [],
        legends: [],
        duration: 0,
        maxDepth: 0,
        colorBy: _legend.WaterfallLegendType.Type,
        errorMarks: []
      };
    }
  }, [traceItems, isFiltered, errors, onErrorClick]);
  return waterfall;
}
function getWaterfallErrorsMarks({
  errors,
  traceItems,
  rootItem,
  onErrorClick
}) {
  const rootTimestampUs = rootItem.timestampUs;
  const traceItemsByIdMap = new Map(traceItems.map(item => [item.id, item]));
  return errors.map(error => {
    var _error$parent, _error$parent2, _error$transaction, _error$span, _parent$color;
    const parent = (_error$parent = error.parent) !== null && _error$parent !== void 0 && _error$parent.id ? traceItemsByIdMap.get((_error$parent2 = error.parent) === null || _error$parent2 === void 0 ? void 0 : _error$parent2.id) : undefined;
    const docId = ((_error$transaction = error.transaction) === null || _error$transaction === void 0 ? void 0 : _error$transaction.id) || ((_error$span = error.span) === null || _error$span === void 0 ? void 0 : _error$span.id);
    return {
      type: 'errorMark',
      error,
      id: error.id,
      verticalLine: false,
      offset: error.timestamp.us - rootTimestampUs,
      skew: getClockSkew({
        itemTimestamp: error.timestamp.us,
        itemDuration: 0,
        parent
      }),
      serviceColor: (_parent$color = parent === null || parent === void 0 ? void 0 : parent.color) !== null && _parent$color !== void 0 ? _parent$color : '',
      onClick: onErrorClick && docId ? () => {
        onErrorClick({
          traceId: rootItem.traceId,
          docId,
          errorCount: 1,
          errorDocId: error.id
        });
      } : undefined
    };
  });
}
function getColorByType(legends) {
  let count = 0;
  for (const {
    type
  } of legends) {
    if (type === _legend.WaterfallLegendType.ServiceName) count++;
    if (count > 1) return _legend.WaterfallLegendType.ServiceName;
  }
  return _legend.WaterfallLegendType.Type;
}
function getLegends(traceItems) {
  const serviceNames = Array.from(new Set(traceItems.map(item => item.serviceName)));
  const types = Array.from(new Set(traceItems.map(item => {
    var _item$type;
    return (_item$type = item.type) !== null && _item$type !== void 0 ? _item$type : '';
  })));
  const palette = (0, _eui.euiPaletteColorBlind)({
    rotations: Math.ceil((serviceNames.length + types.length) / 10)
  });
  let colorIndex = 0;
  const legends = [];
  serviceNames.forEach(serviceName => {
    legends.push({
      type: _legend.WaterfallLegendType.ServiceName,
      value: serviceName,
      color: palette[colorIndex++]
    });
  });
  types.forEach(type => {
    legends.push({
      type: _legend.WaterfallLegendType.Type,
      value: type,
      color: palette[colorIndex++]
    });
  });
  return legends;
}
function createColorLookupMap(legends) {
  return new Map(legends.map(legend => [`${legend.type}:${legend.value}`, legend.color]));
}
function getTraceParentChildrenMap(traceItems, filteredTrace) {
  if (traceItems.length === 0) {
    return {};
  }
  const traceMap = traceItems.reduce((acc, item) => {
    if (item.parentId) {
      (acc[item.parentId] ??= []).push(item);
    } else {
      (acc.root ??= [])[0] = item;
    }
    return acc;
  }, {});

  // TODO: Electing a fallback root item could be delegated to the server. For now
  // As mapping parent->children spans is done clientside, the root election is done
  // clientside as well.
  if (filteredTrace && !traceMap.root) {
    const root = traceItems.slice(1).reduce((acc, span) => acc.timestampUs <= span.timestampUs || acc.id === span.parentId ? acc : span, traceItems[0]);
    traceMap.root = [root];
  }
  return traceMap;
}
let TraceDataState = exports.TraceDataState = /*#__PURE__*/function (TraceDataState) {
  TraceDataState["Full"] = "full";
  TraceDataState["Partial"] = "partial";
  TraceDataState["Empty"] = "empty";
  TraceDataState["Invalid"] = "invalid";
  return TraceDataState;
}({});
function getRootItemOrFallback(traceParentChildrenMap, traceItems) {
  var _traceParentChildrenM;
  if (traceItems.length === 0) {
    return {
      traceState: TraceDataState.Empty
    };
  }
  const rootItem = (_traceParentChildrenM = traceParentChildrenMap.root) === null || _traceParentChildrenM === void 0 ? void 0 : _traceParentChildrenM[0];
  const parentIds = new Set(traceItems.map(({
    id
  }) => id));
  // TODO: Reuse waterfall util methods where possible or if logic is the same
  const orphans = traceItems.filter(item =>
  // Root cannot be an orphan.
  item.id !== (rootItem === null || rootItem === void 0 ? void 0 : rootItem.id) && item.parentId && !parentIds.has(item.parentId));
  if (rootItem) {
    return {
      traceState: orphans.length === 0 ? TraceDataState.Full : TraceDataState.Partial,
      rootItem,
      orphans
    };
  }
  const [fallbackRootItem, ...remainingOrphans] = orphans;
  return {
    traceState: TraceDataState.Partial,
    rootItem: fallbackRootItem,
    orphans: remainingOrphans
  };
}

// TODO: Reuse waterfall util methods where possible or if logic is the same
function reparentOrphansToRoot(rootItem, parentChildMap, orphans) {
  // Some cases with orphans, the root item has no direct link or children, so this
  // might be not initialised. This assigns the array in case of undefined/null to the
  // map.
  const children = parentChildMap[rootItem.id] ??= [];
  children.push(...orphans.map(orphan => ({
    ...orphan,
    parentId: rootItem.id,
    isOrphan: true
  })));
}
function getTraceWaterfall({
  rootItem,
  parentChildMap,
  orphans,
  colorMap,
  colorBy
}) {
  const rootStartMicroseconds = rootItem.timestampUs;
  const visitor = new Set([rootItem.id]);
  reparentOrphansToRoot(rootItem, parentChildMap, orphans);
  function getTraceWaterfallItem(item, depth, parent) {
    var _item$type2, _parentChildMap$item$;
    const startMicroseconds = item.timestampUs;
    const color = colorBy === _legend.WaterfallLegendType.ServiceName && item.serviceName ? colorMap.get(`${_legend.WaterfallLegendType.ServiceName}:${item.serviceName}`) : colorMap.get(`${_legend.WaterfallLegendType.Type}:${(_item$type2 = item.type) !== null && _item$type2 !== void 0 ? _item$type2 : ''}`);
    const traceWaterfallItem = {
      ...item,
      depth,
      offset: startMicroseconds - rootStartMicroseconds,
      skew: getClockSkew({
        itemTimestamp: startMicroseconds,
        itemDuration: item.duration,
        parent
      }),
      color
    };
    const sortedChildren = ((_parentChildMap$item$ = parentChildMap[item.id]) === null || _parentChildMap$item$ === void 0 ? void 0 : _parentChildMap$item$.sort((a, b) => a.timestampUs - b.timestampUs)) || [];
    const flattenedChildren = sortedChildren.flatMap(child => {
      // Check if we have encountered the trace item before.
      // If we have visited the trace item before, then the child waterfall items are already
      // present in the flattened list, so we throw an error to alert the user of duplicated
      // spans. This should guard against circular or unusual links between spans.
      if (visitor.has(child.id)) {
        throw new Error('Duplicate span id detected');
      }

      // If we haven't visited it before, then we can process the waterfall item.
      visitor.add(child.id);
      return getTraceWaterfallItem(child, depth + 1, traceWaterfallItem);
    });
    return [traceWaterfallItem, ...flattenedChildren];
  }
  return getTraceWaterfallItem(rootItem, 0);
}
function getClockSkew({
  itemTimestamp,
  itemDuration,
  parent
}) {
  let skew = 0;
  if (parent) {
    const parentTimestamp = parent.timestampUs;
    const parentStart = parentTimestamp + parent.skew;
    const offsetStart = parentStart - itemTimestamp;
    if (offsetStart > 0) {
      const latency = Math.max(parent.duration - itemDuration, 0) / 2;
      skew = offsetStart + latency;
    }
  }
  return skew;
}
function getTraceWaterfallDuration(flattenedTraceWaterfall) {
  return Math.max(...flattenedTraceWaterfall.map(item => item.offset + item.skew + item.duration), 0);
}