"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPreferredBucketSizeAndDataSource = getPreferredBucketSizeAndDataSource;
exports.intervalToSeconds = intervalToSeconds;
var _common = require("@kbn/data-plugin/common");
var _lodash = require("lodash");
var _document_type = require("../document_type");
var _rollup = require("../rollup");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EVENT_PREFERENCE = [_document_type.ApmDocumentType.ServiceTransactionMetric, _document_type.ApmDocumentType.TransactionMetric, _document_type.ApmDocumentType.TransactionEvent];
function intervalToSeconds(rollupInterval) {
  if (rollupInterval === _rollup.RollupInterval.None) {
    return 0;
  }
  return (0, _common.parseInterval)(rollupInterval).asSeconds();
}
function getPreferredBucketSizeAndDataSource({
  sources,
  bucketSizeInSeconds
}) {
  let preferred;
  const sourcesWithDocs = sources.filter(source => source.hasDocs);
  const sourcesInPreferredOrder = (0, _lodash.orderBy)(sourcesWithDocs, [source => EVENT_PREFERENCE.indexOf(source.documentType), source => intervalToSeconds(source.rollupInterval)], ['asc', 'desc']);
  if (sourcesInPreferredOrder.length > 0) {
    const preferredDocumentType = sourcesInPreferredOrder[0].documentType;
    const sourcesFromPreferredDocumentType = sourcesInPreferredOrder.filter(source => source.documentType === preferredDocumentType);
    preferred = sourcesFromPreferredDocumentType.find(source => {
      const rollupIntervalInSeconds = intervalToSeconds(source.rollupInterval);
      return rollupIntervalInSeconds <= bucketSizeInSeconds;
    }) ||
    // pick 1m from available docs if we can't find a matching bucket size
    (0, _lodash.last)(sourcesFromPreferredDocumentType);
  }
  if (!preferred) {
    preferred = {
      documentType: _document_type.ApmDocumentType.TransactionEvent,
      rollupInterval: _rollup.RollupInterval.None,
      hasDurationSummaryField: false,
      hasDocs: true
    };
  }
  return {
    source: preferred,
    bucketSizeInSeconds: Math.max(bucketSizeInSeconds, intervalToSeconds(preferred.rollupInterval))
  };
}