"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformAndDeleteLegacyActions = void 0;
var _std = require("@kbn/std");
var _saved_objects = require("../../../saved_objects");
var _find_rules = require("../../../application/rule/methods/find/find_rules");
var _bulk_delete = require("../../../application/rule/methods/bulk_delete");
var _types = require("./types");
var _transform_legacy_actions = require("./transform_legacy_actions");
var _validate_actions = require("../validate_actions");
var _transform_raw_actions_to_domain_actions = require("../../../application/rule/transforms/transform_raw_actions_to_domain_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @deprecated Once we are confident all rules relying on side-car actions SO's have been migrated to SO references we should remove this function
 * retrieves legacy actions for SIEM rule and deletes associated sidecar SO
 * @param context RulesClient context
 * @param params.ruleId - id of rule to be migrated
 * @returns
 */
const transformAndDeleteLegacyActions = async (context, rules, skipActionsValidation) => {
  const {
    unsecuredSavedObjectsClient
  } = context;
  try {
    if (rules.length === 0) {
      return {};
    }

    /**
     * On update / patch I'm going to take the actions as they are, better off taking rules client.find (siem.notification) result
     * and putting that into the actions array of the rule, then set the rules onThrottle property, notifyWhen and throttle from null -> actual value (1hr etc..)
     * Then use the rules client to delete the siem.notification
     * Then with the legacy Rule Actions saved object type, just delete it.
     */
    // find it using the references array, not params.ruleAlertId
    const [siemNotifications, legacyRuleActionsSOs] = await Promise.all([(0, _find_rules.findRules)(context, {
      options: {
        filter: 'alert.attributes.alertTypeId:(siem.notifications)',
        hasReference: rules.map(rule => ({
          type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
          id: rule.id
        }))
      }
    }), unsecuredSavedObjectsClient.find({
      type: _types.legacyRuleActionsSavedObjectType,
      hasReference: rules.map(rule => ({
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
        id: rule.id
      }))
    })]);
    const siemNotificationsExist = siemNotifications != null && siemNotifications.data.length > 0;
    const legacyRuleNotificationSOsExist = legacyRuleActionsSOs != null && legacyRuleActionsSOs.saved_objects.length > 0;

    // Assumption: if no legacy sidecar SO or notification rule types exist
    // that reference the rule in question, assume rule actions are not legacy
    if (!siemNotificationsExist && !legacyRuleNotificationSOsExist) {
      return {};
    }
    const deleteLegacyActions = async () => {
      await Promise.all([
      // If the legacy notification rule type ("siem.notification") exist,
      // migration and cleanup are needed
      siemNotificationsExist && (0, _bulk_delete.bulkDeleteRules)(context, {
        ids: siemNotifications.data.map(rule => rule.id)
      }),
      // Delete the legacy sidecar SO if it exists
      legacyRuleNotificationSOsExist && unsecuredSavedObjectsClient.bulkDelete(legacyRuleActionsSOs.saved_objects.map(savedObject => ({
        type: _types.legacyRuleActionsSavedObjectType,
        id: savedObject.id
      })))]);
    };
    const transformedActionsByRuleId = {};
    await (0, _std.asyncForEach)(legacyRuleActionsSOs.saved_objects, async savedObject => {
      const ruleReference = savedObject.references.find(reference => reference.type === _saved_objects.RULE_SAVED_OBJECT_TYPE);
      if (ruleReference == null) {
        throw new Error('Failed to find rule reference on legacy action Saved Object');
      }

      // ruleThrottle can have special values 'no_actions' or 'rule', or a duration like e.g. '1h'
      // 'rule' means actions fire each time the rule runs, whereas a duration means the actions are
      // throttled to only fire once per the specified time duration
      const hasNoThrottledActions = savedObject.attributes.ruleThrottle === 'no_actions' || savedObject.attributes.ruleThrottle === 'rule';
      if (hasNoThrottledActions) {
        return;
      }
      const ruleId = ruleReference.id;
      const transformedActions = (0, _transform_legacy_actions.transformFromLegacyActions)(savedObject.attributes, savedObject.references);
      const transformedReferences = savedObject.references.filter(({
        type
      }) => type === 'action');
      if (!skipActionsValidation) {
        const rule = rules.find(r => r.id === ruleId);
        if (rule == null) {
          throw new Error(`Failed to find rule id: ${ruleId} for validating migrated actions`);
        }
        await validateTransformedActions({
          context,
          actions: transformedActions,
          actionsReferences: transformedReferences,
          rule
        });
      }
      transformedActionsByRuleId[ruleId] = {
        transformedActions,
        transformedReferences
      };
    });
    await deleteLegacyActions();
    return transformedActionsByRuleId;
  } catch (e) {
    context.logger.debug(`Migration has failed for SIEM rules: ${e.message}`);
    throw e;
  }
};
exports.transformAndDeleteLegacyActions = transformAndDeleteLegacyActions;
const validateTransformedActions = async ({
  context,
  actions,
  actionsReferences,
  rule
}) => {
  const ruleType = context.ruleTypeRegistry.get(rule.attributes.alertTypeId);
  await (0, _validate_actions.validateActions)(context, ruleType, {
    ...rule.attributes,
    // set to undefined to avoid both per-action and rule level values clashing
    throttle: undefined,
    notifyWhen: undefined,
    actions: (0, _transform_raw_actions_to_domain_actions.transformRawActionsToDomainActions)({
      ruleId: rule.id,
      actions,
      references: actionsReferences,
      isSystemAction: context.isSystemAction
    })
  });
};