"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkFillGapsByRuleIds = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _types = require("./types");
var _utils = require("./utils");
var _authorization = require("../../../../authorization");
var _batch_backfill_rule_gaps = require("./batch_backfill_rule_gaps");
var _lib = require("../../../../../common/lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_MAX_BACKFILL_CONCURRENCY = 10;
const bulkFillGapsByRuleIds = async (context, {
  rules,
  range
}, options) => {
  var _options$maxBackfillC;
  const errorString = (0, _lib.validateBackfillSchedule)(range.start, range.end);
  if (errorString) {
    throw _boom.default.badRequest(`Error validating backfill schedule parameters "${JSON.stringify(range)}" - ${errorString}`);
  }
  const errored = [];
  const skipped = [];
  const backfilled = [];
  const eventLogClient = await context.getEventLogClient();
  const maxBackfillConcurrency = Math.max((_options$maxBackfillC = options === null || options === void 0 ? void 0 : options.maxBackfillConcurrency) !== null && _options$maxBackfillC !== void 0 ? _options$maxBackfillC : DEFAULT_MAX_BACKFILL_CONCURRENCY, DEFAULT_MAX_BACKFILL_CONCURRENCY);

  // Make sure user has access to these rules
  const rulesByRuleType = (0, _lodash.mapValues)((0, _lodash.groupBy)(rules, rule => `${rule.alertTypeId}<>${rule.consumer}`), groupedRules => ({
    ruleTypeId: groupedRules[0].alertTypeId,
    consumer: groupedRules[0].consumer,
    rules: groupedRules
  }));
  const authorizedRules = [];
  for (const {
    ruleTypeId,
    consumer,
    rules: rulesBatch
  } of Object.values(rulesByRuleType)) {
    try {
      await context.authorization.ensureAuthorized({
        ruleTypeId,
        consumer,
        operation: _authorization.WriteOperations.FillGaps,
        entity: _authorization.AlertingAuthorizationEntity.Rule
      });
      authorizedRules.push(...rulesBatch);
    } catch (error) {
      rulesBatch.forEach(rule => {
        (0, _utils.logProcessedAsAuditEvent)(context, {
          id: rule.id,
          name: rule.name
        }, error);
        errored.push((0, _utils.toBulkGapFillError)(rule, _types.BulkGapsFillStep.ACCESS_VALIDATION, error));
        return;
      });
    }
  }
  await (0, _pMap.default)(authorizedRules, async rule => {
    const backfillResult = await (0, _batch_backfill_rule_gaps.batchBackfillRuleGaps)(context, {
      rule,
      range,
      maxGapCountPerRule: options.maxGapCountPerRule
    });
    switch (backfillResult.outcome) {
      case _types.BulkFillGapsScheduleResult.BACKFILLED:
        backfilled.push(rule);
        break;
      case _types.BulkFillGapsScheduleResult.ERRORED:
        errored.push(backfillResult.error);
        break;
      case _types.BulkFillGapsScheduleResult.SKIPPED:
        skipped.push(rule);
        break;
    }
  }, {
    concurrency: maxBackfillConcurrency
  });
  await eventLogClient.refreshIndex();
  return {
    backfilled,
    skipped,
    errored
  };
};
exports.bulkFillGapsByRuleIds = bulkFillGapsByRuleIds;