"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findGapAutoFillSchedulerLogs = findGapAutoFillSchedulerLogs;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _schemas = require("./schemas");
var _authorization = require("../../../../../authorization");
var _audit_events = require("../../../../../rules_client/common/audit_events");
var _saved_objects = require("../../../../../saved_objects");
var _utils = require("./utils");
var _utils2 = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function findGapAutoFillSchedulerLogs(context, params) {
  try {
    // Validate input parameters
    _schemas.findGapAutoFillSchedulerLogsParamsSchema.validate(params);
  } catch (error) {
    throw _boom.default.badRequest(`Error validating gap auto fill scheduler logs parameters "${JSON.stringify(params)}" - ${error.message}`);
  }
  try {
    var _context$auditLogger;
    // Get the scheduler saved object and perform authorization against the rule types it manages
    const schedulerSO = await (0, _utils2.getGapAutoFillSchedulerSO)({
      context,
      id: params.id,
      operation: _authorization.ReadOperations.FindGapAutoFillSchedulerLogs,
      authAuditAction: _audit_events.GapAutoFillSchedulerAuditAction.GET_LOGS
    });

    // Get the task ID from the scheduler saved object
    const taskId = schedulerSO.id;

    // Get event log client and query for gap-auto-fill-schedule events
    const eventLogClient = await context.getEventLogClient();
    const filters = ['event.action:gap-auto-fill-schedule'];
    if (params.statuses) {
      const statusFilters = `(${params.statuses.map(status => `kibana.gap_auto_fill.execution.status : ${status}`).join(' OR ')})`;
      filters.push(statusFilters);
    }
    const result = await eventLogClient.findEventsBySavedObjectIds('task', [taskId], {
      page: params.page,
      per_page: params.perPage,
      start: params.start,
      end: params.end,
      sort: [{
        sort_field: params.sortField,
        sort_order: params.sortDirection
      }],
      filter: filters.join(' AND ')
    });

    // Log successful get logs
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.gapAutoFillSchedulerAuditEvent)({
      action: _audit_events.GapAutoFillSchedulerAuditAction.GET_LOGS,
      savedObject: {
        type: _saved_objects.GAP_AUTO_FILL_SCHEDULER_SAVED_OBJECT_TYPE,
        id: params.id,
        name: schedulerSO.attributes.name
      }
    }));

    // Transform raw event log data into cleaner format
    const transformedData = result.data.map(_utils.formatGapAutoFillSchedulerLogEntry);
    return {
      data: transformedData,
      total: result.total,
      page: result.page,
      perPage: result.per_page
    };
  } catch (err) {
    const errorMessage = `Failed to get gap fill auto scheduler logs by id: ${params.id}`;
    context.logger.error(`${errorMessage} - ${err}`);
    throw _boom.default.boomify(err, {
      message: errorMessage
    });
  }
}