"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.productDocumentationTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _inferenceCommon = require("@kbn/inference-common");
var _onechatServer = require("@kbn/onechat-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const productDocumentationSchema = _zod.z.object({
  query: _zod.z.string().describe('Search query to retrieve documentation about Elastic products. Rewritten in English to best match the documentation content.'),
  product: _zod.z.enum(['kibana', 'elasticsearch', 'observability', 'security']).optional().describe('Product to filter by: "kibana", "elasticsearch", "observability", or "security"'),
  max: _zod.z.number().optional().default(3).describe('Maximum number of documents to return. Defaults to 3.')
});

// TODO make this configurable, we need a platform level setting for the embedding model
const inferenceId = _inferenceCommon.defaultInferenceEndpoints.ELSER;

// Path to GenAI Settings within the management app
const GENAI_SETTINGS_APP_PATH = '/app/management/ai/genAiSettings';
const productDocumentationTool = coreSetup => {
  // Create a closure that will resolve llmTasks when the handler is called
  const getLlmTasks = async () => {
    const [, plugins] = await coreSetup.getStartServices();
    return plugins.llmTasks;
  };

  // Check if product documentation is installed
  const isProductDocAvailable = async llmTasks => {
    try {
      var _await$llmTasks$retri;
      return (_await$llmTasks$retri = await llmTasks.retrieveDocumentationAvailable({
        inferenceId
      })) !== null && _await$llmTasks$retri !== void 0 ? _await$llmTasks$retri : false;
    } catch {
      return false;
    }
  };
  const baseTool = {
    id: _onechatCommon.platformCoreTools.productDocumentation,
    type: _onechatCommon.ToolType.builtin,
    description: `Search and retrieve documentation about Elastic products (Kibana, Elasticsearch, Elastic Security, Elastic Observability).`,
    schema: productDocumentationSchema,
    handler: async ({
      query,
      product,
      max = 3
    }, {
      modelProvider,
      logger,
      request
    }) => {
      const llmTasks = await getLlmTasks();
      if (!llmTasks) {
        return {
          results: [(0, _onechatServer.createErrorResult)({
            message: 'Product documentation tool is not available. LlmTasks plugin is not available.'
          })]
        };
      }

      // Check if product documentation is installed
      const isAvailable = await isProductDocAvailable(llmTasks);
      if (!isAvailable) {
        // Build the full settings URL using the request's base path (includes space prefix)
        const basePath = coreSetup.http.basePath.get(request);
        const settingsUrl = `${basePath}${GENAI_SETTINGS_APP_PATH}`;
        return {
          results: [(0, _onechatServer.createErrorResult)({
            message: `Product documentation is not installed. To use this tool, please install Elastic documentation from the GenAI Settings page: ${settingsUrl}. Do not perform any other tool calls, and provide the user with a link to install the documentation.`,
            metadata: {
              settingsUrl
            }
          })]
        };
      }
      try {
        // Get the default model to extract the connector
        const model = await modelProvider.getDefaultModel();
        const connector = model.connector;

        // Retrieve documentation
        const result = await llmTasks.retrieveDocumentation({
          searchTerm: query,
          products: product ? [product] : undefined,
          max,
          connectorId: connector.connectorId,
          request,
          inferenceId
        });
        if (!result.success || result.documents.length === 0) {
          return {
            results: [(0, _onechatServer.createErrorResult)({
              message: 'No documentation found for the given query.',
              metadata: {
                query,
                product: product || 'all'
              }
            })]
          };
        }

        // Return documentation results
        return {
          results: result.documents.map(doc => ({
            type: _onechatCommon.ToolResultType.other,
            data: {
              reference: {
                url: doc.url,
                title: doc.title
              },
              partial: doc.summarized,
              content: {
                title: doc.title,
                url: doc.url,
                content: doc.content,
                summarized: doc.summarized
              }
            }
          }))
        };
      } catch (error) {
        logger.error(`Error retrieving product documentation: ${error.message}`);
        return {
          results: [(0, _onechatServer.createErrorResult)({
            message: `Failed to retrieve product documentation: ${error.message}`
          })]
        };
      }
    },
    tags: [],
    // Tool is always available - handler will check if docs are installed and provide guidance
    availability: {
      cacheMode: 'global',
      handler: async () => {
        return {
          status: 'available'
        };
      }
    }
  };
  return baseTool;
};
exports.productDocumentationTool = productDocumentationTool;