"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getWorkflowExecutionStatusTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _prompts = require("@kbn/onechat-genai-utils/prompts");
var _workflows = require("@kbn/onechat-genai-utils/tools/utils/workflows");
var _results = require("@kbn/onechat-genai-utils/tools/utils/results");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getWorkflowExecutionStatusSchema = _zod.z.object({
  executionId: _zod.z.string().describe(`ID of the workflow execution (execution_id) to retrieve the status for.`)
});
const getWorkflowExecutionStatusTool = ({
  workflowsManagement
}) => {
  const {
    management: workflowApi
  } = workflowsManagement;
  return {
    id: _onechatCommon.platformCoreTools.getWorkflowExecutionStatus,
    type: _onechatCommon.ToolType.builtin,
    description: (0, _prompts.cleanPrompt)(`Retrieve the status of a workflow execution.

    If the workflow execution has completed, the final output will be returned. Otherwise, the execution status will be returned.

    **important**: do **NOT** call this tool directly after calling a workflow tool.
    Instead, if the workflow didn't complete, tell the user they can ask you to check the execution.
    `),
    schema: getWorkflowExecutionStatusSchema,
    handler: async ({
      executionId
    }, {
      spaceId
    }) => {
      const execution = await (0, _workflows.getExecutionState)({
        executionId,
        spaceId,
        workflowApi
      });
      if (execution) {
        return {
          results: [(0, _results.otherResult)({
            execution
          })]
        };
      } else {
        return {
          results: [(0, _results.errorResult)(`Workflow execution with ID '${executionId}' not found.`)]
        };
      }
    },
    tags: []
  };
};
exports.getWorkflowExecutionStatusTool = getWorkflowExecutionStatusTool;