import { ac as AgentCard, w as MessageSendParams, S as SendMessageResponse, B as Message, aw as Task, aO as TaskStatusUpdateEvent, aQ as TaskArtifactUpdateEvent, Z as TaskPushNotificationConfig, b as SetTaskPushNotificationConfigResponse, X as TaskIdParams, c as GetTaskPushNotificationConfigResponse, a5 as ListTaskPushNotificationConfigParams, j as ListTaskPushNotificationConfigResponse, a7 as DeleteTaskPushNotificationConfigParams, g as DeleteTaskPushNotificationConfigResponse, V as TaskQueryParams, G as GetTaskResponse, C as CancelTaskResponse, i as JSONRPCResponse, J as JSONRPCErrorResponse } from '../types-DNKcmF0f.cjs';

type A2AStreamEventData = Message | Task | TaskStatusUpdateEvent | TaskArtifactUpdateEvent;
interface A2AClientOptions {
    agentCardPath?: string;
    fetchImpl?: typeof fetch;
}
/**
 * A2AClient is a TypeScript HTTP client for interacting with A2A-compliant agents.
 */
declare class A2AClient {
    private agentCardPromise;
    private requestIdCounter;
    private serviceEndpointUrl?;
    private customFetchImpl?;
    /**
     * Constructs an A2AClient instance from an AgentCard.
     * @param agentCard The AgentCard object.
     * @param options Optional. The options for the A2AClient including the fetch/auth implementation.
     */
    constructor(agentCard: AgentCard | string, options?: A2AClientOptions);
    /**
     * Dynamically resolves the fetch implementation to use for requests.
     * Prefers a custom implementation if provided, otherwise falls back to the global fetch.
     * @returns The fetch implementation.
     * @param args Arguments to pass to the fetch implementation.
     * @throws If no fetch implementation is available.
     */
    private _fetch;
    /**
     * Creates an A2AClient instance by fetching the AgentCard from a URL then constructing the A2AClient.
     * @param agentCardUrl The URL of the agent card.
     * @param options Optional. The options for the A2AClient including the fetch/auth implementation.
     * @returns A Promise that resolves to a new A2AClient instance.
     */
    static fromCardUrl(agentCardUrl: string, options?: A2AClientOptions): Promise<A2AClient>;
    /**
     * Helper method to make a generic JSON-RPC POST request.
     * @param method The RPC method name.
     * @param params The parameters for the RPC method.
     * @returns A Promise that resolves to the RPC response.
     */
    private _postRpcRequest;
    /**
     * Internal helper method to fetch the RPC service endpoint.
     * @param url The URL to fetch.
     * @param rpcRequest The JSON-RPC request to send.
     * @param acceptHeader The Accept header to use.  Defaults to "application/json".
     * @returns A Promise that resolves to the fetch HTTP response.
     */
    private _fetchRpc;
    /**
     * Sends a message to the agent.
     * The behavior (blocking/non-blocking) and push notification configuration
     * are specified within the `params.configuration` object.
     * Optionally, `params.message.contextId` or `params.message.taskId` can be provided.
     * @param params The parameters for sending the message, including the message content and configuration.
     * @returns A Promise resolving to SendMessageResponse, which can be a Message, Task, or an error.
     */
    sendMessage(params: MessageSendParams): Promise<SendMessageResponse>;
    /**
     * Sends a message to the agent and streams back responses using Server-Sent Events (SSE).
     * Push notification configuration can be specified in `params.configuration`.
     * Optionally, `params.message.contextId` or `params.message.taskId` can be provided.
     * Requires the agent to support streaming (`capabilities.streaming: true` in AgentCard).
     * @param params The parameters for sending the message.
     * @returns An AsyncGenerator yielding A2AStreamEventData (Message, Task, TaskStatusUpdateEvent, or TaskArtifactUpdateEvent).
     * The generator throws an error if streaming is not supported or if an HTTP/SSE error occurs.
     */
    sendMessageStream(params: MessageSendParams): AsyncGenerator<A2AStreamEventData, void, undefined>;
    /**
     * Sets or updates the push notification configuration for a given task.
     * Requires the agent to support push notifications (`capabilities.pushNotifications: true` in AgentCard).
     * @param params Parameters containing the taskId and the TaskPushNotificationConfig.
     * @returns A Promise resolving to SetTaskPushNotificationConfigResponse.
     */
    setTaskPushNotificationConfig(params: TaskPushNotificationConfig): Promise<SetTaskPushNotificationConfigResponse>;
    /**
     * Gets the push notification configuration for a given task.
     * @param params Parameters containing the taskId.
     * @returns A Promise resolving to GetTaskPushNotificationConfigResponse.
     */
    getTaskPushNotificationConfig(params: TaskIdParams): Promise<GetTaskPushNotificationConfigResponse>;
    /**
     * Lists the push notification configurations for a given task.
     * @param params Parameters containing the taskId.
     * @returns A Promise resolving to ListTaskPushNotificationConfigResponse.
     */
    listTaskPushNotificationConfig(params: ListTaskPushNotificationConfigParams): Promise<ListTaskPushNotificationConfigResponse>;
    /**
     * Deletes the push notification configuration for a given task.
     * @param params Parameters containing the taskId and push notification configuration ID.
     * @returns A Promise resolving to DeleteTaskPushNotificationConfigResponse.
     */
    deleteTaskPushNotificationConfig(params: DeleteTaskPushNotificationConfigParams): Promise<DeleteTaskPushNotificationConfigResponse>;
    /**
     * Retrieves a task by its ID.
     * @param params Parameters containing the taskId and optional historyLength.
     * @returns A Promise resolving to GetTaskResponse, which contains the Task object or an error.
     */
    getTask(params: TaskQueryParams): Promise<GetTaskResponse>;
    /**
     * Cancels a task by its ID.
     * @param params Parameters containing the taskId.
     * @returns A Promise resolving to CancelTaskResponse, which contains the updated Task object or an error.
     */
    cancelTask(params: TaskIdParams): Promise<CancelTaskResponse>;
    /**
     * @template TExtensionParams The type of parameters for the custom extension method.
     * @template TExtensionResponse The type of response expected from the custom extension method.
     * This should extend JSONRPCResponse. This ensures the extension response is still a valid A2A response.
     * @param method Custom JSON-RPC method defined in the AgentCard's extensions.
     * @param params Extension paramters defined in the AgentCard's extensions.
     * @returns A Promise that resolves to the RPC response.
     */
    callExtensionMethod<TExtensionParams, TExtensionResponse extends JSONRPCResponse>(method: string, params: TExtensionParams): Promise<TExtensionResponse>;
    /**
     * Resubscribes to a task's event stream using Server-Sent Events (SSE).
     * This is used if a previous SSE connection for an active task was broken.
     * Requires the agent to support streaming (`capabilities.streaming: true` in AgentCard).
     * @param params Parameters containing the taskId.
     * @returns An AsyncGenerator yielding A2AStreamEventData (Message, Task, TaskStatusUpdateEvent, or TaskArtifactUpdateEvent).
     */
    resubscribeTask(params: TaskIdParams): AsyncGenerator<A2AStreamEventData, void, undefined>;
    /**
     * Parses an HTTP response body as an A2A Server-Sent Event stream.
     * Each 'data' field of an SSE event is expected to be a JSON-RPC 2.0 Response object,
     * specifically a SendStreamingMessageResponse (or similar structure for resubscribe).
     * @param response The HTTP Response object whose body is the SSE stream.
     * @param originalRequestId The ID of the client's JSON-RPC request that initiated this stream.
     * Used to validate the `id` in the streamed JSON-RPC responses.
     * @returns An AsyncGenerator yielding the `result` field of each valid JSON-RPC success response from the stream.
     */
    private _parseA2ASseStream;
    /**
     * Processes a single SSE event's data string, expecting it to be a JSON-RPC response.
     * @param jsonData The string content from one or more 'data:' lines of an SSE event.
     * @param originalRequestId The ID of the client's request that initiated the stream.
     * @returns The `result` field of the parsed JSON-RPC success response.
     * @throws Error if data is not valid JSON, not a valid JSON-RPC response, an error response, or ID mismatch.
     */
    private _processSseEventData;
    isErrorResponse(response: JSONRPCResponse): response is JSONRPCErrorResponse;
    /**
     * Fetches the Agent Card from the agent's well-known URI and caches its service endpoint URL.
     * This method is called by the constructor.
     * @param agentBaseUrl The base URL of the A2A agent (e.g., https://agent.example.com)
     * @param agentCardPath path to the agent card, defaults to .well-known/agent-card.json
     * @returns A Promise that resolves to the AgentCard.
     */
    private _fetchAndCacheAgentCard;
    /**
   * Retrieves the Agent Card.
   * If an `agentBaseUrl` is provided, it fetches the card from that specific URL.
   * Otherwise, it returns the card fetched and cached during client construction.
   * @param agentBaseUrl Optional. The base URL of the agent to fetch the card from.
   * @param agentCardPath path to the agent card, defaults to .well-known/agent-card.json
   * If provided, this will fetch a new card, not use the cached one from the constructor's URL.
   * @returns A Promise that resolves to the AgentCard.
   */
    getAgentCard(agentBaseUrl?: string, agentCardPath?: string): Promise<AgentCard>;
    /**
     * Determines the agent card URL based on the agent URL.
     * @param agentBaseUrl The agent URL.
     * @param agentCardPath Optional relative path to the agent card, defaults to .well-known/agent-card.json
     */
    private resolveAgentCardUrl;
    /**
     * Gets the RPC service endpoint URL. Ensures the agent card has been fetched first.
     * @returns A Promise that resolves to the service endpoint URL string.
     */
    private _getServiceEndpoint;
}

interface HttpHeaders {
    [key: string]: string;
}
/**
 * Generic interface for handling authentication for HTTP requests.
 *
 * - For each HTTP request, this handler is called to provide additional headers to the request through
 *   the headers() function.
 * - After the server returns a response, the shouldRetryWithHeaders() function is called.  Usually this
 *   function responds to a 401 or 403 response or JSON-RPC codes, but can respond to any other signal -
 *   that is an implementation detail of the AuthenticationHandler.
 * - If the shouldRetryWithHeaders() function returns new headers, then the request should retried with the provided
 *   revised headers.  These provisional headers may, or may not, be optimistically stored for subsequent requests -
 *   that is an implementation detail of the AuthenticationHandler.
 * - If the request is successful and the onSuccessfulRetry() is defined, then the onSuccessfulRetry() function is
 *   called with the headers that were used to successfully complete the request.  This callback provides an
 *   opportunity to save the headers for subsequent requests if they were not already saved.
 *
 */
interface AuthenticationHandler {
    /**
     * Provides additional HTTP request headers.
     * @returns HTTP headers which may include Authorization if available.
     */
    headers: () => Promise<HttpHeaders>;
    /**
     * For every HTTP response (even 200s) the shouldRetryWithHeaders() method is called.
     * This method is supposed to check if the request needs to be retried and if, yes,
     * return a set of headers. An A2A server might indicate auth failures in its response
     * by JSON-rpc codes, HTTP codes like 401, 403 or headers like WWW-Authenticate.
     *
     * @param req The RequestInit object used to invoke fetch()
     * @param res The fetch Response object
     * @returns If the HTTP request should be retried then returns the HTTP headers to use,
     * 	or returns undefined if no retry should be made.
     */
    shouldRetryWithHeaders: (req: RequestInit, res: Response) => Promise<HttpHeaders | undefined>;
    /**
     * If the last HTTP request using the headers from shouldRetryWithHeaders() was successful, and
     * this function is implemented, then it will be called with the headers provided from
     * shouldRetryWithHeaders().
     *
     * This callback allows transient headers to be saved for subsequent requests only when they
     * are validated by the server.
    */
    onSuccessfulRetry?: (headers: HttpHeaders) => Promise<void>;
}
/**
 * Higher-order function that wraps fetch with authentication handling logic.
 * Returns a new fetch function that automatically handles authentication retries for 401/403 responses.
 *
 * @param fetchImpl The underlying fetch implementation to wrap
 * @param authHandler Authentication handler for managing auth headers and retries
 * @returns A new fetch function with authentication handling capabilities
 *
 * Usage examples:
 * - const authFetch = createAuthHandlingFetch(fetch, authHandler);
 * - const response = await authFetch(url, options);
 * - const response = await authFetch(url); // Direct function call
 */
declare function createAuthenticatingFetchWithRetry(fetchImpl: typeof fetch, authHandler: AuthenticationHandler): typeof fetch;

export { A2AClient, type A2AClientOptions, type AuthenticationHandler, type HttpHeaders, createAuthenticatingFetchWithRetry };
