"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CachedFetchIndexApiLogic = void 0;
var _kea = require("kea");
var _lodash = require("lodash");
var _index_name_logic = require("../../components/search_index/index_name_logic");
var _fetch_index_api_logic = require("./fetch_index_api_logic");
var _api = require("../../../common/types/api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FETCH_INDEX_POLLING_DURATION = 5000; // 5 seconds
const FETCH_INDEX_POLLING_DURATION_ON_FAILURE = 30000; // 30 seconds

const CachedFetchIndexApiLogic = exports.CachedFetchIndexApiLogic = (0, _kea.kea)({
  key: props => props.http,
  actions: {
    clearPollTimeout: true,
    createPollTimeout: duration => ({
      duration
    }),
    setTimeoutId: id => ({
      id
    }),
    startPolling: indexName => ({
      indexName
    }),
    stopPolling: true
  },
  connect: {
    actions: [_fetch_index_api_logic.FetchIndexApiLogic, ['apiSuccess', 'apiError', 'apiReset', 'makeRequest']],
    values: [_fetch_index_api_logic.FetchIndexApiLogic, ['data as fetchIndexApiData', 'status'], _index_name_logic.IndexNameLogic, ['indexName']]
  },
  events: ({
    values
  }) => ({
    beforeUnmount: () => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
    }
  }),
  listeners: ({
    actions,
    values,
    props
  }) => ({
    apiError: () => {
      if (values.pollTimeoutId) {
        actions.createPollTimeout(FETCH_INDEX_POLLING_DURATION_ON_FAILURE);
      }
    },
    apiSuccess: () => {
      if (values.pollTimeoutId) {
        actions.createPollTimeout(FETCH_INDEX_POLLING_DURATION);
      }
    },
    createPollTimeout: ({
      duration
    }) => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
      const timeoutId = setTimeout(() => {
        actions.makeRequest({
          indexName: values.indexName,
          http: props.http
        });
      }, duration);
      actions.setTimeoutId(timeoutId);
    },
    startPolling: ({
      indexName
    }) => {
      // Recurring polls are created by apiSuccess and apiError, depending on pollTimeoutId
      if (values.pollTimeoutId) {
        if (indexName === values.indexName) return;
        clearTimeout(values.pollTimeoutId);
      }
      if (indexName) {
        actions.makeRequest({
          indexName,
          http: props.http
        });
        actions.createPollTimeout(FETCH_INDEX_POLLING_DURATION);
      }
    },
    stopPolling: () => {
      if (values.pollTimeoutId) {
        clearTimeout(values.pollTimeoutId);
      }
      actions.clearPollTimeout();
    }
  }),
  path: ['content_connectors', 'content', 'api', 'fetch_index_api_wrapper'],
  reducers: {
    indexData: [null, {
      apiReset: () => null,
      apiSuccess: (currentState, newIndexData) => {
        return (0, _lodash.isEqual)(currentState, newIndexData) ? currentState : newIndexData;
      }
    }],
    pollTimeoutId: [null, {
      clearPollTimeout: () => null,
      setTimeoutId: (_, {
        id
      }) => id
    }]
  },
  selectors: ({
    selectors
  }) => ({
    isInitialLoading: [() => [selectors.status, selectors.indexData], (status, indexData) => {
      return status === _api.Status.IDLE || indexData === null && status === _api.Status.LOADING;
    }]
  })
});