"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getVulnStatsTrendDocIndexingPromises = exports.aggregateLatestFindings = void 0;
exports.removeFindingsStatsTask = removeFindingsStatsTask;
exports.scheduleFindingsStatsTask = scheduleFindingsStatsTask;
exports.setupFindingsStatsTask = setupFindingsStatsTask;
exports.taskRunner = taskRunner;
exports.validateBenchmarkScoreTemplate = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _cloudSecurityPostureCommon = require("@kbn/cloud-security-posture-common");
var _v = require("../routes/benchmark_rules/get_states/v1");
var _get_safe_posture_type_runtime_mapping = require("../../common/runtime_mappings/get_safe_posture_type_runtime_mapping");
var _get_identifier_runtime_mapping = require("../../common/runtime_mappings/get_identifier_runtime_mapping");
var _constants = require("../../common/constants");
var _task_manager_util = require("../lib/task_manager_util");
var _task_state = require("./task_state");
var _mapping_field_util = require("../lib/mapping_field_util");
var _benchmark_score_mapping = require("../create_indices/benchmark_score_mapping");
var _create_indices = require("../create_indices/create_indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CSPM_FINDINGS_STATS_TASK_ID = 'cloud_security_posture-findings_stats';
const CSPM_FINDINGS_STATS_TASK_TYPE = 'cloud_security_posture-stats_task';

// Default config for template fixing operations
const DEFAULT_CONFIG = {
  enabled: true,
  serverless: {
    enabled: true
  },
  enableExperimental: []
};
const fetchBenchmarkScoreTemplate = async esClient => {
  var _templateResponse$ind;
  const templateResponse = await esClient.indices.getIndexTemplate({
    name: _constants.BENCHMARK_SCORE_INDEX_TEMPLATE_NAME
  });
  return (_templateResponse$ind = templateResponse.index_templates[0]) === null || _templateResponse$ind === void 0 ? void 0 : _templateResponse$ind.index_template;
};
const verifyTemplateStructure = (template, logger) => {
  var _template$template, _template$template$ma;
  if (!(template !== null && template !== void 0 && (_template$template = template.template) !== null && _template$template !== void 0 && (_template$template$ma = _template$template.mappings) !== null && _template$template$ma !== void 0 && _template$template$ma.properties)) {
    logger.error(`Template ${_constants.BENCHMARK_SCORE_INDEX_TEMPLATE_NAME} has no mapping properties`);
    return false;
  }
  return true;
};
const checkFieldMappings = (templateProperties, expectedProperties, logger) => {
  const missingFields = [];
  for (const [fieldName] of Object.entries(expectedProperties)) {
    const templateField = templateProperties[fieldName];
    if (!templateField) {
      logger.warn(`${_constants.BENCHMARK_SCORE_INDEX_TEMPLATE_NAME} field missing '${fieldName}'`);
      missingFields.push(fieldName);
    }
  }
  return missingFields;
};

// Comprehensive template validation that checks all fields against benchmarkScoreMapping
const validateBenchmarkScoreTemplate = async (esClient, logger) => {
  try {
    const template = await fetchBenchmarkScoreTemplate(esClient);

    // Check if template has basic structure
    if (!verifyTemplateStructure(template, logger)) {
      logger.warn(`Template ${_constants.BENCHMARK_SCORE_INDEX_TEMPLATE_NAME} has no mapping properties, will trigger fixing`);
      try {
        await (0, _create_indices.createBenchmarkScoreIndex)(esClient, DEFAULT_CONFIG, logger);
      } catch (fixError) {
        logger.error('Failed to fix template with missing properties:', fixError);
        return false;
      }

      // Verify the fix worked
      const verifyTemplate = await fetchBenchmarkScoreTemplate(esClient);
      if (!verifyTemplateStructure(verifyTemplate, logger)) {
        logger.error(`Template still has no mapping properties after fixing attempt`);
        return false;
      }
      return true;
    }
    const templateProperties = template.template.mappings.properties;
    const expectedProperties = _benchmark_score_mapping.benchmarkScoreMapping.properties;
    if (!expectedProperties) {
      logger.warn('Expected mapping has no properties');
      return false;
    }
    const missingFields = checkFieldMappings(templateProperties, expectedProperties, logger);
    if (missingFields.length > 0) {
      logger.warn(`Template ${_constants.BENCHMARK_SCORE_INDEX_TEMPLATE_NAME} has field mapping issues. Will trigger fixing.`);
      try {
        await (0, _create_indices.createBenchmarkScoreIndex)(esClient, DEFAULT_CONFIG, logger);
      } catch (fixError) {
        logger.error('Failed to fix template with field mapping issues:', fixError);
        return false;
      }

      // Verify the fix worked by re-checking field mappings
      const verifyTemplate = await fetchBenchmarkScoreTemplate(esClient);
      if (!verifyTemplateStructure(verifyTemplate, logger)) {
        return false;
      }
      const verifyTemplateProperties = verifyTemplate.template.mappings.properties;
      const stillMissingFields = checkFieldMappings(verifyTemplateProperties, expectedProperties, logger);
      if (stillMissingFields.length > 0) {
        logger.error(`Template ${_constants.BENCHMARK_SCORE_INDEX_TEMPLATE_NAME} fields still missing after fixing: ${stillMissingFields.join(', ')}`);
        return false;
      }
      return true;
    }
    logger.debug(`Template ${_constants.BENCHMARK_SCORE_INDEX_TEMPLATE_NAME} mapping validation passed - all fields match expected mapping`);
    return true;
  } catch (error) {
    logger.error('Error during template validation:', error);
    return false;
  }
};
exports.validateBenchmarkScoreTemplate = validateBenchmarkScoreTemplate;
async function scheduleFindingsStatsTask(taskManager, logger) {
  await (0, _task_manager_util.scheduleTaskSafe)(taskManager, {
    id: CSPM_FINDINGS_STATS_TASK_ID,
    taskType: CSPM_FINDINGS_STATS_TASK_TYPE,
    schedule: {
      interval: `${_constants.CSPM_FINDINGS_STATS_INTERVAL}m`
    },
    state: _task_state.emptyState,
    params: {}
  }, logger);
}
async function removeFindingsStatsTask(taskManager, logger) {
  await (0, _task_manager_util.removeTaskSafe)(taskManager, CSPM_FINDINGS_STATS_TASK_ID, logger);
}
function setupFindingsStatsTask(taskManager, coreStartServices, logger) {
  try {
    taskManager.registerTaskDefinitions({
      [CSPM_FINDINGS_STATS_TASK_TYPE]: {
        title: 'Aggregate latest findings index for score calculation',
        stateSchemaByVersion: _task_state.stateSchemaByVersion,
        createTaskRunner: taskRunner(coreStartServices, logger)
      }
    });
    logger.info(`Registered task successfully [Task: ${CSPM_FINDINGS_STATS_TASK_TYPE}]`);
  } catch (errMsg) {
    const error = (0, _securitysolutionEsUtils.transformError)(errMsg);
    logger.error(`Task registration failed [Task: ${CSPM_FINDINGS_STATS_TASK_TYPE}] ${error.message}`);
  }
}
function taskRunner(coreStartServices, logger) {
  return ({
    taskInstance
  }) => {
    const state = taskInstance.state;
    return {
      async run() {
        try {
          logger.info(`Runs task: ${CSPM_FINDINGS_STATS_TASK_TYPE}`);
          const startServices = await coreStartServices;
          const esClient = startServices[0].elasticsearch.client.asInternalUser;
          const encryptedSoClient = startServices[0].savedObjects.createInternalRepository([_constants.INTERNAL_CSP_SETTINGS_SAVED_OBJECT_TYPE]);
          const status = await aggregateLatestFindings(esClient, encryptedSoClient, logger);
          const updatedState = {
            runs: state.runs + 1,
            health_status: status
          };
          return {
            state: updatedState
          };
        } catch (errMsg) {
          const error = (0, _securitysolutionEsUtils.transformError)(errMsg);
          logger.warn(`Error executing alerting health check task: ${error.message}`);
          const updatedState = {
            runs: state.runs + 1,
            health_status: 'error'
          };
          return {
            state: updatedState
          };
        }
      }
    };
  };
}
const getScoreAggregationQuery = () => ({
  score_by_policy_template: {
    terms: {
      field: 'safe_posture_type'
    },
    aggs: {
      total_findings: {
        value_count: {
          field: 'result.evaluation'
        }
      },
      passed_findings: {
        filter: {
          term: {
            'result.evaluation': 'passed'
          }
        }
      },
      failed_findings: {
        filter: {
          term: {
            'result.evaluation': 'failed'
          }
        }
      },
      score_by_cluster_id: {
        terms: {
          field: 'asset_identifier'
        },
        aggs: {
          total_findings: {
            value_count: {
              field: 'result.evaluation'
            }
          },
          passed_findings: {
            filter: {
              term: {
                'result.evaluation': 'passed'
              }
            }
          },
          failed_findings: {
            filter: {
              term: {
                'result.evaluation': 'failed'
              }
            }
          }
        }
      },
      score_by_benchmark_id: {
        terms: {
          field: 'rule.benchmark.id'
        },
        aggs: {
          benchmark_versions: {
            terms: {
              field: 'rule.benchmark.version'
            },
            aggs: {
              total_findings: {
                value_count: {
                  field: 'result.evaluation'
                }
              },
              passed_findings: {
                filter: {
                  term: {
                    'result.evaluation': 'passed'
                  }
                }
              },
              failed_findings: {
                filter: {
                  term: {
                    'result.evaluation': 'failed'
                  }
                }
              }
            }
          }
        }
      }
    }
  }
});
const getScoreQuery = filteredRules => ({
  index: _cloudSecurityPostureCommon.CDR_LATEST_NATIVE_MISCONFIGURATIONS_INDEX_ALIAS,
  size: 0,
  // creates the safe_posture_type and asset_identifier runtime fields
  runtime_mappings: {
    ...(0, _get_identifier_runtime_mapping.getIdentifierRuntimeMapping)(),
    ...(0, _get_safe_posture_type_runtime_mapping.getSafePostureTypeRuntimeMapping)()
  },
  query: {
    bool: {
      must: [{
        range: {
          '@timestamp': {
            gte: 'now-1d',
            lte: 'now'
          }
        }
      }],
      must_not: filteredRules
    }
  },
  aggs: {
    score_by_namespace: {
      terms: {
        field: 'data_stream.namespace'
      },
      aggs: getScoreAggregationQuery()
    }
  }
});
const getVulnStatsTrendQuery = () => ({
  index: _cloudSecurityPostureCommon.CDR_LATEST_NATIVE_VULNERABILITIES_INDEX_PATTERN,
  size: 0,
  query: {
    match_all: {}
  },
  aggs: {
    critical: {
      filter: {
        term: {
          'vulnerability.severity': _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.CRITICAL
        }
      }
    },
    high: {
      filter: {
        term: {
          'vulnerability.severity': _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.HIGH
        }
      }
    },
    medium: {
      filter: {
        term: {
          'vulnerability.severity': _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.MEDIUM
        }
      }
    },
    low: {
      filter: {
        term: {
          'vulnerability.severity': _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.LOW
        }
      }
    },
    vulnerabilities_stats_by_cloud_account: {
      terms: {
        field: 'cloud.account.id'
      },
      aggs: {
        cloud_account_id: {
          terms: {
            field: 'cloud.account.id',
            size: 1
          }
        },
        cloud_account_name: {
          terms: {
            field: 'cloud.account.name',
            size: 1
          }
        },
        critical: {
          filter: {
            term: {
              'vulnerability.severity': _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.CRITICAL
            }
          }
        },
        high: {
          filter: {
            term: {
              'vulnerability.severity': _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.HIGH
            }
          }
        },
        medium: {
          filter: {
            term: {
              'vulnerability.severity': _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.MEDIUM
            }
          }
        },
        low: {
          filter: {
            term: {
              'vulnerability.severity': _cloudSecurityPostureCommon.VULNERABILITIES_SEVERITY.LOW
            }
          }
        }
      }
    }
  }
});
const getFindingsScoresByNamespaceIndexingPromises = (esClient, scoresByNamespaceBuckets, isCustomScore) => {
  return scoresByNamespaceBuckets.flatMap(namespaceBucket => {
    const namespace = namespaceBucket.key || 'unspecified'; // default fallback if key is empty

    return namespaceBucket.score_by_policy_template.buckets.map(policyTemplateTrend => {
      const clustersStats = Object.fromEntries(policyTemplateTrend.score_by_cluster_id.buckets.map(clusterStats => {
        return [clusterStats.key, {
          total_findings: clusterStats.total_findings.value,
          passed_findings: clusterStats.passed_findings.doc_count,
          failed_findings: clusterStats.failed_findings.doc_count
        }];
      }));
      const benchmarkStats = Object.fromEntries(policyTemplateTrend.score_by_benchmark_id.buckets.map(benchmarkIdBucket => {
        const benchmarkVersions = Object.fromEntries(benchmarkIdBucket.benchmark_versions.buckets.map(benchmarkVersionBucket => {
          return [(0, _mapping_field_util.toBenchmarkMappingFieldKey)(benchmarkVersionBucket.key), {
            total_findings: benchmarkVersionBucket.total_findings.value,
            passed_findings: benchmarkVersionBucket.passed_findings.doc_count,
            failed_findings: benchmarkVersionBucket.failed_findings.doc_count
          }];
        }));
        return [benchmarkIdBucket.key, benchmarkVersions];
      }));
      return esClient.index({
        index: _constants.BENCHMARK_SCORE_INDEX_DEFAULT_NS,
        document: {
          policy_template: policyTemplateTrend.key,
          passed_findings: policyTemplateTrend.passed_findings.doc_count,
          failed_findings: policyTemplateTrend.failed_findings.doc_count,
          total_findings: policyTemplateTrend.total_findings.value,
          score_by_cluster_id: clustersStats,
          score_by_benchmark_id: benchmarkStats,
          is_enabled_rules_score: isCustomScore,
          namespace
        }
      });
    });
  });
};
const getVulnStatsTrendDocIndexingPromises = (esClient, vulnStatsAggs) => {
  if (!vulnStatsAggs) return;
  const scoreByCloudAccount = Object.fromEntries(vulnStatsAggs.vulnerabilities_stats_by_cloud_account.buckets.map(accountScore => {
    var _accountScore$cloud_a;
    const cloudAccountId = accountScore.key;
    return [cloudAccountId, {
      cloudAccountId: accountScore.key,
      cloudAccountName: ((_accountScore$cloud_a = accountScore.cloud_account_name.buckets[0]) === null || _accountScore$cloud_a === void 0 ? void 0 : _accountScore$cloud_a.key) || '',
      critical: accountScore.critical.doc_count,
      high: accountScore.high.doc_count,
      medium: accountScore.medium.doc_count,
      low: accountScore.low.doc_count
    }];
  }));
  return esClient.index({
    index: _constants.BENCHMARK_SCORE_INDEX_DEFAULT_NS,
    document: {
      policy_template: _constants.VULN_MGMT_POLICY_TEMPLATE,
      critical: vulnStatsAggs.critical.doc_count,
      high: vulnStatsAggs.high.doc_count,
      medium: vulnStatsAggs.medium.doc_count,
      low: vulnStatsAggs.low.doc_count,
      vulnerabilities_stats_by_cloud_account: scoreByCloudAccount
    }
  });
};
exports.getVulnStatsTrendDocIndexingPromises = getVulnStatsTrendDocIndexingPromises;
const aggregateLatestFindings = async (esClient, encryptedSoClient, logger) => {
  try {
    var _customScoreIndexQuer, _fullScoreIndexQueryR;
    const startAggTime = performance.now();

    // Validate + attempting to fix benchmark score template mapping before aggregating findings
    const templateValidationSuccess = await validateBenchmarkScoreTemplate(esClient, logger);
    if (!templateValidationSuccess) {
      logger.error('Template validation failed and could not be fixed. Stopping findings aggregation.');
      return 'error';
    }
    const rulesFilter = await (0, _v.getMutedRulesFilterQuery)(encryptedSoClient);
    const customScoreIndexQueryResult = await esClient.search(getScoreQuery(rulesFilter));
    const fullScoreIndexQueryResult = await esClient.search(getScoreQuery([]));
    const vulnStatsTrendIndexQueryResult = await esClient.search(getVulnStatsTrendQuery());
    if (!customScoreIndexQueryResult.aggregations && !vulnStatsTrendIndexQueryResult.aggregations) {
      logger.warn(`No data found in latest findings index`);
      return 'warning';
    }
    const totalAggregationTime = performance.now() - startAggTime;
    logger.debug(`Executed aggregation query [Task: ${CSPM_FINDINGS_STATS_TASK_TYPE}] [Duration: ${Number(totalAggregationTime).toFixed(2)}ms]`);
    const customScoresByNamespaceBuckets = ((_customScoreIndexQuer = customScoreIndexQueryResult.aggregations) === null || _customScoreIndexQuer === void 0 ? void 0 : _customScoreIndexQuer.score_by_namespace.buckets) || [];
    const fullScoresByNamespaceBuckets = ((_fullScoreIndexQueryR = fullScoreIndexQueryResult.aggregations) === null || _fullScoreIndexQueryR === void 0 ? void 0 : _fullScoreIndexQueryR.score_by_namespace.buckets) || [];
    const findingsCustomScoresByNamespaceDocIndexingPromises = getFindingsScoresByNamespaceIndexingPromises(esClient, customScoresByNamespaceBuckets, true);
    const findingsFullScoresByNamespaceDocIndexingPromises = getFindingsScoresByNamespaceIndexingPromises(esClient, fullScoresByNamespaceBuckets, false);
    const vulnStatsTrendDocIndexingPromises = getVulnStatsTrendDocIndexingPromises(esClient, vulnStatsTrendIndexQueryResult.aggregations);
    const startIndexTime = performance.now();

    // executing indexing commands
    await Promise.all([findingsCustomScoresByNamespaceDocIndexingPromises, findingsFullScoresByNamespaceDocIndexingPromises, vulnStatsTrendDocIndexingPromises].filter(Boolean));
    const totalIndexTime = Number(performance.now() - startIndexTime).toFixed(2);
    logger.debug(`Finished saving results [Task: ${CSPM_FINDINGS_STATS_TASK_TYPE}] [Duration: ${totalIndexTime}ms]`);
    const totalTaskTime = Number(performance.now() - startAggTime).toFixed(2);
    logger.debug(`Finished run ended [Task: ${CSPM_FINDINGS_STATS_TASK_TYPE}] [Duration: ${totalTaskTime}ms]`);
    return 'ok';
  } catch (errMsg) {
    const error = (0, _securitysolutionEsUtils.transformError)(errMsg);
    logger.error(`Failure during task run [Task: ${CSPM_FINDINGS_STATS_TASK_TYPE}] ${error.message}`);
    logger.error(errMsg);
    return 'error';
  }
};
exports.aggregateLatestFindings = aggregateLatestFindings;