"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleHeaderValidate = handleHeaderValidate;
exports.handleKVValidate = handleKVValidate;
var _util = require("../../util");
var _processors = require("../../util/processors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function handleKVValidate({
  state,
  client
}) {
  const kvProcessor = state.kvProcessor;
  const packageName = state.packageName;
  const dataStreamName = state.dataStreamName;

  // Pick logSamples if there was no header detected.
  const samples = state.header ? state.kvLogMessages : state.logSamples;
  const {
    errors
  } = await verifyKVProcessor(kvProcessor, samples, client);
  if (errors.length > 0) {
    return {
      errors,
      lastExecutedChain: 'kvValidate'
    };
  }

  // Converts JSON Object into a string and parses it as a array of JSON strings
  const additionalProcessors = state.additionalProcessors;
  additionalProcessors.push(kvProcessor[0]);
  const samplesObject = await buildJSONSamples(state.logSamples, additionalProcessors, client);
  const jsonSamples = samplesObject.map(log => log[packageName]).map(log => log[dataStreamName]).map(log => JSON.stringify(log));
  return {
    jsonSamples,
    additionalProcessors,
    errors: [],
    lastExecutedChain: 'kvValidate'
  };
}
async function handleHeaderValidate({
  state,
  client
}) {
  const grokPattern = state.grokPattern;
  const grokProcessor = (0, _processors.createGrokProcessor)([grokPattern]);
  const pipeline = {
    processors: grokProcessor,
    on_failure: [(0, _processors.createPassthroughFailureProcessor)()]
  };
  const {
    pipelineResults,
    errors
  } = await (0, _util.testPipeline)(state.logSamples, pipeline, client);
  if (errors.length > 0) {
    return {
      errors,
      lastExecutedChain: 'kv_header_validate'
    };
  }
  const kvLogMessages = pipelineResults.map(entry => entry.message);
  const additionalProcessors = state.additionalProcessors;
  additionalProcessors.push(grokProcessor[0]);
  return {
    kvLogMessages,
    additionalProcessors,
    errors: [],
    lastExecutedChain: 'kv_header_validate'
  };
}
async function verifyKVProcessor(kvProcessor, samples, client) {
  // This processor removes the original message field in the  output
  const pipeline = {
    processors: [kvProcessor[0], (0, _processors.createRemoveProcessor)()],
    on_failure: [(0, _processors.createPassthroughFailureProcessor)()]
  };
  const {
    errors
  } = await (0, _util.testPipeline)(samples, pipeline, client);
  return {
    errors
  };
}
async function buildJSONSamples(samples, processors, client) {
  const pipeline = {
    processors: [...processors, (0, _processors.createRemoveProcessor)()],
    on_failure: [(0, _processors.createPassthroughFailureProcessor)()]
  };
  const {
    pipelineResults
  } = await (0, _util.testPipeline)(samples, pipeline, client);
  return pipelineResults;
}